/* GnomerMind - A MasterMind(R)-based game for Gnome
 * (C) 2001 Germano Rizzo
 *
 * face.c - main graphic interface implementation
 * Author: Germano Rizzo
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "common.h"
#include <libgnomeui/gnome-window-icon.h>
#include "gnomermind.h"
#include "menu.h"
#include "prefs.h"
#include "keyboard.h"
#include "themes.h"
#include "sounds.h"
#include "cache.h"

gboolean ask;

/*
 * g*=board window, p*=palette window, s*=solution window
 */

GtkWidget *gwin, *swin;
static GtkWidget *gtable, *gtpix[MAXH + 3][MAXR + 2], *tok[MAXH][MAXR],
  *pin[MAXR][MAXH];
static GtkWidget *pwin, *ptable, *ptpix[MAXC + 3][3], *ptok[MAXC], *ctrl;
static GtkWidget *stable, *stpix[MAXH + 2][3], *stok[MAXH];
static GtkWidget *gcmd[MAXH][MAXR], *pcmd[MAXC], *ccmd;

/**
 * hide_n_seek
 * 
 * Hides the solution window and relaunches the game
 **/
static gboolean
hide_n_seek ()
{
  manage (2, 0, 0);

  return TRUE;
}

/**
 * puttok
 * @c: the column of the token to change
 * @r: the row of the token to change
 * @col: the color of the new token
 *
 * Changes the color of a token in the field
 **/
inline void
puttok (gint c, gint r, gint col)
{
  g_assert (c >= 0 && c < holez);
  g_assert (r >= 0 && r < roundz);
  g_assert (col >= 0 && col <= colorz);

  gtk_pixmap_set (GTK_PIXMAP (tok[c][r]), stock.pix[TOK + col], NULL);

#ifdef HAVE_ESD
  sound_play ((col && 1), FALSE);
#endif
}

/**
 * putpins
 * @row: the row of the pin set to change
 * @b: the number of black pins in that set
 * @w: the number of white pins in that set
 * 
 * Changes the color of a `pin' set in the field
 **/
inline void
putpins (gint row, gint b, gint w)
{
  gint i;

  g_assert (row >= 0 && row < roundz);
  g_assert (b >= 0 && b <= holez);
  g_assert (w >= 0 && w <= holez);
  g_assert (b + w >= 0 && b + w <= holez);

#ifdef HAVE_ESD
  sound_play (GUESS, FALSE);
#endif

  for (i = 0; i < b; i++)
    gtk_pixmap_set (GTK_PIXMAP (pin[row][i]), stock.pix[PIN + 2],
		    stock.bit[MPIN + i]);

  for (i = b; i < b + w; i++)
    gtk_pixmap_set (GTK_PIXMAP (pin[row][i]), stock.pix[PIN + 1],
		    stock.bit[MPIN + i]);

  for (i = b + w; i < holez; i++)
    gtk_pixmap_set (GTK_PIXMAP (pin[row][i]), stock.pix[PIN + 0],
		    stock.bit[MPIN + i]);
}

/**
 * pick
 * @color: the token color (0->empty, 1-10->colors)
 * @present: whether it will be present or not
 * 
 * Changes the presence of a token in the palette window
 **/
inline void
pick (gint color, gint present)
{
  g_assert (color > 0 && color <= colorz);

  gtk_pixmap_set (GTK_PIXMAP (ptok[color - 1]),
		  stock.pix[TOK + (present * color)], NULL);

#ifdef HAVE_ESD
  sound_play ((present * PUT), FALSE);
#endif
}

/**
 * chstate
 * 
 * Changes the `state' icon in the palette window
 **/
inline void
chstate ()
{
  g_assert (state >= 0 && state <= 3);

  gtk_pixmap_set (GTK_PIXMAP (ptpix[colorz + 1][1]), stock.pix[TILE], NULL);
  gtk_pixmap_set (GTK_PIXMAP (ctrl), stock.pix[STT + state], NULL);
}

/**
 * getcmd
 * @widget: ignored, only for callback
 * @event: the generated event
 * @data: encodes the row and column of the pressed button
 *
 * Catches a click in the game window and calls the manager
 **/
static void
getcmd (GtkWidget * widget, GdkEventButton * event, gpointer data)
{
  gint x, y;

  x = GPOINTER_TO_INT (data) % MAXR;
  y = GPOINTER_TO_INT (data) / MAXR;
  g_assert (x >= 0 && x < holez);
  g_assert (y >= 0 && y < roundz);

  if (event->type == GDK_BUTTON_PRESS)
    switch (event->button)
      {
      case 1:
	manage (0, x, y);
	break;
      case 3:
	{
	  gint col;

	  if ((col = try[roundz - y - 1][x]) != 0)
	    direct_sel (col);
	  else if (act == roundz - y - 1)
	    rnd_sel (x);
	}
      }
}

/**
 * pgetcmd
 * @widget: ignored, only for callback
 * @event: the generated event
 * @data: encodes the color of the pressed button
 * 
 * Catches a click in the palette and calls the backend
 **/
static void
pgetcmd (GtkWidget * widget, GdkEventButton * event, gpointer data)
{
  gint c;

  c = GPOINTER_TO_INT (data);
  g_assert (c > 0 && c <= colorz);

  if (event->type == GDK_BUTTON_PRESS)
    switch (event->button)
      {
      case 1:
	manage (1, c, 0);
	break;
      case 3:
	direct_sel (c);
      }
}

/**
 * cgetcmd
 * @widget: ignored, only for callback
 * @ignored: ignored, only for callback
 * @data: ignored, only for callback
 * 
 * Catches a click in the command box and calls the backend
 **/
static void
cgetcmd (GtkWidget * widget, GdkEventButton * ignored, gpointer data)
{
  manage (2, 0, 0);
}

/**
 * greset
 * 
 * Graphically resets the game window
 **/
void
greset ()
{
  gint i, j;

#ifdef HAVE_ESD
  sound_play (RESET, TRUE);
#endif

  for (i = 0; i < roundz; i++)
    {
      putpins (i, 0, 0);
      for (j = 0; j < holez; j++)
	puttok (j, i, 0);
    }
}

/**
 * preset
 * 
 * Graphically resets the palette window
 **/
void
preset ()
{
  gint i;

  for (i = 1; i <= colorz; i++)
    pick (i, TRUE);
  chstate ();
}

/**
 * solution
 * @sol: the field that had to be guessed
 * @success: whether the game has come to success or not
 * 
 * Displays the solution window
 **/
void
solution (gboolean success)
{
  gint i;

  if (success)
    {
      gtk_window_set_title (GTK_WINDOW (swin), _("Compliments! :)"));
      msgpush (_("Compliments! :)"));

#ifdef HAVE_ESD
      sound_play (SUCCESS, TRUE);
#endif
    }
  else
    {
      gtk_window_set_title (GTK_WINDOW (swin), _("Too bad! :'("));
      msgpush (_("Too bad! :'("));

#ifdef HAVE_ESD
      sound_play (FAIL, TRUE);
#endif
    }

  for (i = 0; i < holez; i++)
    gtk_pixmap_set (GTK_PIXMAP (stok[i]),
		    stock.pix[TOK + field[i]], stock.bit[MTOK]);

  gtk_widget_show_all (swin);
}

/**
 * retok
 *
 * Redraws all the tokens field
 **/
void
retok ()
{
  gint i, j;

  for (i = 0; i < roundz; i++)
    {
      putpins (roundz - i - 1, b[i], w[i]);
      for (j = 0; j < holez; j++)
	puttok (j, roundz - i - 1, try[i][j]);
    }

  preset ();
  if (colsel)
    pick (colsel, FALSE);

  if (state == 3)
    solution (FALSE);
  else if (state == 2)
    solution (TRUE);
}

/**
 * solution_hide
 * 
 * Hides the solution window
 **/
inline void
solution_hide ()
{
  gtk_widget_hide (swin);
  msgpop ();
}

/**
 * retheme_board
 *
 * Draws the boards with the current theme
 **/
void
retheme_board ()
{
  gint i, j;

  gtk_pixmap_set (GTK_PIXMAP (gtpix[0][0]), stock.pix[UL], NULL);
  gtk_pixmap_set (GTK_PIXMAP (gtpix[holez + 2][0]), stock.pix[UR], NULL);
  gtk_pixmap_set (GTK_PIXMAP (gtpix[0][roundz + 1]), stock.pix[BL], NULL);
  gtk_pixmap_set (GTK_PIXMAP (gtpix[holez + 2][roundz + 1]), stock.pix[BR],
		  NULL);

  gtk_pixmap_set (GTK_PIXMAP (ptpix[0][0]), stock.pix[UL], NULL);
  gtk_pixmap_set (GTK_PIXMAP (ptpix[colorz + 2][0]), stock.pix[UR], NULL);
  gtk_pixmap_set (GTK_PIXMAP (ptpix[0][1]), stock.pix[L], NULL);
  gtk_pixmap_set (GTK_PIXMAP (ptpix[colorz + 2][1]), stock.pix[R], NULL);
  gtk_pixmap_set (GTK_PIXMAP (ptpix[0][2]), stock.pix[BL], NULL);
  gtk_pixmap_set (GTK_PIXMAP (ptpix[colorz + 2][2]), stock.pix[BR], NULL);

  gtk_pixmap_set (GTK_PIXMAP (stpix[0][0]), stock.pix[UL], NULL);
  gtk_pixmap_set (GTK_PIXMAP (stpix[holez + 1][0]), stock.pix[UR], NULL);
  gtk_pixmap_set (GTK_PIXMAP (stpix[0][1]), stock.pix[L], NULL);
  gtk_pixmap_set (GTK_PIXMAP (stpix[holez + 1][1]), stock.pix[R], NULL);
  gtk_pixmap_set (GTK_PIXMAP (stpix[0][2]), stock.pix[BL], NULL);
  gtk_pixmap_set (GTK_PIXMAP (stpix[holez + 1][2]), stock.pix[BR], NULL);

  for (i = 1; i < holez + 2; i++)
    {
      gtk_pixmap_set (GTK_PIXMAP (gtpix[i][0]), stock.pix[U], NULL);
      gtk_pixmap_set (GTK_PIXMAP (gtpix[i][roundz + 1]), stock.pix[B], NULL);
    }

  for (i = 1; i < roundz + 1; i++)
    {
      gtk_pixmap_set (GTK_PIXMAP (gtpix[0][i]), stock.pix[L], NULL);
      gtk_pixmap_set (GTK_PIXMAP (gtpix[holez + 2][i]), stock.pix[R], NULL);
    }

  for (i = 1; i < holez + 2; i++)
    for (j = 1; j < roundz + 1; j++)
      gtk_pixmap_set (GTK_PIXMAP (gtpix[i][j]), stock.pix[TILE], NULL);


  for (i = 1; i < colorz + 2; i++)
    {
      gtk_pixmap_set (GTK_PIXMAP (ptpix[i][0]), stock.pix[U], NULL);
      gtk_pixmap_set (GTK_PIXMAP (ptpix[i][1]), stock.pix[TILE], NULL);
      gtk_pixmap_set (GTK_PIXMAP (ptpix[i][2]), stock.pix[B], NULL);
    }

  for (i = 1; i < holez + 1; i++)
    {
      gtk_pixmap_set (GTK_PIXMAP (stpix[i][0]), stock.pix[U], NULL);
      gtk_pixmap_set (GTK_PIXMAP (stpix[i][1]), stock.pix[TILE], NULL);
      gtk_pixmap_set (GTK_PIXMAP (stpix[i][2]), stock.pix[B], NULL);
    }

  retok ();

  for (i = 0; i < holez; i++)
    for (j = 0; j < roundz; j++)
      gtk_widget_shape_combine_mask (gcmd[i][j], stock.bit[MTOK], 0, 0);

  for (i = 0; i < colorz; i++)
    gtk_widget_shape_combine_mask (pcmd[i], stock.bit[MTOK], 0, 0);

  gtk_widget_shape_combine_mask (ccmd, stock.bit[MSTT], 0, 0);

  for (i = 1; i < roundz + 2; i++)
    gtk_fixed_move (GTK_FIXED (gtable), gtpix[0][i], 0,
		    stock.ext + (i - 1) * stock.base);

  for (i = 1; i < holez + 3; i++)
    {
      gtk_fixed_move (GTK_FIXED (gtable), gtpix[i][0],
		      stock.ext + (i - 1) * stock.base, 0);
      for (j = 1; j < roundz + 2; j++)
	gtk_fixed_move (GTK_FIXED (gtable), gtpix[i][j],
			stock.ext + (i - 1) * stock.base,
			stock.ext + (j - 1) * stock.base);
    }

  for (i = 0; i < holez; i++)
    {
      gtk_fixed_move (GTK_FIXED (stable), stok[i], stock.ext + i * stock.base,
		      stock.ext);
      for (j = 0; j < roundz; j++)
	{
	  gtk_fixed_move (GTK_FIXED (gtable), gcmd[i][j],
			  stock.ext + (i + 1) * stock.base,
			  stock.ext + j * stock.base);
	  gtk_fixed_move (GTK_FIXED (gtable), pin[j][i], stock.ext,
			  stock.ext + j * stock.base);
	}
    }

  for (i = 1; i < 3; i++)
    {
      gtk_fixed_move (GTK_FIXED (stable), stpix[0][i], 0,
		      stock.ext + (i - 1) * stock.base);
      gtk_fixed_move (GTK_FIXED (ptable), ptpix[0][i], 0,
		      stock.ext + (i - 1) * stock.base);
    }

  for (i = 1; i < holez + 2; i++)
    {
      gtk_fixed_move (GTK_FIXED (stable), stpix[i][0],
		      stock.ext + (i - 1) * stock.base, 0);
      for (j = 1; j < 3; j++)
	gtk_fixed_move (GTK_FIXED (stable), stpix[i][j],
			stock.ext + (i - 1) * stock.base,
			stock.ext + (j - 1) * stock.base);
    }

  for (i = 1; i < colorz + 3; i++)
    {
      gtk_fixed_move (GTK_FIXED (ptable), ptpix[i][0],
		      stock.ext + (i - 1) * stock.base, 0);
      for (j = 1; j < 3; j++)
	gtk_fixed_move (GTK_FIXED (ptable), ptpix[i][j],
			stock.ext + (i - 1) * stock.base,
			stock.ext + (j - 1) * stock.base);
    }

  for (i = 0; i < colorz; i++)
    gtk_fixed_move (GTK_FIXED (ptable), pcmd[i],
		    stock.ext + i * stock.base, stock.ext);

  gtk_fixed_move (GTK_FIXED (ptable), ccmd,
		  stock.ext + colorz * stock.base, stock.ext);
}

/**
 * inflate_board
 *
 * Redimensions the boards
 **/
void
inflate_board ()
{
  gint i, j;

  if (h < holez)
    {

      for (i = 0; i < roundz; i++)
	for (j = h; j < holez; j++)
	  {
	    gtk_container_remove (GTK_CONTAINER (gtable), gcmd[j][i]);
	    gtk_container_remove (GTK_CONTAINER (gtable), pin[i][j]);
	  }

      for (i = 0; i < roundz + 2; i++)
	for (j = h + 3; j < holez + 3; j++)
	  gtk_container_remove (GTK_CONTAINER (gtable), gtpix[j][i]);

      for (i = 1; i < roundz + 1; i++)
	gtk_pixmap_set (GTK_PIXMAP (gtpix[h + 2][i]), stock.pix[R], NULL);
      gtk_pixmap_set (GTK_PIXMAP (gtpix[h + 2][0]), stock.pix[UR], NULL);
      gtk_pixmap_set (GTK_PIXMAP (gtpix[h + 2][roundz + 1]), stock.pix[BR],
		      NULL);

      for (i = h; i < holez; i++)
	gtk_container_remove (GTK_CONTAINER (stable), stok[i]);
      for (i = 0; i < 3; i++)
	for (j = h + 2; j < holez + 2; j++)
	  gtk_container_remove (GTK_CONTAINER (stable), stpix[j][i]);

      gtk_pixmap_set (GTK_PIXMAP (stpix[h + 1][0]), stock.pix[UR], NULL);
      gtk_pixmap_set (GTK_PIXMAP (stpix[h + 1][1]), stock.pix[R], NULL);
      gtk_pixmap_set (GTK_PIXMAP (stpix[h + 1][2]), stock.pix[BR], NULL);
    }
  else if (h > holez)
    {
      gtk_pixmap_set (GTK_PIXMAP (gtpix[holez + 2][0]), stock.pix[U], NULL);
      gtk_pixmap_set (GTK_PIXMAP (gtpix[holez + 2][roundz + 1]),
		      stock.pix[B], NULL);
      for (i = 1; i < roundz + 1; i++)
	gtk_pixmap_set (GTK_PIXMAP (gtpix[holez + 2][i]), stock.pix[TILE],
			NULL);

      for (i = holez + 3; i < h + 2; i++)
	{
	  gtpix[i][0] = gtk_pixmap_new (stock.pix[U], NULL);
	  gtpix[i][roundz + 1] = gtk_pixmap_new (stock.pix[B], NULL);
	  for (j = 1; j < roundz + 1; j++)
	    gtpix[i][j] = gtk_pixmap_new (stock.pix[TILE], NULL);
	}
      gtpix[h + 2][0] = gtk_pixmap_new (stock.pix[UR], NULL);
      gtpix[h + 2][roundz + 1] = gtk_pixmap_new (stock.pix[BR], NULL);
      for (j = 1; j < roundz + 1; j++)
	gtpix[h + 2][j] = gtk_pixmap_new (stock.pix[R], NULL);

      for (i = holez + 3; i < h + 3; i++)
	{
	  gtk_fixed_put (GTK_FIXED (gtable), gtpix[i][0],
			 stock.ext + (i - 1) * stock.base, 0);
	  for (j = 1; j < roundz + 2; j++)
	    gtk_fixed_put (GTK_FIXED (gtable), gtpix[i][j],
			   stock.ext + (i - 1) * stock.base,
			   stock.ext + (j - 1) * stock.base);
	}

      for (i = 0; i < roundz; i++)
	for (j = holez; j < h; j++)
	  {
	    gcmd[j][i] = gtk_event_box_new ();
	    gtk_widget_shape_combine_mask (GTK_WIDGET (gcmd[j][i]),
					   stock.bit[MTOK], 0, 0);
	    gtk_widget_add_events (gcmd[j][i], GDK_BUTTON_PRESS_MASK);
	    gtk_signal_connect (GTK_OBJECT (gcmd[j][i]), "button_press_event",
				GTK_SIGNAL_FUNC (getcmd),
				GINT_TO_POINTER (i * MAXR + j));

	    tok[j][i] = gtk_pixmap_new (stock.pix[TOK + i + 1], NULL);
	    gtk_container_add (GTK_CONTAINER (gcmd[j][i]), tok[j][i]);

	    gtk_fixed_put (GTK_FIXED (gtable), gcmd[j][i],
			   stock.ext + (j + 1) * stock.base,
			   stock.ext + i * stock.base);

	    pin[i][j] =
	      gtk_pixmap_new (stock.pix[PIN + 0], stock.pix[MPIN + j]);
	    gtk_fixed_put (GTK_FIXED (gtable), pin[i][j], stock.ext,
			   stock.ext + i * stock.base);
	  }

      gtk_pixmap_set (GTK_PIXMAP (stpix[holez + 1][0]), stock.pix[U], NULL);
      gtk_pixmap_set (GTK_PIXMAP (stpix[holez + 1][1]), stock.pix[TILE],
		      NULL);
      gtk_pixmap_set (GTK_PIXMAP (stpix[holez + 1][2]), stock.pix[B], NULL);

      for (i = holez + 2; i < h + 1; i++)
	{
	  stpix[i][0] = gtk_pixmap_new (stock.pix[U], NULL);
	  stpix[i][1] = gtk_pixmap_new (stock.pix[TILE], NULL);
	  stpix[i][2] = gtk_pixmap_new (stock.pix[B], NULL);
	}

      stpix[h + 1][0] = gtk_pixmap_new (stock.pix[UR], NULL);
      stpix[h + 1][1] = gtk_pixmap_new (stock.pix[R], NULL);
      stpix[h + 1][2] = gtk_pixmap_new (stock.pix[BR], NULL);

      for (i = holez + 2; i < h + 2; i++)
	{
	  gtk_fixed_put (GTK_FIXED (stable), stpix[i][0],
			 stock.ext + (i - 1) * stock.base, 0);
	  gtk_fixed_put (GTK_FIXED (stable), stpix[i][1],
			 stock.ext + (i - 1) * stock.base, stock.ext);
	  gtk_fixed_put (GTK_FIXED (stable), stpix[i][2],
			 stock.ext + (i - 1) * stock.base,
			 stock.ext + stock.base);
	}

      for (i = holez; i < h; i++)
	{
	  stok[i] = gtk_pixmap_new (stock.pix[TOK + 0], stock.bit[MTOK]);
	  gtk_fixed_put (GTK_FIXED (stable), stok[i],
			 stock.ext + i * stock.base, stock.ext);
	}
    }
  holez = h;

  if (c < colorz)
    {
      for (i = c; i < colorz; i++)
	gtk_container_remove (GTK_CONTAINER (ptable), pcmd[i]);

      for (i = c + 3; i < colorz + 3; i++)
	for (j = 0; j < 3; j++)
	  gtk_container_remove (GTK_CONTAINER (ptable), ptpix[i][j]);

      gtk_pixmap_set (GTK_PIXMAP (ptpix[c + 2][0]), stock.pix[UR], NULL);
      gtk_pixmap_set (GTK_PIXMAP (ptpix[c + 2][1]), stock.pix[R], NULL);
      gtk_pixmap_set (GTK_PIXMAP (ptpix[c + 2][2]), stock.pix[BR], NULL);
      gtk_fixed_move (GTK_FIXED (ptable), ccmd, stock.ext + c * stock.base,
		      stock.ext);

    }
  else if (c > colorz)
    {
      gtk_pixmap_set (GTK_PIXMAP (ptpix[colorz + 2][0]), stock.pix[U], NULL);
      gtk_pixmap_set (GTK_PIXMAP (ptpix[colorz + 2][1]), stock.pix[TILE],
		      NULL);
      gtk_pixmap_set (GTK_PIXMAP (ptpix[colorz + 2][2]), stock.pix[B], NULL);

      for (i = colorz + 3; i < c + 2; i++)
	{
	  ptpix[i][0] = gtk_pixmap_new (stock.pix[U], NULL);
	  ptpix[i][1] = gtk_pixmap_new (stock.pix[TILE], NULL);
	  ptpix[i][2] = gtk_pixmap_new (stock.pix[B], NULL);
	}

      ptpix[c + 2][0] = gtk_pixmap_new (stock.pix[UR], NULL);
      ptpix[c + 2][1] = gtk_pixmap_new (stock.pix[R], NULL);
      ptpix[c + 2][2] = gtk_pixmap_new (stock.pix[BR], NULL);

      for (i = colorz + 3; i < c + 3; i++)
	{
	  gtk_fixed_put (GTK_FIXED (ptable), ptpix[i][0],
			 stock.ext + (i - 1) * stock.base, 0);
	  gtk_fixed_put (GTK_FIXED (ptable), ptpix[i][1],
			 stock.ext + (i - 1) * stock.base, stock.ext);
	  gtk_fixed_put (GTK_FIXED (ptable), ptpix[i][2],
			 stock.ext + (i - 1) * stock.base,
			 stock.ext + stock.base);
	}

      for (i = colorz; i < c; i++)
	{
	  pcmd[i] = gtk_event_box_new ();
	  gtk_widget_shape_combine_mask (GTK_WIDGET (pcmd[i]),
					 stock.bit[MTOK], 0, 0);
	  gtk_widget_add_events (pcmd[i], GDK_BUTTON_PRESS_MASK);
	  gtk_signal_connect (GTK_OBJECT (pcmd[i]), "button_press_event",
			      GTK_SIGNAL_FUNC (pgetcmd),
			      GINT_TO_POINTER (i + 1));

	  ptok[i] = gtk_pixmap_new (stock.pix[TOK + i + 1], NULL);
	  gtk_container_add (GTK_CONTAINER (pcmd[i]), ptok[i]);

	  gtk_fixed_put (GTK_FIXED (ptable), pcmd[i],
			 stock.ext + (i) * stock.base, stock.ext);
	}

      gtk_fixed_move (GTK_FIXED (ptable), ccmd, stock.ext + c * stock.base,
		      stock.ext);

    }
  colorz = c;

  if (r < roundz)
    {
      for (i = 0; i < holez; i++)
	for (j = r; j < roundz; j++)
	  {
	    gtk_container_remove (GTK_CONTAINER (gtable), gcmd[i][j]);
	    gtk_container_remove (GTK_CONTAINER (gtable), pin[j][i]);
	  }
      for (i = 0; i < holez + 3; i++)
	for (j = r + 2; j < roundz + 2; j++)
	  gtk_container_remove (GTK_CONTAINER (gtable), gtpix[i][j]);

      for (i = 1; i < holez + 2; i++)
	gtk_pixmap_set (GTK_PIXMAP (gtpix[i][r + 1]), stock.pix[B], NULL);
      gtk_pixmap_set (GTK_PIXMAP (gtpix[0][r + 1]), stock.pix[BL], NULL);
      gtk_pixmap_set (GTK_PIXMAP (gtpix[holez + 2][r + 1]), stock.pix[BR],
		      NULL);
    }
  else if (r > roundz)
    {
      gtk_pixmap_set (GTK_PIXMAP (gtpix[0][roundz + 1]), stock.pix[L], NULL);
      gtk_pixmap_set (GTK_PIXMAP (gtpix[holez + 2][roundz + 1]),
		      stock.pix[R], NULL);
      for (i = 1; i < holez + 2; i++)
	gtk_pixmap_set (GTK_PIXMAP (gtpix[i][roundz + 1]), stock.pix[TILE],
			NULL);

      for (i = roundz + 2; i < r + 1; i++)
	{
	  gtpix[0][i] = gtk_pixmap_new (stock.pix[L], NULL);
	  gtpix[holez + 2][i] = gtk_pixmap_new (stock.pix[R], NULL);
	  for (j = 1; j < holez + 2; j++)
	    gtpix[j][i] = gtk_pixmap_new (stock.pix[TILE], NULL);
	}

      gtpix[0][r + 1] = gtk_pixmap_new (stock.pix[BL], NULL);
      gtpix[holez + 2][r + 1] = gtk_pixmap_new (stock.pix[BR], NULL);
      for (i = 1; i < holez + 2; i++)
	gtpix[i][r + 1] = gtk_pixmap_new (stock.pix[B], NULL);

      for (i = roundz + 2; i < r + 2; i++)
	{
	  gtk_fixed_put (GTK_FIXED (gtable), gtpix[0][i], 0,
			 stock.ext + (i - 1) * stock.base);
	  for (j = 1; j < holez + 3; j++)
	    gtk_fixed_put (GTK_FIXED (gtable), gtpix[j][i],
			   stock.ext + (j - 1) * stock.base,
			   stock.ext + (i - 1) * stock.base);
	}

      for (i = roundz; i < r; i++)
	for (j = 0; j < holez; j++)
	  {
	    gcmd[j][i] = gtk_event_box_new ();
	    gtk_widget_shape_combine_mask (GTK_WIDGET (gcmd[j][i]),
					   stock.bit[MTOK], 0, 0);
	    gtk_widget_add_events (gcmd[j][i], GDK_BUTTON_PRESS_MASK);
	    gtk_signal_connect (GTK_OBJECT (gcmd[j][i]), "button_press_event",
				GTK_SIGNAL_FUNC (getcmd),
				GINT_TO_POINTER (i * MAXR + j));

	    tok[j][i] = gtk_pixmap_new (stock.pix[TOK + 0], NULL);
	    gtk_container_add (GTK_CONTAINER (gcmd[j][i]), tok[j][i]);

	    gtk_fixed_put (GTK_FIXED (gtable), gcmd[j][i],
			   stock.ext + (j + 1) * stock.base,
			   stock.ext + i * stock.base);

	    pin[i][j] =
	      gtk_pixmap_new (stock.pix[PIN + 0], stock.bit[MPIN + j]);
	    gtk_fixed_put (GTK_FIXED (gtable), pin[i][j], stock.ext,
			   stock.ext + i * stock.base);
	  }
    }
  roundz = r;

  gtk_widget_show_all (gtable);
  gtk_widget_show_all (ptable);
}

/**
 * build_board
 *
 * No surprise, builds the boards... :)
 **/
void
build_board ()
{
  gnome_window_icon_set_default_from_file (g_strconcat
					   (GNOME_ICONDIR,
					    G_DIR_SEPARATOR_S,
					    "gnomermind.png", NULL));

  gwin = gnome_app_new (PACKAGE, g_strconcat (app, " ", VERSION, NULL));
  gtk_window_set_policy (GTK_WINDOW (gwin), FALSE, TRUE, TRUE);
  gtk_window_set_wmclass (GTK_WINDOW (gwin), "GnomerMind", "board");
  gtk_signal_connect (GTK_OBJECT (gwin), "delete-event",
		      GTK_SIGNAL_FUNC (gtk_main_quit), NULL);
  gtk_signal_connect (GTK_OBJECT (gwin), "destroy",
		      GTK_SIGNAL_FUNC (gtk_main_quit), NULL);
  gtk_widget_add_events (gwin, GDK_KEY_PRESS_MASK);
  gtk_signal_connect (GTK_OBJECT (gwin), "key_press_event",
		      GTK_SIGNAL_FUNC (key_press), NULL);
  add_bars (gwin);

  pwin = gnome_app_new (app, _("Palette"));
  gtk_window_set_policy (GTK_WINDOW (pwin), FALSE, TRUE, TRUE);
  gtk_window_set_wmclass (GTK_WINDOW (pwin), "GnomerMind", "palette");
  gtk_signal_connect (GTK_OBJECT (pwin), "delete-event",
		      GTK_SIGNAL_FUNC (gtk_main_quit), NULL);
  gtk_signal_connect (GTK_OBJECT (pwin), "destroy",
		      GTK_SIGNAL_FUNC (gtk_main_quit), NULL);
  gtk_widget_add_events (pwin, GDK_KEY_PRESS_MASK);
  gtk_signal_connect (GTK_OBJECT (pwin), "key_press_event",
		      GTK_SIGNAL_FUNC (key_press), NULL);

  swin = gnome_app_new (app, "");
  gtk_window_set_wmclass (GTK_WINDOW (swin), "GnomerMind", "solution");
  gtk_window_set_policy (GTK_WINDOW (swin), FALSE, TRUE, TRUE);
  gtk_signal_connect (GTK_OBJECT (swin), "delete-event",
		      GTK_SIGNAL_FUNC (hide_n_seek), NULL);
  gtk_signal_connect (GTK_OBJECT (swin), "destroy",
		      GTK_SIGNAL_FUNC (hide_n_seek), NULL);
  gtk_window_set_transient_for (GTK_WINDOW (swin), GTK_WINDOW (gwin));
  gtk_widget_add_events (swin, GDK_KEY_PRESS_MASK);
  gtk_signal_connect (GTK_OBJECT (swin), "key_press_event",
		      GTK_SIGNAL_FUNC (key_press), NULL);

  /*
   * apply the geometry settings - taken from
   * GnomeHello - (c) 1999, Havoc Pennington <hp@pobox.com>
   */

  if (geometry != NULL)
    {
      gint x, y, w, h;
      if (gnome_parse_geometry (geometry, &x, &y, &w, &h))
	{
	  if (x != -1)
	    {
	      gtk_widget_set_uposition (gwin, x, y);
	    }

	  if (w != -1)
	    {
	      gtk_window_set_default_size (GTK_WINDOW (gwin), w, h);
	    }
	}
      else
	{
	  g_warning (_("Could not parse the geometry string you provided"));
	  geometry = NULL;
	}
    }

  /*
   * we'll create an holez=0 colorz=0 roundz=0 board and
   * then call inflate_board to complete it (hence the name!;)
   */

  gtable = gtk_fixed_new ();
  ptable = gtk_fixed_new ();
  stable = gtk_fixed_new ();

  gtpix[0][0] = gtk_pixmap_new (stock.pix[UL], NULL);
  gtpix[1][0] = gtk_pixmap_new (stock.pix[U], NULL);
  gtpix[2][0] = gtk_pixmap_new (stock.pix[UR], NULL);
  gtpix[0][1] = gtk_pixmap_new (stock.pix[BL], NULL);
  gtpix[1][1] = gtk_pixmap_new (stock.pix[B], NULL);
  gtpix[2][1] = gtk_pixmap_new (stock.pix[BR], NULL);

  gtk_fixed_put (GTK_FIXED (gtable), gtpix[0][0], 0, 0);
  gtk_fixed_put (GTK_FIXED (gtable), gtpix[1][0], stock.ext, 0);
  gtk_fixed_put (GTK_FIXED (gtable), gtpix[2][0], stock.ext + stock.base, 0);
  gtk_fixed_put (GTK_FIXED (gtable), gtpix[0][1], 0, stock.ext);
  gtk_fixed_put (GTK_FIXED (gtable), gtpix[1][1], stock.ext, stock.ext);
  gtk_fixed_put (GTK_FIXED (gtable), gtpix[2][1], stock.ext + stock.base,
		 stock.ext);

  ptpix[0][0] = gtk_pixmap_new (stock.pix[UL], NULL);
  ptpix[1][0] = gtk_pixmap_new (stock.pix[U], NULL);
  ptpix[2][0] = gtk_pixmap_new (stock.pix[UR], NULL);
  ptpix[0][1] = gtk_pixmap_new (stock.pix[L], NULL);
  ptpix[1][1] = gtk_pixmap_new (stock.pix[TILE], NULL);
  ptpix[2][1] = gtk_pixmap_new (stock.pix[R], NULL);
  ptpix[0][2] = gtk_pixmap_new (stock.pix[BL], NULL);
  ptpix[1][2] = gtk_pixmap_new (stock.pix[B], NULL);
  ptpix[2][2] = gtk_pixmap_new (stock.pix[BR], NULL);

  gtk_fixed_put (GTK_FIXED (ptable), ptpix[0][0], 0, 0);
  gtk_fixed_put (GTK_FIXED (ptable), ptpix[1][0], stock.ext, 0);
  gtk_fixed_put (GTK_FIXED (ptable), ptpix[2][0], stock.ext + stock.base, 0);
  gtk_fixed_put (GTK_FIXED (ptable), ptpix[0][1], 0, stock.ext);
  gtk_fixed_put (GTK_FIXED (ptable), ptpix[1][1], stock.ext, stock.ext);
  gtk_fixed_put (GTK_FIXED (ptable), ptpix[2][1], stock.ext + stock.base,
		 stock.ext);
  gtk_fixed_put (GTK_FIXED (ptable), ptpix[0][2], 0, stock.ext + stock.base);
  gtk_fixed_put (GTK_FIXED (ptable), ptpix[1][2], stock.ext,
		 stock.ext + stock.base);
  gtk_fixed_put (GTK_FIXED (ptable), ptpix[2][2], stock.ext + stock.base,
		 stock.ext + stock.base);

  stpix[0][0] = gtk_pixmap_new (stock.pix[UL], NULL);
  stpix[1][0] = gtk_pixmap_new (stock.pix[UR], NULL);
  stpix[0][1] = gtk_pixmap_new (stock.pix[L], NULL);
  stpix[1][1] = gtk_pixmap_new (stock.pix[R], NULL);
  stpix[0][2] = gtk_pixmap_new (stock.pix[BL], NULL);
  stpix[1][2] = gtk_pixmap_new (stock.pix[BR], NULL);

  gtk_fixed_put (GTK_FIXED (stable), stpix[0][0], 0, 0);
  gtk_fixed_put (GTK_FIXED (stable), stpix[1][0], stock.ext, 0);
  gtk_fixed_put (GTK_FIXED (stable), stpix[0][1], 0, stock.ext);
  gtk_fixed_put (GTK_FIXED (stable), stpix[1][1], stock.ext, stock.ext);
  gtk_fixed_put (GTK_FIXED (stable), stpix[0][2], 0, stock.ext + stock.base);
  gtk_fixed_put (GTK_FIXED (stable), stpix[1][2], stock.ext,
		 stock.ext + stock.base);

  ccmd = gtk_event_box_new ();
  gtk_widget_shape_combine_mask (GTK_WIDGET (ccmd), stock.bit[MSTT], 0, 0);
  gtk_widget_add_events (ccmd, GDK_BUTTON_PRESS_MASK);
  gtk_signal_connect (GTK_OBJECT (ccmd), "button_press_event",
		      GTK_SIGNAL_FUNC (cgetcmd), NULL);

  ctrl = gtk_pixmap_new (stock.pix[STT], NULL);
  gtk_container_add (GTK_CONTAINER (ccmd), ctrl);
  gtk_fixed_put (GTK_FIXED (ptable), ccmd, stock.ext, stock.ext);

  gnome_app_set_contents (GNOME_APP (gwin), gtable);
  gnome_app_set_contents (GNOME_APP (pwin), ptable);
  gnome_app_set_contents (GNOME_APP (swin), stable);

  h = holez;
  c = colorz;
  r = roundz;
  holez = 0;
  colorz = 0;
  roundz = 0;
  inflate_board ();

  gtk_widget_show_all (gwin);
  gtk_widget_show_all (pwin);
}
