/* GnomerMind - A MasterMind(R)-based game for Gnome
 * (C) 2001 Germano Rizzo
 *
 * themes.c - routines to control themes
 * Author: Germano Rizzo
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "common.h"
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <unistd.h>
#include <dirent.h>
#include <sys/stat.h>
#include <string.h>
#include "menu.h"
#include "cache.h"

static const gchar *invalid_theme[] = {
  N_("The file %s does not exist"),
  N_("The dimensions of the file %s seem to mismatch"),
  N_("The theme is invalid")
};

GList *theme_list = NULL;
gchar *theme = NULL;
gchar *newtheme = NULL;
gchar *theme_path = NULL;

/**
 * getpath
 * @thname: the theme name
 * @name: the `base' name of the pixmap
 *
 * Returns the complete path of a pixmap
 **/
inline gchar *
getpath (gchar * thname, const gchar * name)
{
  return g_strconcat (theme_path, thname, G_DIR_SEPARATOR_S, g_strdup (name),
		      ".png", NULL);
}

/**
 * check
 * @path: the path of the file
 * @echo: must it show a message on errors or not?
 * 
 * Checks if a file exists
 **/
static gboolean
check (gchar * path, gboolean echo)
{
  struct stat buf;

  if (!stat (path, &buf) && S_ISREG (buf.st_mode))
    return TRUE;

  if (echo)
    {
      g_warning (invalid_theme[0], path);
      g_warning (invalid_theme[2]);
    }

  return FALSE;
}

/**
 * verify
 * @pixpath: the path of the pixmap
 *
 * Returns the path of a pixmap and verifies its existence
 **/
void
verify (gchar * pixpath)
{
  if (!check (pixpath, FALSE))
    {
      gchar *row1, *row2, *row3;

      row1 = _("Pixmap files not correctly installed.");
      row2 = g_strdup_printf (_("The file %s can't be loaded"), pixpath);
      row3 = _("Try installing again...");
      g_error ("%s\n%s\n%s\n", row1, row2, row3);
    }
}

/**
 * getdims
 * @ppat: the file path
 * @w: pointer to the width
 * @h: pointer to the height
 * 
 * Returns the dimensions of a pixmap
 **/
static void
getdims (gchar * ppat, gint * w, gint * h)
{
  GdkPixbuf *tmpimg;

  tmpimg = gdk_pixbuf_new_from_file (ppat);

  g_assert (tmpimg != NULL);

  *w = gdk_pixbuf_get_width (tmpimg);
  *h = gdk_pixbuf_get_height (tmpimg);

  gdk_pixbuf_unref (tmpimg);
}

/**
 * getabout
 *
 * Returns the ABOUT file path of the current theme
 **/
inline gchar *
getabout (gchar * thname)
{
  return g_strconcat (theme_path, thname, G_DIR_SEPARATOR_S, "ABOUT", NULL);
}

/**
 * checkdim
 * @path: the path of the pixmap
 * @w: the width to compare
 * @h: the height to compare
 *
 * Checks if the dimensions of a pixmap are the expected ones
 **/
static gboolean
checkdim (gchar * path, gint w, gint h)
{
  if (check (path, TRUE))
    {
      gint pw, ph;

      getdims (path, &pw, &ph);

      if ((w == pw) && (h == ph))
	return TRUE;

      g_warning (invalid_theme[1], path);
      g_warning (invalid_theme[2]);
      return FALSE;
    }

  return FALSE;
}

/**
 * validate
 * @tname: the theme name
 *
 * Validates a given theme
 **/
gboolean
validate (gchar * tname)
{
  gint i, b, e;

  if (!check (getpath (tname, "l"), TRUE))
    return FALSE;

  getdims (getpath (tname, "l"), &e, &b);

  /*
   * for the `regular' BASExBASE items
   */

  for (i = 0; i < PIX - 8; i++)
    if (!checkdim (getpath (tname, pix_base_name[i]), b, b))
      return FALSE;

  /*
   * for the lower and upper borders
   */

  for (i = PIX - 8; i < PIX - 6; i++)
    if (!checkdim (getpath (tname, pix_base_name[i]), b, e))
      return FALSE;

  /*
   * for the left and right ones
   */

  for (i = PIX - 6; i < PIX - 4; i++)
    if (!checkdim (getpath (tname, pix_base_name[i]), e, b))
      return FALSE;

  /*
   * for the corners
   */

  for (i = PIX - 4; i < PIX; i++)
    if (!checkdim (getpath (tname, pix_base_name[i]), e, e))
      return FALSE;

  /*
   * for the masks
   */

  for (i = 0; i < BIT; i++)
    if (!checkdim (getpath (tname, bit_base_name[i]), b, b))
      return FALSE;

  /*
   * for the ABOUT file
   */

  if (!check (getabout (tname), TRUE))
    return FALSE;

  return TRUE;
}

/**
 * gen_theme_list
 *
 * (Re)generates a list with the valid themes found
 **/
void
gen_theme_list ()
{
  struct dirent **diritem;
  gint n, i;

  if (theme_list)
    g_list_free (theme_list);

  theme_list = NULL;

  n = scandir (THEMESPATH, &diritem, 0, alphasort);

  for (i = n - 1; i >= 0; i--)
    {
      gchar *dir;

      dir = diritem[i]->d_name;
      if (dir[0] != '.')
	if (validate (dir))
	  theme_list = g_list_prepend (theme_list, dir);
    }

  if (!theme_list)
    g_error (_("No valid theme found!"));
}

/**
 * theme_sel
 * @widget: ignored, only for callback
 * @data: the selected string
 *
 * Manages a GtkOptionsMenu selection
 **/
void
theme_sel (GtkWidget * widget, gchar * data)
{
  newtheme = data;

  g_assert (newtheme);
}

/**
 * theme_compare
 *
 * Tells if the theme selection is different
 **/
inline gboolean
theme_compare ()
{
  return (strcmp (theme, newtheme) == 0);
}

/**
 * theme_set
 *
 * Makes the current selection the actual theme
 **/
gboolean
theme_set ()
{
  theme_path = THEMESPATH;
  if (validate (newtheme))
    {
      theme_cache_free ();
      theme = newtheme;
      theme_cache_load ();
      return TRUE;
    }
  gen_theme_list ();
  newtheme = theme;
  return FALSE;
}

/**
 * theme_about
 *
 * Displays the about dialog for the theme
 **/
void
theme_about ()
{
  gchar **fields;
  gint i = 0;

  fields = g_strsplit (stock.about, "<-->", 4);

  while (fields[i++]);

  if (i > 4)
    {
      const gchar *authors[] = { fields[2], NULL };

      GtkWidget *tabo;
      gchar *name;

      msgpush (_("*** To Paola ***"));

      name = g_strconcat ("The ", theme, " theme", NULL);

      tabo =
	gnome_about_new (name, fields[0], fields[1], authors, fields[3],
			 NULL);
      gtk_window_set_modal (GTK_WINDOW (tabo), TRUE);

      gtk_signal_connect (GTK_OBJECT (tabo), "clicked",
			  GTK_SIGNAL_FUNC (msgpop), NULL);
      gtk_signal_connect (GTK_OBJECT (tabo), "close",
			  GTK_SIGNAL_FUNC (msgpop), NULL);

      gtk_widget_show (tabo);
    }
  else
    g_warning (_("Invalid ABOUT file for theme %s"), theme);

  g_strfreev (fields);
}

/**
 * theme_init
 *
 * Initializes the various routines
 **/
inline void
theme_init ()
{
  theme_path = THEMESPATH;
  gen_theme_list ();
}
