/* GnomeScanUI - Widgets for scan dialogs
 *
 * gnomescanacquisitiondialog.c
 *
 * Copyright © 2006 Étienne Bersac
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <gnomescan.h>
#include "gnomescanui.h"
#include "gnomescanui-intl.h"

#define	GNOME_SCAN_ACQUISITION_DIALOG_PARENT_CLASS(klass)	(GTK_WIDGET_CLASS (g_type_class_peek_parent ((klass))))
#define GNOME_SCAN_ACQUISITION_DIALOG_ERROR			(g_type_qname (GNOME_TYPE_SCAN_ACQUISITION_DIALOG))
#define GET_PRIVATE(obj)					(G_TYPE_INSTANCE_GET_PRIVATE ((obj), GNOME_TYPE_SCAN_ACQUISITION_DIALOG, GnomeScanAcquisitionDialogPrivate))

typedef struct _GnomeScanAcquisitionDialogPrivate		GnomeScanAcquisitionDialogPrivate;

struct _GnomeScanAcquisitionDialogPrivate {
  gboolean		dispose_has_run;
  GtkWidget		*progressbar;
  GtkWidget		*label;
  gint			image_size;
  gint			resolution;
};

enum {
  PROP_0,
  PROP_CONTEXT
};

void		cancel								(GtkWidget *widget,
										 GnomeScanAcquisitionDialog *dialog);

void		gnome_scan_acquisition_dialog_acquisition_started		(GnomeScanContext *context,
										 gsize image_size,
										 GnomeScanAcquisitionDialog *dialog);

void		gnome_scan_acquisition_dialog_data_received			(GnomeScanContext *context,
										 gsize data_size,
										 GnomeScanAcquisitionDialog *dialog);

void		gnome_scan_acquisition_dialog_acquisition_terminated		(GnomeScanContext *context,
										 GnomeScanAcquisitionDialog *dialog);


/********************************
 * 	      GOBJECT		*
 ********************************/

void		gnome_scan_acquisition_dialog_dispose				(GObject *obj);

void		gnome_scan_acquisition_dialog_set_property 			(GObject *obj,
										 guint property_id,
										 const GValue *value,
										 GParamSpec *pspec);

void		gnome_scan_acquisition_dialog_get_property 			(GObject *obj,
										 guint property_id,
										 GValue *value,
										 GParamSpec *pspec);

G_DEFINE_TYPE (GnomeScanAcquisitionDialog, gnome_scan_acquisition_dialog, GTK_TYPE_WINDOW);


void
gnome_scan_acquisition_dialog_class_init (GnomeScanAcquisitionDialogClass *klass)
{
  GObjectClass* gobject_class = G_OBJECT_CLASS (klass);

  gobject_class->set_property	= gnome_scan_acquisition_dialog_set_property;
  gobject_class->get_property	= gnome_scan_acquisition_dialog_get_property;
  gobject_class->dispose   	= gnome_scan_acquisition_dialog_dispose;

  g_type_class_add_private (gobject_class, sizeof (GnomeScanAcquisitionDialogPrivate));

  /* Properties */

  g_object_class_install_property (gobject_class,
				   PROP_CONTEXT,
				   g_param_spec_object ("context",
							"Context",
							"The GnomeScanContext the widget is connected to.",
							GNOME_TYPE_SCAN_CONTEXT,
							G_PARAM_READWRITE));
}
void
gnome_scan_acquisition_dialog_init (GnomeScanAcquisitionDialog *widget)
{
  GnomeScanAcquisitionDialogPrivate *priv = GET_PRIVATE (widget);

  priv->dispose_has_run = FALSE;
  widget->context = NULL;
}

void
gnome_scan_acquisition_dialog_dispose (GObject *obj)
{
  GnomeScanAcquisitionDialog *widget = GNOME_SCAN_ACQUISITION_DIALOG (obj);
  GnomeScanAcquisitionDialogPrivate *priv = GET_PRIVATE (widget);
  GnomeScanAcquisitionDialogClass *b_klass = GNOME_SCAN_ACQUISITION_DIALOG_GET_CLASS (obj);

  /* That would be nice if g_return_if_fail were noiseless. */
  if (priv->dispose_has_run == TRUE) {
    return;
  }

  /* unref devices */
  g_object_unref (widget->context);
  priv->dispose_has_run = TRUE;

  /* chain */
  /*   GNOME_SCAN_ACQUISITION_DIALOG_PARENT_CLASS (b_klass)->dispose (obj); */
}

void
gnome_scan_acquisition_dialog_set_property (GObject *obj,
					    guint property_id,
					    const GValue *value,
					    GParamSpec *pspec)
{
  GnomeScanAcquisitionDialog *dialog = GNOME_SCAN_ACQUISITION_DIALOG(obj);

  switch (property_id) {
  case PROP_CONTEXT:
    dialog->context = GNOME_SCAN_CONTEXT (g_value_dup_object (value));
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID(obj, property_id,pspec);
    break;
  }
}

void
gnome_scan_acquisition_dialog_get_property (GObject *obj,
					    guint property_id,
					    GValue *value,
					    GParamSpec *pspec)
{
  GnomeScanAcquisitionDialog *dialog = GNOME_SCAN_ACQUISITION_DIALOG(obj);

  switch (property_id) {
  case PROP_CONTEXT:
    g_value_set_object (value, dialog->context);
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID(obj,property_id,pspec);
    break;
  }
}

/********************************
 * 	      METHODS		*
 ********************************/

/**
 * gnome_scan_acquisition_dialog_new:
 * @context: a #GnomeScanContext
 * @parent: a #GtkWindow
 * 
 * Create a new #GnomeScanAcquisitionDialog connected to @context. If
 * @parent is not NULL, the new #GnomeScanAcquisitionDialog is set
 * transient for @parent.
 * 
 * Return value: a new #GnomeScanAcquisitionDialog
 */
GtkWidget*
gnome_scan_acquisition_dialog_new (GnomeScanContext *context,
				   GtkWindow *parent)
{
  GtkWidget *dialog, *widget, *vbox, *bbox;
  GnomeScanAcquisitionDialog *ad;
  GnomeScanAcquisitionDialogPrivate *priv;
  gchar *string;

  dialog = GTK_WIDGET (g_object_new (GNOME_TYPE_SCAN_ACQUISITION_DIALOG,
				     "context", context,
				     NULL));

  ad = GNOME_SCAN_ACQUISITION_DIALOG (dialog);
  priv = GET_PRIVATE (ad);

  gtk_widget_hide (dialog);

  /* TODO: provide gnome-scan icon */
  gtk_window_set_icon_name (GTK_WINDOW (dialog),
			    "input-scanner");

  /* Build dialog */

  gtk_window_set_title (GTK_WINDOW (dialog),
			"Acquisition ...");

  gtk_window_set_transient_for (GTK_WINDOW (dialog),
				parent);

  gtk_window_set_position (GTK_WINDOW (dialog),
			   GTK_WIN_POS_CENTER_ON_PARENT);

  gtk_container_set_border_width (GTK_CONTAINER (dialog),
				  12);

  vbox = gtk_vbox_new (FALSE, 0);

  gtk_container_add (GTK_CONTAINER (dialog),
		     vbox);

  /* LABEL */
  string = g_strconcat ("<big><b>", _("Acquisition"), "</b></big>\n\n",
			_("The program is receiving data. Please be patient."),
			NULL);

  widget = gtk_label_new (string);
  g_free (string);

  gtk_label_set_use_markup (GTK_LABEL (widget), TRUE);

  gtk_label_set_line_wrap (GTK_LABEL (widget), TRUE);

  gtk_box_pack_start (GTK_BOX (vbox),
		      widget,
		      TRUE, TRUE, 6);

  /* PROGRESSBAR */
  widget = gtk_progress_bar_new ();

  gtk_box_pack_start (GTK_BOX (vbox),
		      widget,
		      FALSE, TRUE, 0);

  priv->progressbar = widget;

  /* LABEL */
  widget = gtk_label_new (NULL);
  priv->label = widget;

  gtk_label_set_use_markup (GTK_LABEL (widget), TRUE);

  gtk_label_set_justify (GTK_LABEL (widget),
			 GTK_JUSTIFY_LEFT);

  gtk_box_pack_start (GTK_BOX (vbox),
		      widget,
		      FALSE, TRUE, 0);

  /* buttons */
  bbox = gtk_hbutton_box_new ();
  gtk_button_box_set_layout (GTK_BUTTON_BOX (bbox),
			     GTK_BUTTONBOX_END);

  gtk_box_pack_start (GTK_BOX (vbox),
		      bbox,
		      FALSE, TRUE, 6);

  widget = gtk_button_new_from_stock (GTK_STOCK_CANCEL);

  gtk_box_pack_start (GTK_BOX (bbox),
		      widget,
		      FALSE, FALSE, 6);

  g_signal_connect (GTK_BUTTON (widget),
		    "clicked",
		    G_CALLBACK (cancel),
		    ad);


  /* SIGNALS */
  g_signal_connect (context,
		    "acquisition-started",
		    (GCallback) gnome_scan_acquisition_dialog_acquisition_started,
		    ad);

  g_signal_connect (context,
		    "data-received",
		    (GCallback) gnome_scan_acquisition_dialog_data_received,
		    ad);

  g_signal_connect (context,
		    "acquisition-terminated",
		    (GCallback) gnome_scan_acquisition_dialog_acquisition_terminated,
		    ad);

  return dialog;
}

void
gnome_scan_acquisition_dialog_acquisition_started (GnomeScanContext *context,
						   gsize image_size,
						   GnomeScanAcquisitionDialog *dialog)
{
  GnomeScanAcquisitionDialogPrivate *priv = GET_PRIVATE (dialog);
  gchar *string;

  priv->image_size = image_size;

  string = g_strdup_printf (_("0KB of %iKB"), image_size / 1024);
  gtk_progress_bar_set_text (GTK_PROGRESS_BAR (priv->progressbar), string);
  g_free (string);

  gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (priv->progressbar), 0.0);

  string = g_strconcat ("<i>", _("Preparing acquisition ..."), "</i>", NULL);
  gtk_label_set_markup (GTK_LABEL (priv->label),
			string);
  g_free (string);

  gtk_widget_set_sensitive (GTK_WIDGET (dialog), FALSE);
  gtk_widget_show_all (GTK_WIDGET (dialog));

  while (gtk_events_pending ())
    gtk_main_iteration ();
}

/* Update the progressbar */
void
gnome_scan_acquisition_dialog_data_received (GnomeScanContext *context,
					     gsize data_size,
					     GnomeScanAcquisitionDialog *dialog)
{
  gchar *string;
  gdouble fraction;
  gsize image_size;
  GnomeScanAcquisitionDialogPrivate *priv = GET_PRIVATE (dialog);

  gtk_widget_set_sensitive (GTK_WIDGET (dialog), TRUE);

  /* workaround mis computed image_size */
  priv->image_size = MAX (priv->image_size,
			  data_size);
  
  string = g_strdup_printf (_("%iKB of %iKB"), data_size / 1024, priv->image_size / 1024);

  gtk_progress_bar_set_text (GTK_PROGRESS_BAR (priv->progressbar), string);
  g_free (string);

  fraction = (gdouble) data_size / (gdouble) image_size;

  /* TODO: set markup one for all, not at each receiv */
  string = g_strconcat ("<i>", _("Receving data ..."), "</i>", NULL);
  gtk_label_set_markup (GTK_LABEL (priv->label),
			string);
  g_free (string);

  gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (priv->progressbar),
				 fraction);

  while (gtk_events_pending ())
    gtk_main_iteration ();
}

void
gnome_scan_acquisition_dialog_acquisition_terminated (GnomeScanContext *context,
						      GnomeScanAcquisitionDialog *dialog)
{
  gtk_widget_hide (GTK_WIDGET (dialog));

  while (gtk_events_pending ())
    gtk_main_iteration ();
}

void
cancel (GtkWidget *widget,
	GnomeScanAcquisitionDialog *dialog)
{
  GnomeScanAcquisitionDialogPrivate *priv = GET_PRIVATE (dialog);
  gchar *string;

  gtk_widget_set_sensitive (GTK_WIDGET (dialog), FALSE);
  string = g_strconcat ("<i>", _("Canceling acquisition ..."), "</i>", NULL);
  gtk_label_set_markup (GTK_LABEL (priv->label),
			string);
  g_free (string);

  while (gtk_events_pending ())
    gtk_main_iteration ();

  gnome_scan_context_cancel_acquisition (dialog->context);
}
