;; -*-scheme-*-
;; 
;;  Richard -Gilligan- Uschold
;; These are TXF codes and a brief description of each. See taxtxf.scm
;; and txf-export-help.scm
;;
;; See also http://www.turbotax.com/txf/
;;
;; Updated Feb 2013 & Jan 2014. J. Alex Aycinena
;; Added updated tax line info
;; Updated Oct 2009. J. Alex Aycinena
;; Added 'txf-tax-entity-types' and related getter functions
;; Added 'tax-entity-type' argument to tax code getter functions
;; Updated txf data for individual tax returns to version 041 (from 039) and
;;    added tax-line year-effectivity data and code last-year data
;; Added asset and liability/equity tax code categories
;; Added version 041 txf data for Partnership, Corporation, S Corporation,
;;    tax entity types
;; Added 'None' type for no income tax options
;;

(use-modules (gnucash app-utils))

(define txf-tax-entity-types
  (list
   (cons 'F1040 #("Individual, Joint, etc." "Files US Form 1040 Tax Return"))
   (cons 'F1065 #("Partnership" "Files US Form 1065 Tax Return"))
   (cons 'F1120 #("Corporation" "Files US Form 1120 Tax Return"))
   (cons 'F1120S #("S Corporation" "Files US Form 1120S Tax Return"))
   (cons 'Other #("None" "No Income Tax Options Provided"))))

(define (gnc:tax-type-txf-get-code-info tax-entity-types type-code index)
  (if (assv type-code tax-entity-types)
      (let ((tax-entity-type (assv type-code tax-entity-types)))
           (and tax-entity-type
                (vector-ref (cdr tax-entity-type) index)))
      #f))

(define (gnc:txf-get-tax-entity-type type-code)
  (gnc:tax-type-txf-get-code-info txf-tax-entity-types type-code 0))

(define (gnc:txf-get-tax-entity-type-description type-code)
  (gnc:tax-type-txf-get-code-info txf-tax-entity-types type-code 1))

(define (gnc:txf-get-tax-entity-type-codes)
  (map car txf-tax-entity-types))

(define (gnc:txf-get-payer-name-source categories code tax-entity-type)
  (gnc:txf-get-code-info categories code 0 tax-entity-type))
(define (gnc:txf-get-form categories code tax-entity-type)
  (gnc:txf-get-code-info categories code 1 tax-entity-type))
(define (gnc:txf-get-description categories code tax-entity-type)
  (gnc:txf-get-code-info categories code 2 tax-entity-type))
(define (gnc:txf-get-format categories code tax-entity-type)
  (gnc:txf-get-code-info categories code 3 tax-entity-type))
(define (gnc:txf-get-multiple categories code tax-entity-type)
  (gnc:txf-get-code-info categories code 4 tax-entity-type))
(define (gnc:txf-get-category-key categories code tax-entity-type)
  (gnc:txf-get-code-info categories code 5 tax-entity-type))
(define (gnc:txf-get-line-data categories code tax-entity-type)
  (if (assv (string->symbol tax-entity-type) categories)
      (let* ((tax-entity-codes (cdr (assv (string->symbol tax-entity-type)
                                          categories)))
             (category (if (assv code tax-entity-codes)
                           (assv code tax-entity-codes)
                           #f)))
            (if (or (not category) (< (vector-length (cdr category)) 7))
                #f
                (gnc:txf-get-code-info categories code 6 tax-entity-type)))
      #f))
(define (gnc:txf-get-last-year categories code tax-entity-type)
  (if (assv (string->symbol tax-entity-type) categories)
      (let* ((tax-entity-codes (cdr (assv (string->symbol tax-entity-type)
                                          categories)))
             (category (if (assv code tax-entity-codes)
                           (assv code tax-entity-codes)
                           #f)))
            (if (or (not category) (< (vector-length (cdr category)) 8))
                #f
                (gnc:txf-get-code-info categories code 7 tax-entity-type)))
      #f))

(define (gnc:txf-get-help categories code)
  (let ((pair (assv code txf-help-strings)))
    (if pair
        (cdr pair)
        (_ "No help available.") )))

(define (gnc:txf-get-codes categories tax-entity-type)
  (if (assv (string->symbol tax-entity-type) categories)
      (let ((tax-entity-codes (cdr (assv (string->symbol tax-entity-type)
                                                                 categories))))
           (map car tax-entity-codes))
      #f))

(define (gnc:txf-get-code-info categories code index tax-entity-type)
  (if (or (assv (string->symbol tax-entity-type) categories)
          (eqv? tax-entity-type ""))
      (let* ((tax-entity-codes (cdr (assv (if (eqv? tax-entity-type "")
                                              'F1040
                                              (string->symbol tax-entity-type))
                                          categories)))
             (category (if (assv code tax-entity-codes)
                           (assv code tax-entity-codes)
                           #f)))
            (if category
                (and category
                     (vector-ref (cdr category) index))
                #f))
      #f))

(define txf-help-categories
  (list
   (cons 'H000 #(current "help" "Name of Current account is exported." 0 #f ""))
   (cons 'H002 #(parent "help" "Name of Parent account is exported." 0 #f ""))
   (cons 'H003 #(not-impl "help" "Not implemented yet, Do NOT Use!" 0 #f ""))))

(define txf-income-categories
 (list
  (cons 'F1040
   (list
    (cons 'N000 #(none "" "Tax Report Only - No TXF Export" 0 #f ""))

    (cons 'N256 #(not-impl "F1040" "Form 1040" 1 #f ""))
    (cons 'N258 #(none "F1040" "Sick pay or disab. pay" 1 #f "Sick Pay" ((1981 "7") (1980 "8"))))
    (cons 'N269 #(none "F1040" "Taxable fringe benefits" 1 #f "Taxable fringe" ((1981 "7") (1980 "8"))))
    (cons 'N487 #(current "F1040" "Dividend, non-taxable" 3 #f "_DivInc TaxFree" ((1988 "8b") (1987 "9"))))
    (cons 'N683 #(none "F1040" "Qualified dividend" 3 #f "" ((2003 "9b"))))
    (cons 'N261 #(none "F1040" "Alimony received" 1 #f "Alimony receive" ((1988 "11") (1987 "12") (1986 "11") (1981 "10") (1980 "12"))))
    (cons 'N519 #(none "F1040" "RR retirement income" 1 #f "RR retirement i" ((1994 "20a") (1988 "21a") (1987 "20a") (1984 "21a"))))
    (cons 'N520 #(none "F1040" "RR retirement inc, spouse" 1 #f "RR retirement i" ((1994 "20a") (1988 "21a") (1987 "20a") (1984 "21a"))))
    (cons 'N266 #(none "F1040" "Soc. Sec. income" 1 #f "Social Security" ((1994 "20a") (1988 "21a") (1987 "20a") (1984 "21a"))))
    (cons 'N483 #(none "F1040" "Soc. Sec. income, spouse" 1 #f "Social Security" ((1994 "20a") (1988 "21a") (1987 "20a") (1984 "21a"))))
    (cons 'N257 #(none "F1040" "Other income-misc." 1 #f "Other income, m" ((1994 "21") (1988 "22") (1987 "21") (1984 "22") (1982 "21") (1981 "20") (1980 "21"))))
    (cons 'N259 #(none "F1040" "Prizes, awards" 1 #f "Prizes, awards" ((1994 "21") (1988 "22") (1987 "21") (1984 "22") (1982 "21") (1981 "20") (1980 "21"))))

    (cons 'N285 #(not-impl "Sched B" "Schedule B" 1 #f ""))
    (cons 'N287 #(current "Sched B" "Interest income" 3 #f "_IntInc" ((1990 "1") (1982 "2") (1981 "1a") (1980 "1"))))
    (cons 'N288 #(current "Sched B" "Interest, US government" 3 #f "" ((1990 "1") (1982 "2") (1981 "1a") (1980 "1"))))
    (cons 'N289 #(current "Sched B" "Interest, State and muni bond" 3 #f "" ((1990 "1") (1982 "2") (1981 "1a") (1980 "1"))))
    (cons 'N489 #(current "Sched B" "Interest, non-taxable" 3 #f "_IntInc TaxFree" ((1990 "1") (1982 "2") (1981 "1a") (1980 "1"))))
    (cons 'N490 #(current "Sched B" "Interest, taxed only by fed" 3 #f "" ((1990 "1") (1982 "2") (1981 "1a") (1980 "1"))))
    (cons 'N491 #(current "Sched B" "Interest, taxed only by state" 3 #f "" ((1990 "1") (1982 "2") (1981 "1a") (1980 "1"))))
    (cons 'N492 #(current "Sched B" "Interest, OID bonds" 3 #f "" ((1990 "1") (1982 "2") (1981 "1a") (1980 "1"))))
    (cons 'N524 #(current "Sched B" "Interest, Seller-financed mortgage" 3 #f "Seller-financed" ((1982 "1") (1981 "1a") (1980 "1"))))
    (cons 'N286 #(parent "Sched B" "Total dividend income" 3 #f "_DivInc" ((1990 "5") (1984 "4") (1982 "9") (1980 "3"))))

    (cons 'N291 #(not-impl "Sched C" "Schedule C" 1 #t ""))
    (cons 'N292 #(not-impl "Sched C" "Spouse" 0 #t "" ((1980 ""))))
    (cons 'N319 #(not-impl "Sched C" "Principal business/prof" 2 #t "" ((1980 "A"))))
    (cons 'N293 #(none "Sched C" "Gross receipts or sales" 1 #t "" ((2012 "1") (2011 "1b") (1989 "1") (1980 "1a"))))
    (cons 'N303 #(none "Sched C" "Other business income" 1 #t "" ((1989 "6") (1987 "4") (1981 "4b") (1980 "4"))))

    (cons 'N497 #(not-impl "Sched C-EZ" "Schedule C-EZ" 1 #t ""))
    (cons 'N498 #(not-impl "Sched C-EZ" "Spouse" 0 #t "" ((1992 ""))))
    (cons 'N501 #(not-impl "Sched C-EZ" "Principal business/prof" 2 #t "" ((1992 "A"))))
    (cons 'N499 #(none "Sched C-EZ" "Gross receipts" 1 #t "" ((2012 "1") (2011 "1b") (1992 "1"))))

    (cons 'N320 #(not-impl "Sched D" "Schedule D" 1 #f ""))
    (cons 'N321 #(not-impl "Sched D" "Short Term gain/loss - security" 4 #f "" ((1993 "1") (1991 "1a") (1986 "2a") (1985 "1b") (1980 "1"))))
    (cons 'N322 #(not-impl "Sched D" "ST gain/loss - other" 4 #f "" ((1991 "1d") (1989 "2d") (1986 "2c")) 1992))
    (cons 'N323 #(not-impl "Sched D" "Long Term gain/loss - security" 4 #f "" ((1997 "8") (1993 "9") (1992 "9a") (1991 "8a") (1986 "9a") (1983 "9") (1980 "8"))))
    (cons 'N324 #(not-impl "Sched D" "LT gain/loss - other" 4 #f "" ((1992 "9d") (1991 "8d") (1989 "9d") (1986 "9c")) 1992))
    (cons 'N673 #(current "Sched D" "Short/Long gain or loss" 4 #f "" ((2011 "1, 2, or 3 via Form 8949, line 1 or 8, 9, or 10 via Form 8949, line 3") (1999 "1 or 8"))))
    (cons 'N682 #(not-impl "Sched D" "Wash Sale - security" 4 #f "" ((2003 "1 or 8"))))
    (cons 'N488 #(parent "Sched D" "Dividend, cap gain distrib." 3 #f "_LT CapGnDst" ((1997 "13") (1992 "14") (1991 "12") (1986 "13") (1983 "15") (1980 "13"))))
    (cons 'N684 #(none "Sched D" "Div inc., PostMay5 cap gain" 3 #f "" ((2004 "NA - Expired") (2003 "13g")) 2003))
    (cons 'N644 #(current "Sched D" "28% cap gain" 3 #f "" ((2004 "18") (2003 "20") (1997 "8g"))))
    (cons 'N646 #(current "Sched D" "Sec 1202 gain" 3 #f "" ((2004 "18") (2003 "20") (1998 "8"))))
    (cons 'N645 #(current "Sched D" "Unrec sec 1250" 3 #f "" ((2001 "19") (1997 "25"))))
    (cons 'N677 #(current "Sched D" "Qualified 5-year gain" 3 #f "" ((2004 "NA - Expired") (2003 "35") (2001 "8")) 2003))

    (cons 'N325 #(not-impl "Sched E" "Schedule E" 1 #t ""))
    (cons 'N342 #(not-impl "Sched E" "Kind/location of property" 2 #t "" ((1985 "1") (1984 "2") (1982 "") (1981 "V(a)") (1980 "V"))))
    (cons 'N326 #(none "Sched E" "Rents received" 1 #t "" ((2012 "3") (2011 "3b") (1990 "3") (1987 "4") (1981 "3a") (1980 "8(b)"))))
    (cons 'N327 #(none "Sched E" "Royalties received" 1 #t "" ((2012 "4") (2011 "3b") (1990 "4") (1987 "5") (1981 "3b") (1980 "8(c)"))))

    (cons 'N343 #(not-impl "Sched F" "Schedule F" 1 #t ""))
    (cons 'N514 #(not-impl "Sched F" "Spouse" 0 #t "" ((1990 ""))))
    (cons 'N379 #(not-impl "Sched F" "Principal product" 2 #t "" ((1990 "A"))))
    (cons 'N369 #(none "Sched F" "Resales of livestock" 1 #t "" ((2012 "1a") (2011 "1b") (1990 "1"))))
    (cons 'N368 #(none "Sched F" "Sales livestock/crops raised" 1 #t "" ((2012 "2") (2011 "2a") (1990 "4"))))
    (cons 'N371 #(none "Sched F" "Coop. distributions" 1 #t "" ((2011 "3a") (1990 "5a"))))
    (cons 'N372 #(none "Sched F" "Agricultural program payments" 1 #t "" ((2011 "4a") (1990 "6a"))))
    (cons 'N373 #(none "Sched F" "CCC loans reported/election" 1 #t "" ((2011 "5a") (1990 "7a"))))
    (cons 'N374 #(none "Sched F" "CCC loans forfeited or repaid" 1 #t "" ((2011 "5b") (1990 "7b"))))
    (cons 'N375 #(none "Sched F" "Crop insurance proceeds received" 1 #t "" ((2011 "6a") (1990 "8a"))))
    (cons 'N376 #(none "Sched F" "Crop insurance proceeds deferred" 1 #t "" ((2011 "6d") (1990 "8d"))))
    (cons 'N370 #(none "Sched F" "Custom hire (machine work)" 1 #t "" ((2012 "7") (2011 "7a or b") (1990 "9"))))
    (cons 'N377 #(none "Sched F" "Other farm income" 1 #t "" ((2012 "8") (2011 "8a or b") (1990 "10"))))

    (cons 'N380 #(not-impl "F2106" "Form 2106" 1 #t ""))
    (cons 'N387 #(none "F2106" "Emp. expense reimb." 1 #t "" ((2009 "7") (1990 "7,A"))))
    (cons 'N388 #(none "F2106" "Emp. meal exp reimb" 1 #t "" ((2009 "7") (1990 "7,B"))))

    (cons 'N392 #(not-impl "HomeWks" "Home Sale Worksheets, Pub 523" 1 #t ""))
    (cons 'N525 #(not-impl "HomeWks" "Spouse" 0 #t "" ((2000 "")) 2000))
    (cons 'N398 #(not-impl "HomeWks" "Date old home sold" 2 #t "" ((2002 ""))))
    (cons 'N399 #(not-impl "HomeWks" "Date moved into new home" 2 #t "" ((2000 "11a")) 2000))
    (cons 'N393 #(none "HomeWks" "Selling price of old home" 1 #t "" ((2002 "1"))))

    (cons 'N503 #(not-impl "F4137" "Form 4137" 1 #t ""))
    (cons 'N504 #(not-impl "F4137" "Spouse" 0 #t "" ((1990 ""))))
    (cons 'N505 #(none "F4137" "Cash/charge tips received" 1 #t "" ((2007 "2") (1990 "1"))))

    (cons 'N412 #(not-impl "F4684" "Form 4684" 1 #t ""))
    (cons 'N417 #(not-impl "F4684" "Description of property" 2 #t "" ((1990 "1"))))
    (cons 'N414 #(none "F4684" "Insurance/reimb" 1 #t "" ((1990 "3"))))
    (cons 'N415 #(none "F4684" "FMV before casualty" 1 #t "" ((1990 "5"))))
    (cons 'N416 #(none "F4684" "FMV after casualty" 1 #t "" ((1990 "6"))))

    (cons 'N418 #(not-impl "F4797" "Form 4797" 1 #f ""))
    (cons 'N420 #(not-impl "F4797" "LT dep. gain - business" 5 #f "" ((1990 "2"))))
    (cons 'N423 #(not-impl "F4797" "LT dep. gain - res. rent." 5 #f "" ((1990 "2"))))
    (cons 'N421 #(not-impl "F4797" "ST dep. prop. - business" 5 #f "" ((1997 "10") (1992 "11") (1990 "10"))))
    (cons 'N424 #(not-impl "F4797" "ST dep. prop. - res. rent." 5 #f "" ((1997 "10") (1992 "11") (1990 "10"))))

    (cons 'N569 #(not-impl "F4835" "Form 4835" 1 #t ""))
    (cons 'N570 #(not-impl "F4835" "Spouse" 0 #t "" ((1990 ""))))
    (cons 'N571 #(none "F4835" "Sale of livestock/produce" 1 #t "" ((1990 "1"))))
    (cons 'N572 #(none "F4835" "Total co-op distributions" 1 #t "" ((1990 "2a"))))
    (cons 'N573 #(none "F4835" "Agricultural program paymnts" 1 #t "" ((1990 "3a"))))
    (cons 'N574 #(none "F4835" "CCC loans reported/election" 1 #t "" ((1990 "4a"))))
    (cons 'N575 #(none "F4835" "CCC loans forfeited/repaid" 1 #t "" ((1990 "4b"))))
    (cons 'N576 #(none "F4835" "Crop insurance proceeds received" 1 #t "" ((1990 "5a"))))
    (cons 'N577 #(none "F4835" "Crop insurance proceeds deferred" 1 #t "" ((1990 "5d"))))
    (cons 'N578 #(none "F4835" "Other income" 1 #t "" ((1990 "6"))))

    (cons 'N290 #(current "F6251" "Interest, Tax Exempt private activity bond" 3 #f "" ((2008 "12") (2002 "11") (1993 "13") (1991 "6b") (1990 "5b"))))

    (cons 'N427 #(not-impl "F6252" "Form 6252" 1 #t ""))
    (cons 'N436 #(not-impl "F6252" "Description of property" 2 #t "" ((1992 "1") (1990 "A"))))
    (cons 'N428 #(none "F6252" "Selling price" 1 #t "" ((1992 "5") (1990 "1"))))
    (cons 'N429 #(none "F6252" "Debt assumed by buyer" 1 #t "" ((1992 "6") (1990 "1"))))
    (cons 'N431 #(none "F6252" "Depreciation allowed" 1 #t "" ((1992 "9") (1990 "5"))))
    (cons 'N433 #(not-impl "F6252" "Gross profit percentage" 1 #t "" ((1992 "19") (1990 "15"))))
    (cons 'N434 #(none "F6252" "Payments received this year" 1 #t "" ((1992 "21") (1990 "17"))))
    (cons 'N435 #(none "F6252" "Payments received prior years" 1 #t "" ((1992 "23") (1990 "19"))))

    (cons 'N441 #(not-impl "F8815" "Form 8815" 1 #f ""))
    (cons 'N443 #(none "F8815" "Nontaxable education benefits" 1 #f "" ((1990 "3"))))
    (cons 'N444 #(none "F8815" "EE US svgs. bonds proceeds" 1 #f "" ((1990 "5"))))
    (cons 'N445 #(none "F8815" "Post-89 EE bond face value" 1 #f "" ((1999 "6wksht2") (1990 "6wksht1"))))

    (cons 'N446 #(not-impl "Sched K-1" "Schedule K-1 Worksheet" 1 #t ""))
    (cons 'N447 #(not-impl "Sched K-1" "Spouse" 0 #t "" ((1990 ""))))
    (cons 'N457 #(not-impl "Sched K-1" "Partnership or S corp name" 2 #t "" ((2004 "B") (1990 ""))))
    (cons 'N448 #(parent "Sched K-1" "Ordinary income or loss" 1 #t "" ((1990 "1"))))
    (cons 'N449 #(parent "Sched K-1" "Rental RE income or loss" 1 #t "" ((1990 "2"))))
    (cons 'N450 #(parent "Sched K-1" "Other rental income or loss" 1 #t "" ((1990 "3"))))
    (cons 'N455 #(parent "Sched K-1" "Guaranteed partner payments" 1 #t "" ((2004 "4") (1990 "5"))))
    (cons 'N451 #(parent "Sched K-1" "Interest income" 1 #t "" ((2004 "5") (1990 "4a"))))
    (cons 'N452 #(parent "Sched K-1" "Dividends" 1 #t "" ((2004 "6a") (2003 "4b(2)") (1990 "4b"))))
    (cons 'N527 #(parent "Sched K-1" "Royalties" 1 #t "" ((2004 "7") (1990 "4c"))))
    (cons 'N453 #(parent "Sched K-1" "Net ST capital gain or loss" 1 #t "" ((2004 "8") (2003 "4d2") (1990 "4d"))))
    (cons 'N454 #(parent "Sched K-1" "Net LT capital gain or loss" 1 #t "" ((2004 "9a") (2003 "4e2") (2001 "4e1") (1997 "4e2") (1990 "4e"))))
    (cons 'N674 #(none "Sched K-1" "28% rate gain(loss)" 1 #t "" ((2004 "9b") (2003 "25") (2001 "4e2") (1997 "4e1"))))
    (cons 'N675 #(none "Sched K-1" "Qualified 5-year gain" 1 #t "" ((2003 "NA - Expired") (2001 "4e3")) 2002))
    (cons 'N456 #(parent "Sched K-1" "Net Section 1231 gain or loss" 1 #t "" ((2004 "10") (2003 "6b") (1998 "6") (1997 "6b") (1990 "6"))))
    (cons 'N676 #(none "Sched K-1" "Other Income (loss)" 1 #t "" ((2004 "11") (1990 "7"))))
    (cons 'N528 #(parent "Sched K-1" "Tax-exempt interest income" 1 #t "" ((2004 "18") (1992 "19"))))

    (cons 'N458 #(not-impl "W-2" "W-2" 1 #t ""))
    (cons 'N459 #(not-impl "W-2" "Spouse" 0 #t "" ((1990 "")) 1995))
    (cons 'N466 #(not-impl "W-2" "Payer" 2 #t "" ((1993 "c") (1990 "2"))))
    (cons 'N513 #(not-impl "W-2" "Payer, spouse" 2 #t "" ((1993 "c") (1990 "2"))))
    (cons 'N460 #(parent "W-2" "Salary or wages, self" 1 #t "Salary" ((1993 "1") (1990 "10"))))
    (cons 'N506 #(parent "W-2" "Salary or wages, spouse" 1 #t "Salary" ((1993 "1") (1990 "10"))))
    (cons 'N465 #(parent "W-2" "Dependent care benefits, self" 1 #t "" ((1993 "10") (1991 "22") (1990 "15"))))
    (cons 'N512 #(parent "W-2" "Dependent care benefits, spouse" 1 #t "" ((1993 "10") (1991 "22") (1990 "15"))))
    (cons 'N267 #(parent "W-2" "Reimbursed moving expenses, self" 1 #t "" ((1993 "12") (1991 "23"))))
    (cons 'N546 #(parent "W-2" "Reimbursed moving expenses,spouse" 1 #t "" ((1993 "12") (1991 "23"))))

    (cons 'N547 #(not-impl "W-2G" "W-2G" 1 #t ""))
    (cons 'N548 #(not-impl "W-2G" "Spouse" 0 #t "" ((1996 "")) 1996))
    (cons 'N552 #(not-impl "W-2G" "Payer" 2 #t "" ((1996 "")) 1996))
    (cons 'N549 #(parent "W-2G" "Gross winnings" 1 #t "" ((1990 "1"))))

    (cons 'N467 #(not-impl "W2-P" "W-2P" 1 #t ""))
    (cons 'N468 #(not-impl "W2-P" "Spouse" 0 #t "" ((1990 "")) 1990))
    (cons 'N469 #(none "W2-P" "Pensions and annuities-gross" 3 #t "" ((1990 "9")) 1990))
    (cons 'N470 #(none "W2-P" "Pensions and annuities-txble" 3 #t "" ((1990 "10")) 1990))
    (cons 'N471 #(none "W2-P" "IRA distributions - gross" 3 #t "" ((1990 "9")) 1990))
    (cons 'N472 #(none "W2-P" "IRA distributions - taxable" 3 #t "" ((1990 "10")) 1990))

    (cons 'N643 #(not-impl "F1099-DIV" "1099 Div" 1 #f ""))
    (cons 'N647 #(current "F1099-DIV" "NonTaxable distributions" 3 #t "" ((1998 "3") (1990 "1d"))))
    (cons 'N651 #(current "F1099-DIV" "Cash liquidation" 3 #t "" ((1998 "8") (1990 "5"))))
    (cons 'N652 #(current "F1099-DIV" "Non-cash liquidation" 3 #t "" ((1998 "9") (1990 "6"))))

    (cons 'N634 #(not-impl "F1099-G" "1099-G" 1 #t ""))
    (cons 'N635 #(not-impl "F1099-G" "Spouse" 0 #t "" ((1990 ""))))
    (cons 'N479 #(none "F1099-G" "Unemployment compensation" 1 #t "Unemployment In" ((1990 "1"))))
    (cons 'N260 #(none "F1099-G" "State and local refunds" 1 #t "" ((1990 "2"))))

    (cons 'N553 #(not-impl "F1099-MISC" "1099-MISC" 1 #t ""))
    (cons 'N554 #(not-impl "F1099-MISC" "Spouse" 0 #t "" ((1990 "")) 1996))
    (cons 'N564 #(not-impl "F1099-MISC" "Payer" 2 #t "" ((1990 "")) 1996))
    (cons 'N555 #(parent "F1099-MISC" "Rents" 1 #t "" ((1990 "1"))))
    (cons 'N556 #(parent "F1099-MISC" "Royalties" 1 #t "" ((1990 "2"))))
    (cons 'N557 #(parent "F1099-MISC" "Other income" 1 #t "" ((1990 "3"))))
    (cons 'N559 #(parent "F1099-MISC" "Fishing boat proceeds" 1 #t "" ((1990 "5"))))
    (cons 'N560 #(parent "F1099-MISC" "Medical/health payments" 1 #t "" ((1990 "6"))))
    (cons 'N561 #(parent "F1099-MISC" "Nonemployee compensation" 1 #t "" ((1990 "7"))))
    (cons 'N562 #(parent "F1099-MISC" "Crop insurance proceeds" 1 #t "" ((1990 "10"))))

    (cons 'N657 #(not-impl "F1099-OID" "1099 OID" 1 #f ""))
    (cons 'N658 #(current "F1099-OID" "Other periodic int,OID" 3 #t "" ((1990 "2"))))
    (cons 'N661 #(not-impl "F1099-OID" "Description" 2 #t "" ((2013 "7") (1990 "5"))))
    (cons 'N662 #(current "F1099-OID" "OID, US treas obligl" 3 #t "" ((2013 "8") (1999 "6"))))

    (cons 'N473 #(not-impl "F1099-R" "1099R" 1 #t ""))
    (cons 'N474 #(not-impl "F1099-R" "Spouse" 0 #t "" ((1990 ""))))
    (cons 'N603 #(not-impl "F1099-R" "Pension Payer" 2 #t "" ((1996 ""))))
    (cons 'N475 #(parent "F1099-R" "Pension total dist - gross" 1 #t "" ((1990 "1"))))
    (cons 'N476 #(parent "F1099-R" "Pension total dist - taxable" 1 #t "" ((1991 "2a") (1990 "2"))))
    (cons 'N670 #(not-impl "F1099-R" "Pension Tax. not determined" 2 #t "" ((1991 "2b"))))
    (cons 'N671 #(not-impl "F1099-R" "Pension Total distribution" 2 #t "" ((1991 "2b"))))
    (cons 'N667 #(not-impl "F1099-R" "Pension Distrib code 7a" 2 #t "" ((1990 "7a"))))
    (cons 'N668 #(not-impl "F1099-R" "Pension Distrib code 7b" 2 #t "" ((1990 "7b"))))
    (cons 'N669 #(not-impl "F1099-R" "Pension State ID" 2 #t "" ((1990 "11"))))
    (cons 'N655 #(not-impl "F1099-R" "Distribution code 7a" 2 #t "" ((1990 "7a"))))
    (cons 'N656 #(not-impl "F1099-R" "Distribution code 7b" 2 #t "" ((1990 "7b"))))
    (cons 'N664 #(not-impl "F1099-R" "State ID" 2 #t "" ((1990 "11"))))
    (cons 'N604 #(not-impl "F1099-R" "IRA Payer" 2 #t "" ((1996 ""))))
    (cons 'N477 #(parent "F1099-R" "IRA total dist - gross" 1 #t "" ((1990 "1"))))
    (cons 'N478 #(parent "F1099-R" "IRA total dist - taxable" 1 #t "" ((1991 "2a") (1990 "2"))))
    (cons 'N628 #(not-impl "F1099-R" "SIMPLE Payer" 2 #t "" ((1997 ""))))
    (cons 'N623 #(parent "F1099-R" "SIMPLE total distrib-gross" 1 #t "" ((1990 "1"))))
    (cons 'N624 #(parent "F1099-R" "SIMPLE total distrib-taxable" 1 #t "" ((1991 "2a") (1990 "2"))))
    (cons 'N665 #(not-impl "F1099-R" "Taxable not determined" 2 #t "" ((1991 "2b"))))
    (cons 'N666 #(not-impl "F1099-R" "Total distribution" 2 #t "" ((1991 "2b"))))

    (cons 'N629 #(not-impl "F1099-SA" "1099-SA" 1 #t ""))
    (cons 'N630 #(not-impl "F1099-SA" "Spouse" 0 #t "" ((1997 ""))))
    (cons 'N633 #(not-impl "F1099-SA" "Payer" 2 #t "" ((1997 ""))))
    (cons 'N631 #(none "F1099-SA" "MSA gross distribution" 1 #t "" ((1997 "1"))))
    (cons 'N632 #(none "F1099-SA" "MSA earnings on excess contrib" 1 #t "" ((1997 "2"))))

    (cons 'N678 #(not-impl "F1099-Q" "1099 Q" 1 #f ""))
    (cons 'N672 #(none "F1099-Q" "Gross distribution" 1 #t "" ((2002 "1"))))
   )
  )
  (cons 'F1065
   (list
    (cons 'N000 #(none "" "Tax Report Only - No TXF Export" 0 #f ""))

    (cons 'N1617 #(none "F1065" "Gross receipts or sales" 1 #f "" ((2012 "1a") (2011 "1b") (1990 "1a"))))
    (cons 'N1619 #(none "F1065" "Returns and allowances" 1 #f "" ((2012 "1b") (2011 "1d") (1990 "1b"))))
    (cons 'N1629 #(none "F1065" "Other income" 1 #f "" ((1990 "7"))))

    (cons 'N1804 #(none "F1065" "Income-rental" 1 #f "" ((1990 "K3a"))))
    (cons 'N1809 #(none "F1065" "Interest income-Bank, CDs, etc." 1 #f "" ((2004 "K5") (1990 "K4a"))))
    (cons 'N1810 #(none "F1065" "Interest income-US Treasury obligations" 1 #f "" ((2004 "K5") (1990 "K4a"))))
    (cons 'N1811 #(none "F1065" "Dividend income" 1 #f "" ((2004 "K6a") (1990 "K4b"))))
    (cons 'N1813 #(none "F1065" "Royalty income" 1 #f "" ((2004 "K7") (1990 "K4c"))))
    (cons 'N1825 #(none "F1065" "Other income" 1 #f "" ((2004 "K11") (1990 "K7"))))

    (cons 'N1900 #(none "F1065" "Net income(loss) per books" 1 #f "" ((1991 "M-1,1"))))
    (cons 'N1908 #(none "F1065" "Income on books not on return" 1 #f "" ((1992 "M-1,6") (1991 "M-1,5"))))

    (cons 'N1914 #(none "F1065" "Other increases to partners' cap accts" 1 #f "" ((1991 "M-2,4") (1990 "M(d)"))))

    (cons 'N343 #(not-impl "Sched F" "Schedule F" 1 #t ""))
    (cons 'N514 #(not-impl "Sched F" "Spouse" 0 #t "" ((1990 ""))))
    (cons 'N379 #(not-impl "Sched F" "Principal product" 2 #t "" ((1990 "A"))))
    (cons 'N369 #(none "Sched F" "Resales of livestock" 1 #t "" ((2012 "1a") (2011 "1b") (1990 "1"))))
    (cons 'N368 #(none "Sched F" "Sales livestock/crops raised" 1 #t "" ((2012 "2") (2011 "2a") (1990 "4"))))
    (cons 'N371 #(none "Sched F" "Coop. distributions" 1 #t "" ((2011 "3a") (1990 "5a"))))
    (cons 'N372 #(none "Sched F" "Agricultural program payments" 1 #t "" ((2011 "4a") (1990 "6a"))))
    (cons 'N373 #(none "Sched F" "CCC loans reported/election" 1 #t "" ((2011 "5a") (1990 "7a"))))
    (cons 'N374 #(none "Sched F" "CCC loans forfeited or repaid" 1 #t "" ((2011 "5b") (1990 "7b"))))
    (cons 'N375 #(none "Sched F" "Crop insurance proceeds received" 1 #t "" ((2011 "6a") (1990 "8a"))))
    (cons 'N376 #(none "Sched F" "Crop insurance proceeds deferred" 1 #t "" ((2011 "6d") (1990 "8d"))))
    (cons 'N370 #(none "Sched F" "Custom hire (machine work)" 1 #t "" ((2012 "7") (2011 "7a or b") (1990 "9"))))
    (cons 'N377 #(none "Sched F" "Other farm income" 1 #t "" ((2012 "8") (2011 "8a or b") (1990 "10"))))

    (cons 'N418 #(not-impl "F4797" "Form 4797" 1 #f ""))
    (cons 'N420 #(not-impl "F4797" "LT dep. gain - business" 5 #f "" ((1990 "2"))))
    (cons 'N423 #(not-impl "F4797" "LT dep. gain - res. rent." 5 #f "" ((1990 "2"))))
    (cons 'N421 #(not-impl "F4797" "ST dep. prop. - business" 5 #f "" ((1997 "10") (1992 "11") (1990 "10"))))
    (cons 'N424 #(not-impl "F4797" "ST dep. prop. - res. rent." 5 #f "" ((1997 "10") (1992 "11") (1990 "10"))))

    (cons 'N1744 #(parent "F8825" "Gross rents – A" 1 #t "" ((1990 "2A"))))
    (cons 'N1759 #(parent "F8825" "Gross rents – B" 1 #t "" ((1990 "2B"))))
    (cons 'N1774 #(parent "F8825" "Gross rents – C" 1 #t "" ((1990 "2C"))))
    (cons 'N1789 #(parent "F8825" "Gross rents – D" 1 #t "" ((1990 "2D"))))
    (cons 'N1742 #(none "F8825" "Income-other rental real estate" 1 #t "" ((1990 "20a"))))

    (cons 'N446 #(not-impl "Sched K-1" "Schedule K-1 Worksheet" 1 #t ""))
    (cons 'N447 #(not-impl "Sched K-1" "Spouse" 0 #t "" ((1990 ""))))
    (cons 'N457 #(not-impl "Sched K-1" "Partnership or S corp name" 2 #t "" ((2004 "B") (1990 ""))))
    (cons 'N448 #(parent "Sched K-1" "Ordinary income or loss" 1 #t "" ((1990 "1"))))
    (cons 'N449 #(parent "Sched K-1" "Rental RE income or loss" 1 #t "" ((1990 "2"))))
    (cons 'N450 #(parent "Sched K-1" "Other rental income or loss" 1 #t "" ((1990 "3"))))
    (cons 'N455 #(parent "Sched K-1" "Guaranteed partner payments" 1 #t "" ((2004 "4") (1990 "5"))))
    (cons 'N451 #(parent "Sched K-1" "Interest income" 1 #t "" ((2004 "5") (1990 "4a"))))
    (cons 'N452 #(parent "Sched K-1" "Dividends" 1 #t "" ((2004 "6a") (2003 "4b(2)") (1990 "4b"))))
    (cons 'N527 #(parent "Sched K-1" "Royalties" 1 #t "" ((2004 "7") (1990 "4c"))))
    (cons 'N453 #(parent "Sched K-1" "Net ST capital gain or loss" 1 #t "" ((2004 "8") (2003 "4d2") (1990 "4d"))))
    (cons 'N454 #(parent "Sched K-1" "Net LT capital gain or loss" 1 #t "" ((2004 "9a") (2003 "4e2") (2001 "4e1") (1997 "4e2") (1990 "4e"))))
    (cons 'N674 #(none "Sched K-1" "28% rate gain(loss)" 1 #t "" ((2004 "9b") (2003 "25") (2001 "4e2") (1997 "4e1"))))
    (cons 'N675 #(none "Sched K-1" "Qualified 5-year gain" 1 #t "" ((2003 "NA - Expired") (2001 "4e3")) 2002))
    (cons 'N456 #(parent "Sched K-1" "Net Section 1231 gain or loss" 1 #t "" ((2004 "10") (2003 "6b") (1998 "6") (1997 "6b") (1990 "6"))))
    (cons 'N676 #(none "Sched K-1" "Other Income (loss)" 1 #t "" ((2004 "11") (1990 "7"))))
    (cons 'N528 #(parent "Sched K-1" "Tax-exempt interest income" 1 #t "" ((2004 "18") (1992 "19"))))
   )
  )
  (cons 'F1120
   (list
    (cons 'N000 #(none "" "Tax Report Only - No TXF Export" 0 #f ""))

    (cons 'N1003 #(none "F1120" "Gross receipts or sales" 1 #f "" ((2012 "1a") (2011 "1b") (1990 "1a"))))
    (cons 'N1005 #(none "F1120" "Returns and allowances" 1 #f "" ((2012 "1b") (2011 "1d") (1990 "1b"))))
    (cons 'N1007 #(none "F1120" "Interest income" 1 #f "" ((1990 "5"))))
    (cons 'N1009 #(none "F1120" "Gross rents" 1 #f "" ((1990 "6"))))
    (cons 'N1011 #(none "F1120" "Gross royalties" 1 #f "" ((1990 "7"))))

    (cons 'N1021 #(none "F1120" "Div.--< 20% owned corps." 1 #f "" ((1990 "C1"))))
    (cons 'N1023 #(none "F1120" "Div.--20% or > corps." 1 #f "" ((1990 "C2"))))
    (cons 'N1025 #(none "F1120" "Div.--debt-financed stk." 1 #f "" ((1990 "C3"))))
    (cons 'N1027 #(none "F1120" "Div.--< 20% pref. stk." 1 #f "" ((1990 "C4"))))
    (cons 'N1029 #(none "F1120" "Div.--20% or > pref. stk." 1 #f "" ((1990 "C5"))))
    (cons 'N1031 #(none "F1120" "Div.--< 20% foreign. corps." 1 #f "" ((1990 "C6"))))
    (cons 'N1033 #(none "F1120" "Div.--20% or > foreign corps." 1 #f "" ((1990 "C7"))))
    (cons 'N1035 #(none "F1120" "Div.-wholly owned foreign" 1 #f "" ((1990 "C8"))))
    (cons 'N1037 #(none "F1120" "Div.-domestic corps." 1 #f "" ((1990 "C10"))))
    (cons 'N1039 #(none "F1120" "Div.-from certain FSC's" 1 #f "" ((1990 "C12"))))
    (cons 'N1041 #(none "F1120" "Div.-affiliated group members" 1 #f "" ((1990 "C11"))))
    (cons 'N1043 #(none "F1120" "Oth. div.-foreign. corps." 1 #f "" ((1990 "C13"))))
    (cons 'N1045 #(none "F1120" "Inc.-controlled foreign corps." 1 #f "" ((1990 "C14"))))
    (cons 'N1047 #(none "F1120" "Foreign dividend gross up" 1 #f "" ((1990 "C15"))))
    (cons 'N1049 #(none "F1120" "IC-DISC and former DISC div." 1 #f "" ((1990 "C16"))))
    (cons 'N1051 #(none "F1120" "Other dividends" 1 #f "" ((1990 "C17"))))

    (cons 'N1233 #(none "F1120" "Net income(loss) per books" 1 #f "" ((1990 "M-1,1"))))
    (cons 'N1236 #(none "F1120" "Tax-exempt interest income" 1 #f "" ((1992 "M-1,7") (1990 "M-1,7a"))))
    (cons 'N1237 #(none "F1120" "Income on books not on return" 1 #f "" ((1990 "M-1,7"))))

    (cons 'N1239 #(none "F1120" "Other increases" 1 #f "" ((1990 "M-2,3"))))

    (cons 'N418 #(not-impl "F4797" "Form 4797" 1 #f ""))
    (cons 'N420 #(not-impl "F4797" "LT dep. gain - business" 5 #f "" ((1990 "2"))))
    (cons 'N423 #(not-impl "F4797" "LT dep. gain - res. rent." 5 #f "" ((1990 "2"))))
    (cons 'N421 #(not-impl "F4797" "ST dep. prop. - business" 5 #f "" ((1997 "10") (1992 "11") (1990 "10"))))
    (cons 'N424 #(not-impl "F4797" "ST dep. prop. - res. rent." 5 #f "" ((1997 "10") (1992 "11") (1990 "10"))))
   )
  )
  (cons 'F1120S
   (list
    (cons 'N000 #(none "" "Tax Report Only - No TXF Export" 0 #f ""))

    (cons 'N1259 #(none "F1120S" "Gross receipts or sales" 1 #f "" ((2012 "1a") (2011 "1b") (1990 "1a"))))
    (cons 'N1261 #(none "F1120S" "Returns and allowances" 1 #f "" ((2012 "1b") (2011 "1d") (1990 "1b"))))
    (cons 'N1269 #(none "F1120S" "Other income" 1 #f "" ((1990 "5"))))

    (cons 'N1492 #(none "F1120S" "Income-rental" 1 #f "" ((1990 "K3a"))))
    (cons 'N1497 #(none "F1120S" "Interest income-CD's/etc." 1 #f "" ((2004 "K4") (1990 "K4a"))))
    (cons 'N1498 #(none "F1120S" "Interest income-U.S. Treas" 1 #f "" ((2004 "K4") (1990 "K4a"))))
    (cons 'N1499 #(none "F1120S" "Dividend income" 1 #f "" ((2004 "K5a") (1990 "K4b"))))
    (cons 'N1501 #(none "F1120S" "Royalty income" 1 #f "" ((2004"K6") (1990 "K4c"))))
    (cons 'N1507 #(none "F1120S" "Other income" 1 #f "" ((2004 "K10") (1990 "K'6"))))
    (cons 'N1527 #(none "F1120S" "Tax-exempt interest income" 1 #f "" ((2004 "K16a") (1992 "K17") (1990 "K18"))))
    (cons 'N1528 #(none "F1120S" "Other tax-exempt income" 1 #f "" ((2004 "K16b") (1992 "K18") (1990 "K18"))))

    (cons 'N1595 #(none "F1120S" "Net income(loss) per books" 1 #f "" ((1990 "M-1,1"))))
    (cons 'N1601 #(none "F1120S" "Income on books not on return" 1 #f "" ((1990 "M-1,5"))))

    (cons 'N418 #(not-impl "F4797" "Form 4797" 1 #f ""))
    (cons 'N420 #(not-impl "F4797" "LT dep. gain - business" 5 #f "" ((1990 "2"))))
    (cons 'N423 #(not-impl "F4797" "LT dep. gain - res. rent." 5 #f "" ((1990 "2"))))
    (cons 'N421 #(not-impl "F4797" "ST dep. prop. - business" 5 #f "" ((1997 "10") (1992 "11") (1990 "10"))))
    (cons 'N424 #(not-impl "F4797" "ST dep. prop. - res. rent." 5 #f "" ((1997 "10") (1992 "11") (1990 "10"))))

    (cons 'N1382 #(parent "F8825" "Gross rents - A" 1 #t "" ((1990 "2A"))))
    (cons 'N1410 #(parent "F8825" "Gross rents - B" 1 #t "" ((1990 "2B"))))
    (cons 'N1437 #(parent "F8825" "Gross rents - C" 1 #t "" ((1990 "2C"))))
    (cons 'N1464 #(parent "F8825" "Gross rents - D" 1 #t "" ((1990 "2D"))))
    (cons 'N1378 #(none "F8825" "Income-other rental real estate" 1 #t "" ((1990 "20a"))))

    (cons 'N446 #(not-impl "Sched K-1" "Schedule K-1 Worksheet" 1 #t ""))
    (cons 'N447 #(not-impl "Sched K-1" "Spouse" 0 #t "" ((1990 ""))))
    (cons 'N457 #(not-impl "Sched K-1" "Partnership or S corp name" 2 #t "" ((2004 "B") (1990 ""))))
    (cons 'N448 #(parent "Sched K-1" "Ordinary income or loss" 1 #t "" ((1990 "1"))))
    (cons 'N449 #(parent "Sched K-1" "Rental RE income or loss" 1 #t "" ((1990 "2"))))
    (cons 'N450 #(parent "Sched K-1" "Other rental income or loss" 1 #t "" ((1990 "3"))))
    (cons 'N451 #(parent "Sched K-1" "Interest income" 1 #t "" ((2004 "4") (1990 "4a"))))
    (cons 'N452 #(parent "Sched K-1" "Dividends" 1 #t "" ((2004 "5a") (2003 "4b(2)") (1990 "4b"))))
    (cons 'N527 #(parent "Sched K-1" "Royalties" 1 #t "" ((2004 "6") (1990 "4c"))))
    (cons 'N453 #(parent "Sched K-1" "Net ST capital gain or loss" 1 #t "" ((2004 "7") (2003 "4d2") (1990 "4d"))))
    (cons 'N454 #(parent "Sched K-1" "Net LT capital gain or loss" 1 #t "" ((2004 "8a") (2003 "4e2") (2001 "4e1") (1997 "4e2") (1990 "4e"))))
    (cons 'N674 #(none "Sched K-1" "28% rate gain(loss)" 1 #t "" ((2004 "8b") (2003 "25") (2001 "4e2") (1997 "4e1"))))
    (cons 'N675 #(none "Sched K-1" "Qualified 5-year gain" 1 #t "" ((2003 "NA - Expired") (2001 "4e3")) 2002))
    (cons 'N456 #(parent "Sched K-1" "Net Section 1231 gain or loss" 1 #t "" ((2004 "9") (2003 "5b") (1998 "5") (1997 "5b") (1990 "5"))))
    (cons 'N676 #(none "Sched K-1" "Other Income (loss)" 1 #t "" ((2004 "10") (1990 "6"))))
    (cons 'N528 #(parent "Sched K-1" "Tax-exempt interest income" 1 #t "" ((1992 "17"))))
   )
  )
  (cons 'Other
   (list
    (cons 'N000 #(none "" "Tax Report Only - No TXF Export" 0 #f ""))
   )
  )
 )
)

(define txf-expense-categories
 (list
  (cons 'F1040
   (list
    (cons 'N000 #(none "" "Tax Report Only - No TXF Export" 0 #f ""))

    (cons 'N256 #(not-impl "F1040" "Form 1040" 1 #f ""))
    (cons 'N680 #(none "F1040" "Educator expenses,self" 1 #f "" ((2007 "23") (2006 "NA - Expired") (2002 "23"))))
    (cons 'N681 #(none "F1040" "Educator expenses,spouse" 1 #f "" ((2007 "23") (2006 "NA - Expired") (2002 "23"))))
    (cons 'N482 #(none "F1040" "IRA contribution, non-work spouse" 1 #f "" ((1997 "23") (1994 "23b") (1990 "24b") (1989 "25") (1987 "25b") (1985 "26") (1983 "26a") (1982 "25") (1981 "24") (1980 "25")) 2000))
    (cons 'N607 #(none "F1040" "Med savings contribution, self" 1 #f "Med savings con" ((2005 "25") (2004 "35") (2003 "33") (2002 "27") (1998 "25") (1997 "24"))))
    (cons 'N608 #(none "F1040" "Med savings contribution, spouse" 1 #f "Med savings con" ((2005 "25") (2004 "35") (2003 "33") (2002 "27") (1998 "25") (1997 "24"))))
    (cons 'N263 #(none "F1040" "Keogh deduction, self" 1 #f "Keogh deduction" ((2005 "28") (2004 "32") (2003 "30") (2002 "31") (1998 "29") (1997 "28") (1988 "27") (1987 "26") (1984 "27") (1982 "26") (1981 "25") (1980 "26"))))
    (cons 'N516 #(none "F1040" "Keogh deduction, spouse" 1 #f "Keogh deduction" ((2005 "28") (2004 "32") (2003 "30") (2002 "31") (1998 "29") (1997 "28") (1988 "27") (1987 "26") (1984 "27") (1982 "26") (1981 "25") (1980 "26"))))
    (cons 'N517 #(none "F1040" "SEP deduction, self" 1 #f "SEP deduction," ((2005 "28") (2004 "32") (2003 "30") (2002 "31") (1998 "29") (1997 "28") (1988 "27") (1987 "26") (1984 "27") (1982 "26") (1981 "25") (1980 "26"))))
    (cons 'N518 #(none "F1040" "SEP deduction, spouse" 1 #f "SEP deduction," ((2005 "28") (2004 "32") (2003 "30") (2002 "31") (1998 "29") (1997 "28") (1988 "27") (1987 "26") (1984 "27") (1982 "26") (1981 "25") (1980 "26"))))
    (cons 'N609 #(none "F1040" "SIMPLE contribution, self" 1 #f "SIMPLE contribu" ((2005 "28") (2004 "32") (2003 "30") (2002 "31") (1998 "29") (1997 "28") (1988 "27") (1987 "26") (1984 "27") (1982 "26") (1981 "25") (1980 "26"))))
    (cons 'N610 #(none "F1040" "SIMPLE contribution, spouse" 1 #f "SIMPLE contribu" ((2005 "28") (2004 "32") (2003 "30") (2002 "31") (1998 "29") (1997 "28") (1988 "27") (1987 "26") (1984 "27") (1982 "26") (1981 "25") (1980 "26"))))
    (cons 'N265 #(current "F1040" "Early withdrawal penalty" 1 #f "Early withdrawa" ((2005 "30") (2004 "33") (2003 "31") (2002 "32") (1998 "30") (1997 "29") (1988 "28") (1987 "27") (1984 "28") (1982 "27") (1981 "26")(1980 "27"))))
    (cons 'N264 #(none "F1040" "Alimony paid" 1 #f "Alimony paid" ((2005 "31a") (2004 "34a") (2003 "32a") (2002 "33a") (1998 "31a") (1997 "30a") (1988 "29") (1987 "28") (1984 "29") (1982 "28") (1981 "27") (1980 "28"))))
    (cons 'N262 #(none "F1040" "IRA contribution, self" 1 #f "IRA Contrib" ((2005 "32") (2004 "25") (2002 "24") (1997 "23") (1994 "23a") (1990 "24a") (1989 "24") (1987 "25a") (1985 "26") (1983 "26a") (1982 "25") (1981 "24") (1980 "25"))))
    (cons 'N481 #(none "F1040" "IRA contribution, spouse" 1 #f "" ((2005 "32") (2004 "25") (2002 "24") (1997 "23") (1994 "23b") (1990 "24b") (1989 "25") (1987 "25b") (1983 "26a") (1982 "25") (1981 "24") (1980 "25"))))
    (cons 'N636 #(none "F1040" "Student loan interest" 1 #f "Student loan in" ((2005 "33") (2004 "26") (2002 "25") (1998 "24"))))
    (cons 'N613 #(none "F1040" "Fed tax withheld, RR retire, self" 1 #f "Tax:Fed wh, RR" ((2014 "64") (2011 "62") (2009 "61") (2008 "62") (2005 "64") (2004 "63") (2003 "61") (2002 "62") (2001 "59") (2000 "58") (1998 "57") (1997 "54") (1996 "52") (1995 "55") (1991 "54") (1990 "55") (1989 "56") (1987 "54") (1986 "56") (1983 "57") (1982 "60") (1980 "55"))))
    (cons 'N614 #(none "F1040" "Fed tax withheld, RR retire, spouse" 1 #f "Tax Spouse:Fed" ((2014 "64") (2011 "62") (2009 "61") (2008 "62") (2005 "64") (2004 "63") (2003 "61") (2002 "62") (2001 "59") (2000 "58") (1998 "57") (1997 "54") (1996 "52") (1995 "55") (1991 "54") (1990 "55") (1989 "56") (1987 "54") (1986 "56") (1983 "57") (1982 "60") (1980 "55"))))
    (cons 'N611 #(none "F1040" "Fed tax withheld, Soc. Sec., self" 1 #f "Tax:Fed wh, Soc" ((2014 "64") (2011 "62") (2009 "61") (2008 "62") (2005 "64") (2004 "63") (2003 "61") (2002 "62") (2001 "59") (2000 "58") (1998 "57") (1997 "54") (1996 "52") (1995 "55") (1991 "54") (1990 "55") (1989 "56") (1987 "54") (1986 "56") (1983 "57") (1982 "60") (1980 "55"))))
    (cons 'N612 #(none "F1040" "Fed tax withheld, Soc. Sec., spouse" 1 #f "Tax Spouse:Fed" ((2014 "64") (2011 "62") (2009 "61") (2008 "62") (2005 "64") (2004 "63") (2003 "61") (2002 "62") (2001 "59") (2000 "58") (1998 "57") (1997 "54") (1996 "52") (1995 "55") (1991 "54") (1990 "55") (1989 "56") (1987 "54") (1986 "56") (1983 "57") (1982 "60") (1980 "55"))))
    (cons 'N615 #(current "F1040" "Fed tax withheld, dividend income" 3 #f "" ((2014 "64") (2011 "62") (2009 "61") (2008 "62") (2005 "64") (2004 "63") (2003 "61") (2002 "62") (2001 "59") (2000 "58") (1998 "57") (1997 "54") (1996 "52") (1995 "55") (1991 "54") (1990 "55") (1989 "56") (1987 "54") (1986 "56") (1983 "57") (1982 "60") (1980 "55"))))
    (cons 'N616 #(current "F1040" "Fed tax withheld, interest income" 3 #f "" ((2014 "64") (2011 "62") (2009 "61") (2008 "62") (2005 "64") (2004 "63") (2003 "61") (2002 "62") (2001 "59") (2000 "58") (1998 "57") (1997 "54") (1996 "52") (1995 "55") (1991 "54") (1990 "55") (1989 "56") (1987 "54") (1986 "56") (1983 "57") (1982 "60") (1980 "55"))))
    (cons 'N521 #(none "F1040" "Federal estimated tax, qrtrly" 6 #f "Tax:Fed Estimat" ((2014 "65") (2011 "63") (2009 "62") (2008 "63") (2005 "65") (2004 "64") (2003 "62") (2002 "63") (2001 "60") (2000 "59") (1998 "58") (1997 "55") (1996 "53") (1995 "56") (1991 "55") (1990 "56") (1989 "57") (1987 "55") (1986 "57") (1983 "58") (1982 "61") (1980 "56"))))
    (cons 'N268 #(none "F1040" "Fed. estimated tax" 1 #f "" ((2014 "65") (2011 "63") (2009 "62") (2008 "63") (2005 "65") (2004 "64") (2003 "62") (2002 "63") (2001 "60") (2000 "59") (1998 "58") (1997 "55") (1996 "53") (1995 "56") (1991 "55") (1990 "56") (1989 "57") (1987 "55") (1986 "57") (1983 "58") (1982 "61") (1980 "56"))))

    (cons 'N270 #(not-impl "Sched A" "Schedule A" 1 #f ""))
    (cons 'N273 #(none "Sched A" "Medicine and drugs" 1 #f "Medical:Medicin" ((1990 "1") (1987 "1a") (1982 "1") (1980 "2"))))
    (cons 'N274 #(none "Sched A" "Medical travel and lodging" 1 #f "" ((1990 "1") (1987 "1b") (1984 "2b") (1983 "4b") (1982 "5b") (1980 "6c"))))
    (cons 'N484 #(none "Sched A" "Doctors, dentists, hospitals" 1 #f "Medical:Doctor" ((1990 "1") (1987 "1a") (1984 "2a") (1983 "4a") (1982 "5a") (1980 "6a&b"))))
    (cons 'N275 #(none "Sched A" "State income taxes" 1 #f "" ((1987 "5") (1984 "6") (1983 "8") (1980 "11"))))
    (cons 'N522 #(parent "Sched A" "State estimated tax, qrtrly" 6 #f "" ((1987 "5") (1984 "6") (1983 "8") (1980 "11"))))
    (cons 'N544 #(none "Sched A" "Local income taxes" 1 #f "" ((1987 "5") (1984 "6") (1983 "8") (1980 "11"))))
    (cons 'N276 #(none "Sched A" "Real estate taxes" 1 #f "Tax:Property" ((1987 "6") (1984 "7") (1983 "9") (1980 "12"))))
    (cons 'N535 #(none "Sched A" "Personal property taxes" 1 #f "" ((2011 "7") (2009 "8") (1987 "7") (1984 "9") (1983 "11") (1980 "14"))))
    (cons 'N277 #(none "Sched A" "Other taxes" 3 #f "" ((1994 "8") (1987 "7") (1984 "9") (1983 "11") (1982 "14") (1980 "15"))))
    (cons 'N283 #(none "Sched A" "Home mortgage interest (1098)" 1 #f "Mortgage Int:Ba" ((1994 "10") (1987 "9a") (1984 "11a") (1983 "13a") (1982 "16a") (1980 "17"))))
    (cons 'N545 #(none "Sched A" "Home mortgage interest (no 1098)" 1 #f "" ((1994 "11") (1987 "9b") (1984 "11b") (1983 "13b") (1982 "16b") (1980 "17"))))
    (cons 'N284 #(none "Sched A" "Points paid (no 1098)" 1 #f "Mortgage Points" ((1994 "12") (1987 "10") (1984 "13") (1983 "15") (1982 "18") (1980 "19"))))
    (cons 'N278 #(none "Sched A" "Credit card interest" 1 #f "" ((1984 "12") (1983 "14") (1982 "17") (1980 "18")) 1986))
    (cons 'N279 #(none "Sched A" "Other personal interest" 1 #f "" ((1987 "12a") (1984 "13") (1983 "15") (1982 "18") (1980 "19")) 1990))
    (cons 'N280 #(none "Sched A" "Cash charity contributions" 1 #f "" ((2007 "16") (1994 "15") (1991 "13") (1987 "14") (1984 "15") (1983 "17") (1982 "20") (1980 "21"))))
    (cons 'N485 #(none "Sched A" "Non-cash charity contributions" 1 #f "" ((2007 "17") (1994 "16") (1991 "14") (1987 "15") (1984 "16") (1983 "18") (1982 "21") (1980 "22"))))
    (cons 'N271 #(none "Sched A" "Subscriptions" 1 #f "" ((2007 "21") (1994 "20") (1991 "19") (1987 "20") (1984 "22") (1983 "24") (1982 "26") (1980 "31"))))
    (cons 'N281 #(none "Sched A" "Tax preparation fees" 1 #f "" ((2007 "22") (1994 "21") (1991 "20") (1987 "21") (1984 "21") (1983 "23") (1982 "25b") (1981 "30b") (1980 "31"))))
    (cons 'N282 #(none "Sched A" "Investment management fees" 1 #f "Investment Inte" ((2007 "23") (1994 "22") (1991 "20") (1987 "21") (1984 "22") (1983 "24") (1982 "26") (1980 "31"))))
    (cons 'N486 #(none "Sched A" "Misc., subject to 2% AGI limit" 1 #f "" ((2007 "23") (1994 "22") (1991 "20") (1987 "21"))))
    (cons 'N272 #(none "Sched A" "Gambling losses" 1 #f "" ((2007 "28") (1995 "27") (1994 "28") (1991 "25") (1990 "26") (1987 "25") (1984 "22") (1983 "24") (1982 "26") (1980 "31"))))
    (cons 'N523 #(none "Sched A" "Misc., no 2% AGI limit" 1 #f "" ((2007 "28") (1995 "27") (1994 "28") (1991 "25") (1990 "26") (1987 "25") (1984 "22") (1983 "24") (1982 "26") (1980 "31"))))

    (cons 'N291 #(not-impl "Sched C" "Schedule C" 1 #t ""))
    (cons 'N296 #(none "Sched C" "Returns and allowances" 1 #t "" ((1989 "2") (1980 "1b"))))
    (cons 'N304 #(none "Sched C" "Advertising" 1 #t "" ((1989 "8") (1980 "6"))))
    (cons 'N306 #(none "Sched C" "Car and truck expenses" 1 #t "" ((2003 "9") (1989 "10") (1982 "9") (1980 "10"))))
    (cons 'N307 #(none "Sched C" "Commissions and fees" 1 #t "" ((2003 "10") (1989 "11") (1982 "10") (1980 "11"))))
    (cons 'N685 #(none "Sched C" "Contract labor" 1 #t "" ((2003 "11"))))
    (cons 'N309 #(none "Sched C" "Depletion" 1 #t "" ((1989 "12") (1982 "11") (1980 "12"))))
    (cons 'N308 #(none "Sched C" "Employee benefit programs" 1 #t "" ((1982 "14") (1982 "15"))))
    (cons 'N310 #(none "Sched C" "Insurance, other than health" 1 #t "" ((1990 "15") (1982 "16") (1980 "17"))))
    (cons 'N311 #(none "Sched C" "Interest expense, mortgage" 1 #t "" ((1990 "16a") (1986 "17a") (1985 "19") (1982 "17") (1980 "18"))))
    (cons 'N312 #(none "Sched C" "Interest expense, other" 1 #t "" ((1990 "16b") (1986 "17b") (1985 "21") (1982 "17") (1980 "18"))))
    (cons 'N298 #(none "Sched C" "Legal and professional fees" 1 #t "" ((1990 "17") (1989 "18") (1986 "19") (1985 "18") (1982 "19") (1980 "20"))))
    (cons 'N313 #(none "Sched C" "Office expenses" 1 #t "" ((1990 "18") (1989 "19") (1982 "20") (1980 "21"))))
    (cons 'N314 #(none "Sched C" "Pension/profit sharing plans" 1 #t "" ((1990 "19") (1989 "20") (1986 "21") (1985 "22") (1982 "21") (1980 "22"))))
    (cons 'N299 #(none "Sched C" "Rent/lease vehicles, equip." 1 #t "" ((1990 "20a") (1989 "21a") (1986 "22") (1985 "23") (1984 "22") (1981 "23") (1980 "24"))))
    (cons 'N300 #(none "Sched C" "Rent/lease other bus. prop." 1 #t "" ((1990 "20b") (1989 "21b") (1986 "22") (1985 "23") (1982 "22") (1981 "23") (1980 "24"))))
    (cons 'N315 #(none "Sched C" "Repairs and maintenance" 1 #t "" ((1990 "21") (1989 "22") (1986 "23") (1985 "24") (1982 "23") (1981 "24") (1980 "25"))))
    (cons 'N301 #(none "Sched C" "Supplies (not from Cost of Goods Sold)" 1 #t "" ((1990 "22") (1989 "23") (1986 "24") (1985 "25") (1982 "24") (1981 "25") (1980 "26"))))
    (cons 'N316 #(none "Sched C" "Taxes and licenses" 1 #t "" ((1990 "23") (1989 "24") (1986 "25") (1985 "26") (1982 "25") (1981 "26") (1980 "27"))))
    (cons 'N317 #(none "Sched C" "Travel" 1 #t "" ((1990 "24a") (1989 "25a") (1987 "26a") (1986 "26") (1985 "27") (1982 "26") (1981 "27") (1980 "29"))))
    (cons 'N294 #(none "Sched C" "Meals and entertainment" 1 #t "" ((1990 "24b") (1989 "25b") (1987 "26b") (1986 "26") (1985 "27") (1982 "26") (1981 "27") (1980 "29"))))
    (cons 'N318 #(none "Sched C" "Utilities" 1 #t "" ((1990 "25") (1989 "26") (1986 "27") (1985 "28") (1982 "27") (1981 "28") (1980 "30"))))
    (cons 'N297 #(none "Sched C" "Wages paid" 1 #t "" ((1990 "26") (1989 "27") (1986 "28a") (1985 "29a") (1982 "28a") (1981 "29a") (1980 "31a"))))
    (cons 'N302 #(current "Sched C" "Other business expenses" 3 #t "" ((2011 "27a") (1993 "27") (1990 "27b") (1989 "28") (1987 "29") (1986 "30") (1985 "31") (1982 "30") (1981 "31") (1980 "32"))))
    (cons 'N305 #(none "Sched C" "Bad debts" 1 #t "" ((2011 "27a") (2003 "27") (1989 "9") (1982 "7") (1980 "8"))))
    (cons 'N493 #(none "Sched C" "Purchases, cost of goods" 1 #t "" ((1996 "36") (1991 "34") (1990 "32") (1989 "33") (1983 "III,2") (1982 "C-1,2") (1980 "C-1,2a"))))
    (cons 'N295 #(none "Sched C" "Cost of goods sold" 1 #t "" ((1990 "37") (1989 "36") (1988 "III,5") (1980 "C-1,5")) 1992))
    (cons 'N494 #(none "Sched C" "Labor, cost of goods sold" 1 #t "" ((1996 "37") (1991 "35") (1990 "33") (1989 "34") (1983 "III,3") (1980 "C-1,3"))))
    (cons 'N495 #(none "Sched C" "Materials, cost of goods" 1 #t "" ((1996 "38") (1991 "36") (1990 "34") (1989 "35") (1983 "III,4") (1980 "C-1,4"))))
    (cons 'N496 #(none "Sched C" "Other costs, cost of goods" 1 #t "" ((1996 "39") (1991 "37") (1990 "35") (1989 "36") (1988 "III,5") (1980 "C-1,5"))))

    (cons 'N497 #(not-impl "Sched C-EZ" "Schedule C-EZ" 1 #t ""))
    (cons 'N500 #(none "Sched C-EZ" "Total expenses" 1 #t "" ((1992 "2"))))

    (cons 'N325 #(not-impl "Sched E" "Schedule E" 1 #t ""))
    (cons 'N328 #(none "Sched E" "Advertising" 1 #t "" ((1990 "5") (1987 "6") (1981 "4") (1980 "VII"))))
    (cons 'N329 #(none "Sched E" "Auto and travel" 1 #t "" ((1990 "6") (1987 "7") (1981 "5") (1980 "VII"))))
    (cons 'N330 #(none "Sched E" "Cleaning and maintenance" 1 #t "" ((1990 "7") (1987 "8") (1981 "6") (1980 "VII"))))
    (cons 'N331 #(none "Sched E" "Commissions" 1 #t "" ((1990 "8") (1987 "9") (1981 "7") (1980 "VII"))))
    (cons 'N332 #(none "Sched E" "Insurance" 1 #t "" ((1990 "9") (1987 "10") (1981 "8") (1980 "VII"))))
    (cons 'N333 #(none "Sched E" "Legal and professional fees" 1 #t "" ((1990 "10") (1987 "11") (1985 "9") (1981 "10") (1980 "VII"))))
    (cons 'N502 #(none "Sched E" "Management fees" 1 #t "" ((1992 "11") (1990 "18")  (1987 "19") (1985 "17") (1981 "16") (1980 "VII"))))
    (cons 'N334 #(none "Sched E" "Mortgage interest expense" 1 #t "" ((1992 "12") (1990 "11") (1987 "12") (1985 "10") (1981 "9") (1980 "VII"))))
    (cons 'N335 #(none "Sched E" "Other interest expense" 1 #t "" ((1992 "13") (1990 "12") (1987 "13") (1985 "11") (1981 "9") (1980 "VII"))))
    (cons 'N336 #(none "Sched E" "Repairs" 1 #t "" ((1992 "14") (1990 "13") (1987 "14") (1985 "12") (1981 "11") (1980 "VII"))))
    (cons 'N337 #(none "Sched E" "Supplies" 1 #t "" ((1992 "15") (1990 "14") (1987 "15") (1985 "13") (1981 "12") (1980 "VII"))))
    (cons 'N338 #(none "Sched E" "Taxes" 1 #t "" ((1992 "16") (1990 "15") (1987 "16") (1985 "14") (1981 "13") (1980 "VII"))))
    (cons 'N339 #(none "Sched E" "Utilities" 1 #t "" ((1992 "17") (1990 "16") (1987 "17") (1985 "15") (1981 "14") (1980 "VII"))))
    (cons 'N340 #(none "Sched E" "Wages paid" 1 #t "" ((2011 "19") (1992 "18") (1990 "17") (1987 "18") (1985 "16") (1981 "15") (1980 "VII")) 1991))
    (cons 'N341 #(current "Sched E" "Other expenses" 3 #t "" ((2011 "19") (1990 "18") (1987 "19") (1985 "17") (1981 "16") (1980 "VII"))))

    (cons 'N343 #(not-impl "Sched F" "Schedule F" 1 #t ""))
    (cons 'N378 #(none "Sched F" "Cost of resale livestock/items" 1 #t "" ((2012 "1b") (2011 "1d") (1990 "2"))))
    (cons 'N354 #(none "Sched F" "Breeding fees" 1 #t "" ((1990 "12")) 1991))
    (cons 'N543 #(none "Sched F" "Car and truck expenses" 1 #t "" ((2011 "10") (1992 "12") (1991 "13") (1990 "34"))))
    (cons 'N366 #(none "Sched F" "Chemicals" 1 #t "" ((2011 "11") (1992 "13") (1991 "14") (1990 "13"))))
    (cons 'N362 #(none "Sched F" "Conservation expenses" 1 #t "" ((2011 "12") (1992 "14") (1991 "15") (1990 "14"))))
    (cons 'N367 #(none "Sched F" "Custom hire expenses" 1 #t "" ((2011 "13") (1992 "15") (1991 "16") (1990 "15"))))
    (cons 'N364 #(none "Sched F" "Employee benefit programs" 1 #t "" ((2011 "15") (1992 "17") (1991 "18") (1990 "17"))))
    (cons 'N350 #(none "Sched F" "Feed purchased" 1 #t "" ((2011 "16") (1992 "18") (1991 "19") (1990 "18"))))
    (cons 'N352 #(none "Sched F" "Fertilizers and lime" 1 #t "" ((2011 "17") (1992 "19") (1991 "20") (1990 "19"))))
    (cons 'N361 #(none "Sched F" "Freight and trucking" 1 #t "" ((2011 "18") (1992 "20") (1991 "21") (1990 "20"))))
    (cons 'N356 #(none "Sched F" "Gasoline, fuel, and oil" 1 #t "" ((2011 "19") (1992 "21") (1991 "22") (1990 "21"))))
    (cons 'N359 #(none "Sched F" "Insurance, other than health" 1 #t "" ((2011 "20") (1992 "22") (1991 "23") (1990 "22"))))
    (cons 'N346 #(none "Sched F" "Interest expense, mortgage" 1 #t "" ((2011 "21a") (1992 "23a") (1991 "24a") (1990 "23a"))))
    (cons 'N347 #(none "Sched F" "Interest expense, other" 1 #t "" ((2011 "21b") (1992 "23b") (1991 "24b") (1990 "23b"))))
    (cons 'N344 #(none "Sched F" "Labor hired" 1 #t "" ((2011 "22") (1992 "24") (1991 "25") (1990 "24"))))
    (cons 'N363 #(none "Sched F" "Pension/profit sharing plans" 1 #t "" ((2011 "23") (1992 "25") (1991 "26") (1990 "25"))))
    (cons 'N349 #(none "Sched F" "Rent/lease vehicles, equip." 1 #t "" ((2011 "24a") (1992 "26a") (1991 "27a") (1990 "26a"))))
    (cons 'N348 #(none "Sched F" "Rent/lease land, animals" 1 #t "" ((2011 "24b") (1992 "26b") (1991 "27b") (1990 "26b"))))
    (cons 'N345 #(none "Sched F" "Repairs and maintenance" 1 #t "" ((2011 "25") (1992 "27") (1991 "28") (1990 "27"))))
    (cons 'N351 #(none "Sched F" "Seeds and plants purchased" 1 #t "" ((2011 "26") (1992 "28") (1991 "29") (1990 "28"))))
    (cons 'N357 #(none "Sched F" "Storage and warehousing" 1 #t "" ((2011 "27") (1992 "29") (1991 "30") (1990 "29"))))
    (cons 'N353 #(none "Sched F" "Supplies purchased" 1 #t "" ((2011 "28") (1992 "30") (1991 "31") (1990 "30"))))
    (cons 'N358 #(none "Sched F" "Taxes" 1 #t "" ((2011 "29") (1992 "31") (1991 "32") (1990 "31"))))
    (cons 'N360 #(none "Sched F" "Utilities" 1 #t "" ((2011 "30") (1992 "32") (1991 "33") (1990 "32"))))
    (cons 'N355 #(none "Sched F" "Vet, breeding, and medicine" 1 #t "" ((2011 "31") (1992 "33") (1991 "34") (1990 "33"))))
    (cons 'N365 #(current "Sched F" "Other farm expenses" 3 #t "" ((2011 "32") (1992 "34") (1991 "35") (1990 "34"))))

    (cons 'N565 #(not-impl "Sched H" "Schedule H" 1 #t ""))
    (cons 'N566 #(not-impl "Sched H" "Spouse" 0 #t "" ((1995 "")) 1995))
    (cons 'N567 #(parent "Sched H" "Cash wages paid" 1 #t "" ((1995 "1"))))
    (cons 'N568 #(parent "Sched H" "Federal tax withheld" 1 #t "" ((2013 "7") (1995 "5"))))

    (cons 'N380 #(not-impl "F2106" "Form 2106" 1 #t ""))
    (cons 'N382 #(none "F2106" "Automobile expenses" 1 #t "" ((1990 "1"))))
    (cons 'N384 #(none "F2106" "Local transportation exp." 1 #t "" ((1990 "2"))))
    (cons 'N383 #(none "F2106" "Travel (away from home)" 1 #t "" ((1990 "3"))))
    (cons 'N381 #(none "F2106" "Education expenses" 1 #t "" ((1990 "4"))))
    (cons 'N385 #(none "F2106" "Other business expenses" 1 #t "" ((1990 "4"))))
    (cons 'N389 #(none "F2106" "Job seeking expenses" 1 #t "" ((1990 "4"))))
    (cons 'N390 #(none "F2106" "Special clothing expenses" 1 #t "" ((1990 "4"))))
    (cons 'N391 #(none "F2106" "Employee home office expenses" 1 #t "" ((1990 "4"))))
    (cons 'N386 #(none "F2106" "Meal/entertainment expenses" 1 #t "" ((1990 "5"))))

    (cons 'N392 #(not-impl "HomeWks" "Home Sale Worksheets, Pub 523" 1 #t ""))
    (cons 'N394 #(none "HomeWks" "Expense of sale" 1 #t "" ((2002 "2"))))
    (cons 'N395 #(none "HomeWks" "Basis of home sold" 1 #t "" ((2002 "4"))))
    (cons 'N396 #(none "HomeWks" "Fixing-up expenses" 1 #t "" ((2000 "8")) 2000))

    (cons 'N400 #(not-impl "F2441" "Form 2441" 1 #f ""))
    (cons 'N401 #(current "F2441" "Child care - day care" 1 #f "Childcare" ((1990 "1"))))
    (cons 'N402 #(current "F2441" "Child care - household" 1 #f "Household, Decu" ((1990 "1"))))

    (cons 'N403 #(not-impl "F3903" "Form 3903" 1 #t ""))
    (cons 'N526 #(not-impl "F3903" "Spouse" 0 #t "" ((2001 ""))))
    (cons 'N406 #(none "F3903" "Trans./store hshld goods" 1 #t "" ((2001 "1"))))
    (cons 'N407 #(none "F3903" "Travel and lodging" 1 #t "" ((2001 "2"))))
    (cons 'N404 #(none "F3903" "Meals during moving" 1 #t "" ((1993 "6")) 1993))
    (cons 'N405 #(none "F3903" "Meals househunting & temp" 1 #t "" ((1993 "11")) 1993))
    (cons 'N408 #(none "F3903" "Pre-moving expense" 1 #t "" ((1993 "9")) 1993))
    (cons 'N409 #(none "F3903" "Temp. living exp." 1 #t "" ((1993 "10")) 1993))
    (cons 'N410 #(none "F3903" "Moving sale cost" 1 #t "" ((1993 "14")) 1993))
    (cons 'N411 #(none "F3903" "Moving purch. cost" 1 #t "" ((1993 "15")) 1993))

    (cons 'N412 #(not-impl "F4684" "Form 4684" 1 #t ""))
    (cons 'N413 #(none "F4684" "Basis of casualty property" 1 #t "" ((1990 "2"))))

    (cons 'N418 #(not-impl "F4797" "Form 4797" 1 #f ""))
    (cons 'N419 #(not-impl "F4797" "Long Term dep. loss - business" 5 #f "" ((1990 "2"))))
    (cons 'N422 #(not-impl "F4797" "Long Term dep. loss - res. rent." 5 #f "" ((1990 "2"))))
    (cons 'N421 #(not-impl "F4797" "Short Term dep. prop. - business" 5 #f "" ((1997 "10") (1992 "11") (1990 "10"))))
    (cons 'N424 #(not-impl "F4797" "Short Term dep. prop. - res. rent." 5 #f "" ((1997 "10") (1992 "11") (1990 "10"))))

    (cons 'N569 #(not-impl "F4835" "Form 4835" 1 #t ""))
    (cons 'N579 #(none "F4835" "Car and truck expenses" 1 #t "" ((1992 "8") (1991 "9"))))
    (cons 'N580 #(none "F4835" "Chemicals" 1 #t "" ((1992 "9") (1991 "10") (1990 "9"))))
    (cons 'N581 #(none "F4835" "Conservation expenses" 1 #t "" ((1992 "10") (1991 "11") (1990 "10"))))
    (cons 'N582 #(none "F4835" "Custom hire (machine work)" 1 #t "" ((1992 "11") (1991 "12") (1990 "11"))))
    (cons 'N583 #(none "F4835" "Employee benefit programs" 1 #t "" ((1992 "13") (1991 "14") (1990 "13"))))
    (cons 'N584 #(none "F4835" "Feed purchased" 1 #t "" ((1992 "14") (1991 "15") (1990 "14"))))
    (cons 'N585 #(none "F4835" "Fertilizers and lime" 1 #t "" ((1992 "15") (1991 "16") (1990 "15"))))
    (cons 'N586 #(none "F4835" "Freight and trucking" 1 #t "" ((1992 "16") (1991 "17") (1990 "16"))))
    (cons 'N587 #(none "F4835" "Gasoline, fuel, and oil" 1 #t "" ((1992 "17") (1991 "18") (1990 "17"))))
    (cons 'N588 #(none "F4835" "Insurance (other than health)" 1 #t "" ((1992 "18") (1991 "19") (1990 "18"))))
    (cons 'N589 #(none "F4835" "Interest expense, mortgage" 1 #t "" ((1992 "19a") (1991 "20a") (1990 "19a"))))
    (cons 'N590 #(none "F4835" "Interest expense, other" 1 #t "" ((1992 "19b") (1991 "20b") (1990 "19b"))))
    (cons 'N591 #(none "F4835" "Labor hired" 1 #t "" ((1992 "20") (1991 "21") (1990 "20"))))
    (cons 'N592 #(none "F4835" "Pension/profit-sharing plans" 1 #t "" ((1992 "21") (1991 "22") (1990 "21"))))
    (cons 'N593 #(none "F4835" "Rent/lease vehicles, equip." 1 #t "" ((1992 "22a") (1991 "23a") (1990 "22a"))))
    (cons 'N594 #(none "F4835" "Rent/lease land, animals" 1 #t "" ((1992 "22b") (1991 "23b") (1990 "22b"))))
    (cons 'N595 #(none "F4835" "Repairs and maintenance" 1 #t "" ((1992 "23") (1991 "24") (1990 "23"))))
    (cons 'N596 #(none "F4835" "Seeds and plants purchased" 1 #t "" ((1992 "24") (1991 "25") (1990 "24"))))
    (cons 'N597 #(none "F4835" "Storage and warehousing" 1 #t "" ((1992 "25") (1991 "26") (1990 "25"))))
    (cons 'N598 #(none "F4835" "Supplies purchased" 1 #t "" ((1992 "26") (1991 "27") (1990 "26"))))
    (cons 'N599 #(none "F4835" "Taxes" 1 #t "" ((1992 "27") (1991 "28") (1990 "27"))))
    (cons 'N600 #(none "F4835" "Utilities" 1 #t "" ((1992 "28") (1991 "29") (1990 "28"))))
    (cons 'N601 #(none "F4835" "Vet, breeding, medicine" 1 #t "" ((1992 "29") (1991 "30") (1990 "29"))))
    (cons 'N602 #(current "F4835" "Other expenses" 3 #t "" ((1992 "30") (1991 "31") (1990 "30"))))

    (cons 'N425 #(not-impl "F4952" "Form 4952" 1 #f ""))
    (cons 'N426 #(none "F4952" "Investment interest expense" 1 #f "_IntExp" ((1990 "1"))))

    (cons 'N427 #(not-impl "F6252" "Form 6252" 1 #t ""))
    (cons 'N430 #(none "F6252" "Basis of property sold" 1 #t "" ((1992 "8") (1990 "4"))))
    (cons 'N432 #(none "F6252" "Expenses of sale" 1 #t "" ((1992 "11") (1990 "7"))))

    (cons 'N437 #(not-impl "F8606" "Form 8606" 1 #t ""))
    (cons 'N439 #(none "F8606" "IRA contribs-nondeductible" 1 #t "" ((1993 "1") (1989 "2") (1988 "4") (1987 "3"))))

    (cons 'N441 #(not-impl "F8815" "Form 8815" 1 #f ""))
    (cons 'N442 #(none "F8815" "Qualified higher education expenses" 1 #f "" ((1990 "2"))))

    (cons 'N536 #(not-impl "F8829" "Form 8829" 1 #t ""))
    (cons 'N537 #(none "F8829" "Deductible mortgage interest" 1 #t "" ((1991 "10b"))))
    (cons 'N538 #(none "F8829" "Real estate taxes" 1 #t "" ((1991 "11b"))))
    (cons 'N539 #(none "F8829" "Insurance" 1 #t "" ((1991 "17b"))))
    (cons 'N540 #(none "F8829" "Repairs and maintenance" 1 #t "" ((2006 "19b") (1991 "18b"))))
    (cons 'N541 #(none "F8829" "Utilities" 1 #t "" ((2006 "20b") (1991 "19b"))))
    (cons 'N542 #(none "F8829" "Other expenses" 1 #t "" ((2006 "21b") (1991 "20b"))))

    (cons 'N617 #(not-impl "F8839" "Form 8839" 1 #f ""))
    (cons 'N618 #(none "F8839" "Adoption fees" 1 #f "" ((1998 "5") (1997 "3"))))
    (cons 'N619 #(none "F8839" "Court costs" 1 #f "" ((1998 "5") (1997 "3"))))
    (cons 'N620 #(none "F8839" "Attorney fees" 1 #f "" ((1998 "5") (1997 "3"))))
    (cons 'N621 #(none "F8839" "Traveling expenses" 1 #f "" ((1998 "5") (1997 "3"))))
    (cons 'N622 #(none "F8839" "Other expenses" 1 #f "" ((1998 "5") (1997 "3"))))

    (cons 'N639 #(not-impl "F8863" "Form 8863" 1 #f ""))
    (cons 'N637 #(none "F8863" "Hope cr. qual. expenses" 1 #f "" ((2012 "1") (1998 "1c"))))
    (cons 'N638 #(none "F8863" "Lifetime cr. qual. expenses" 1 #f "" ((2012 "10") (2005 "3c") (1998 "4c"))))

    (cons 'N446 #(not-impl "Sched K-1" "Schedule K-1 Worksheet" 1 #t ""))
    (cons 'N448 #(parent "Sched K-1" "Ordinary income or loss" 1 #t "" ((1990 "1"))))
    (cons 'N449 #(parent "Sched K-1" "Rental RE income or loss" 1 #t "" ((1990 "2"))))
    (cons 'N450 #(parent "Sched K-1" "Other rental income or loss" 1 #t "" ((1990 "3"))))
    (cons 'N453 #(parent "Sched K-1" "Net ST capital gain or loss" 1 #t "" ((2004 "8") (2003 "4d2") (1990 "4d"))))
    (cons 'N454 #(parent "Sched K-1" "Net LT capital gain or loss" 1 #t "" ((2004 "9a") (2003 "4e2") (2001 "4e1") (1997 "4e2") (1990 "4e"))))
    (cons 'N674 #(none "Sched K-1" "28% rate gain(loss)" 1 #t "" ((2004 "9b") (2003 "25") (2001 "4e2") (1997 "4e1"))))
    (cons 'N456 #(parent "Sched K-1" "Net Section 1231 gain or loss" 1 #t "" ((2004 "10") (2003 "6b") (1998 "6") (1997 "6b") (1990 "6"))))
    (cons 'N676 #(none "Sched K-1" "Other Income (loss)" 1 #t "" ((2004 "11") (1990 "7"))))
    (cons 'N679 #(none "Sched K-1" "Total Foreign Taxes" 1 #t "" ((2004 "16") (2001 "17g") (2000 "17f") (1990 "17e"))))

    (cons 'N458 #(not-impl "W-2" "W-2" 1 #t ""))
    (cons 'N461 #(parent "W-2" "Federal tax withheld, self" 1 #t "Tax:Fed" ((1993 "2") (1990 "9"))))
    (cons 'N507 #(parent "W-2" "Federal tax withheld, spouse" 1 #t "" ((1993 "2") (1990 "9"))))
    (cons 'N462 #(parent "W-2" "Soc. Sec. tax withheld, self" 1 #t "Tax:Soc Sec" ((1993 "4") (1990 "11"))))
    (cons 'N508 #(parent "W-2" "Soc. Sec. tax withheld, spouse" 1 #t "" ((1993 "4") (1990 "11"))))
    (cons 'N480 #(parent "W-2" "Medicare tax withheld, self" 1 #t "Tax:Medicare" ((1993 "6") (1991 "15"))))
    (cons 'N510 #(parent "W-2" "Medicare tax withheld, spouse" 1 #t "" ((1993 "6") (1991 "15"))))
    (cons 'N464 #(parent "W-2" "State tax withheld, self" 1 #t "Tax:State" ((2001 "17") (1993 "18") (1990 "24"))))
    (cons 'N511 #(parent "W-2" "State tax withheld, spouse" 1 #t "" ((2001 "17") (1993 "18") (1990 "24"))))
    (cons 'N463 #(parent "W-2" "Local tax withheld, self" 1 #t "" ((2001 "19") (1993 "21") (1990 "27"))))
    (cons 'N509 #(parent "W-2" "Local tax withheld, spouse" 1 #t "" ((2001 "19") (1993 "21") (1990 "27"))))

    (cons 'N547 #(not-impl "W-2G" "W-2G" 1 #t ""))
    (cons 'N550 #(parent "W-2G" "Federal tax withheld" 1 #t "" ((2013 "4") (1990 "2"))))
    (cons 'N551 #(parent "W-2G" "State tax withheld" 1 #t "" ((2013 "15") (1990 "14"))))

    (cons 'N643 #(not-impl "F1099-DIV" "1099 Div" 1 #f ""))
    (cons 'N648 #(current "F1099-DIV" "Investment Expense, div" 3 #t "" ((1998 "5") (1990 "1e"))))
    (cons 'N649 #(current "F1099-DIV" "Foreign tax, div" 3 #t "" ((1998 "6") (1990 "3"))))
    (cons 'N650 #(not-impl "F1099-DIV" "Foreign country, div" 2 #t "" ((1998 "7") (1990 "4"))))

    (cons 'N634 #(not-impl "F1099-G" "1099-G" 1 #t ""))
    (cons 'N605 #(not-impl "F1099-G" "Unemployment comp repaid" 1 #t "" ((1990 "1a")) 1990))
    (cons 'N606 #(none "F1099-G" "Fed tax withheld, unemployment comp" 1 #t "" ((1990 "4"))))

    (cons 'N640 #(not-impl "F1099-INT" "1099 INT" 1 #f ""))
    (cons 'N653 #(current "F1099-INT" "Intestment expense, int" 3 #t "" ((1999 "5"))))
    (cons 'N641 #(current "F1099-INT" "Foreign tax, int" 3 #t "" ((1999 "6") (1990 "5"))))
    (cons 'N642 #(not-impl "F1099-INT" "Foreign country, int" 2 #t "" ((1999 "7") (1990 "6"))))

    (cons 'N553 #(not-impl "F1099-MISC" "1099-MISC" 1 #t ""))
    (cons 'N558 #(parent "F1099-MISC" "Federal tax withheld" 1 #t "" ((1990 "4"))))
    (cons 'N563 #(parent "F1099-MISC" "State tax withheld" 1 #t "" ((2001 "16") (1990 "11"))))
    (cons 'N654 #(not-impl "F1099-MISC" "State ID" 2 #t "" ((2001 "17") (1990 "12"))))

    (cons 'N657 #(not-impl "F1099-OID" "1099 OID" 1 #f ""))
    (cons 'N659 #(current "F1099-OID" "Early wdrawal pen, OID" 3 #t "" ((1990 "3"))))
    (cons 'N660 #(current "F1099-OID" "Fed tax wheld, OID" 3 #t "" ((1990 "4"))))
    (cons 'N663 #(current "F1099-OID" "Investment Expense, OID" 3 #t "" ((2013 "9") (1999 "7"))))

    (cons 'N473 #(not-impl "F1099-R" "1099R" 1 #t ""))
    (cons 'N665 #(not-impl "F1099-R" "Taxable not determined" 2 #t "" ((1991 "2b"))))
    (cons 'N670 #(not-impl "F1099-R" "Pension Tax. not determined" 2 #t "" ((1991 "2b"))))
    (cons 'N529 #(parent "F1099-R" "Pension federal withholding" 1 #t "" ((1990 "4"))))
    (cons 'N530 #(parent "F1099-R" "Pension state withholding" 1 #t "" ((2011 "12") (1990 "10"))))
    (cons 'N669 #(not-impl "F1099-R" "Pension State ID" 2 #t "" ((2011 "13") (1990 "11"))))
    (cons 'N531 #(parent "F1099-R" "Pension local withholding" 1 #t "" ((2011 "15") (1993 "13") (1990 "12"))))
    (cons 'N532 #(parent "F1099-R" "IRA federal withholding" 1 #t "" ((1990 "4"))))
    (cons 'N533 #(parent "F1099-R" "IRA state withholding" 1 #t "" ((2011 "12") (1990 "10"))))
    (cons 'N664 #(not-impl "F1099-R" "State ID" 2 #t "" ((2011 "13") (1990 "11"))))
    (cons 'N534 #(parent "F1099-R" "IRA local withholding" 1 #t "" ((2011 "15") (1993 "13") (1990 "12"))))
    (cons 'N625 #(parent "F1099-R" "SIMPLE federal withholding" 1 #t "" ((1990 "4"))))
    (cons 'N626 #(parent "F1099-R" "SIMPLE state withholding" 1 #t "" ((2011 "12") (1990 "10"))))
    (cons 'N627 #(parent "F1099-R" "SIMPLE local withholding" 1 #t "" ((2011 "15") (1993 "13") (1990 "12"))))
   )
  )
  (cons 'F1065
   (list
    (cons 'N000 #(none "" "Tax Report Only - No TXF Export" 0 #f ""))

    (cons 'N1664 #(none "F1065" "Other salaries and wages" 1 #f "" ((2009 "9") (1990 "9a"))))
    (cons 'N1665 #(none "F1065" "Guaranteed payments-use of capital" 1 #f "" ((1990 "10"))))
    (cons 'N1666 #(none "F1065" "Guaranteed payments-partners" 1 #f "" ((1990 "10"))))
    (cons 'N1667 #(none "F1065" "Guaranteed pmts-medical insurance premiums" 1 #f "" ((1990 "10"))))
    (cons 'N1668 #(none "F1065" "Repairs and maintenance" 1 #f "" ((2009 "11") (1990 "15"))))
    (cons 'N1670 #(none "F1065" "Bad debts" 1 #f "" ((2009 "12") (1990 "14"))))
    (cons 'N1672 #(none "F1065" "Rents" 1 #f "" ((2009 "13") (1990 "11"))))
    (cons 'N1674 #(none "F1065" "State franchise or inc. tax" 1 #f "" ((2009 "14") (1990 "13"))))
    (cons 'N1676 #(none "F1065" "Local property taxes" 1 #f "" ((2009 "14") (1990 "13"))))
    (cons 'N1678 #(none "F1065" "Payroll taxes" 1 #f "" ((2009 "14") (1990 "13"))))
    (cons 'N1680 #(none "F1065" "Other miscellaneous taxes" 1 #f "" ((2009 "14") (1990 "13"))))
    (cons 'N1681 #(none "F1065" "Licenses" 1 #f "" ((2009 "14") (1990 "13"))))
    (cons 'N1682 #(none "F1065" "Interest expense" 1 #f "" ((2009 "15") (1990 "12"))))
    (cons 'N1684 #(none "F1065" "Depletion" 1 #f "" ((1990 "17"))))
    (cons 'N1686 #(none "F1065" "Retirement plans/etc." 1 #f "" ((1991 "18") (1990 "18a"))))
    (cons 'N1688 #(none "F1065" "Employee benefit programs" 1 #f "" ((1991 "19") (1990 "18b"))))
    (cons 'N1736 #(none "F1065" "Meals and entertainment" 1 #f "" ((1991 "20") (1990 "19"))))
    (cons 'N1740 #(none "F1065" "Other" 1 #f "" ((1991 "20") (1990 "19"))))

    (cons 'N1634 #(none "F1065" "Purchases" 1 #f "" ((2011 "Form 1125-A, 2") (1990 "A2"))))
    (cons 'N1636 #(none "F1065" "Cost of labor" 1 #f "" ((2011 "Form 1125-A, 3") (1990 "A3"))))
    (cons 'N1638 #(none "F1065" "Add'l section 263A costs" 1 #f "" ((2011 "Form 1125-A, 4") (1991 "A4") (1990 "A4a"))))
    (cons 'N1656 #(none "F1065" "Meals and entertainment" 1 #f "" ((2011 "Form 1125-A, 5") (1991 "A5") (1990 "A4b"))))
    (cons 'N1658 #(none "F1065" "Other costs" 1 #f "" ((2011 "Form 1125-A, 5") (1991 "A5") (1990 "A4b"))))

    (cons 'N1805 #(none "F1065" "Expenses-rental" 1 #f "" ((1990 "K3b"))))
    (cons 'N1829 #(none "F1065" "Charitable contributions" 1 #f "" ((2004 "K13a") (1990 "K8"))))
    (cons 'N1848 #(none "F1065" "Int. exp. on investment debts" 1 #f "" ((2005 "K13b") (2004 "K13c") (1997 "K14a") (1990 "K12a"))))
    (cons 'N1856 #(none "F1065" "Section 59(e) expenses" 1 #f "" ((2005 "K13c(2)") (2004 "K13d(2)") (1995 "K18b") (1990 "K18a"))))
    (cons 'N1845 #(none "F1065" "Other deductions" 1 #f "" ((2005 "K13d") (2004 "K13e") (1990 "K11"))))
    (cons 'N1853 #(none "F1065" "Total foreign taxes" 1 #f "" ((2004 "K16l") (2001 "K17g") (2000 "K17f") (1990 "K17e"))))
    (cons 'N1857 #(none "F1065" "Tax-exempt interest income" 1 #f "" ((2004 "K18a") (1990 "K19"))))
    (cons 'N1859 #(none "F1065" "Other tax-exempt income" 1 #f "" ((2004 "K18b") (1992 "K20") (1990 "K19"))))
    (cons 'N1861 #(none "F1065" "Nondeductible expenses" 1 #f "" ((2004 "K18c") (1992 "K21") (1990 "K19"))))
    (cons 'N1831 #(none "F1065" "Deds.-portfolio income (loss)" 1 #f "" ((2005 "K20b") (2004 "K13b") (1990 "K10"))))

    (cons 'N1903 #(none "F1065" "Depreciation per books" 1 #f "" ((1992 "M-1,4a&7a") (1991 "M-1,3a&6a"))))
    (cons 'N1904 #(none "F1065" "Expenses on books not on return" 1 #f "" ((1992 "M-1,4") (1991 "M-1,3"))))

    (cons 'N1920 #(none "F1065" "Other decreases to partners' cap accts" 1 #f "" ((1991 "M-2,7") (1990 "M(e)"))))

    (cons 'N343 #(not-impl "Sched F" "Schedule F" 1 #t ""))
    (cons 'N378 #(none "Sched F" "Cost of resale livestock/items" 1 #t "" ((2012 "1b") (2011 "1d") (1990 "2"))))
    (cons 'N354 #(none "Sched F" "Breeding fees" 1 #t "" ((1990 "12")) 1991))
    (cons 'N543 #(none "Sched F" "Car and truck expenses" 1 #t "" ((2011 "10") (1992 "12") (1991 "13") (1990 "34"))))
    (cons 'N366 #(none "Sched F" "Chemicals" 1 #t "" ((2011 "11") (1992 "13") (1991 "14") (1990 "13"))))
    (cons 'N362 #(none "Sched F" "Conservation expenses" 1 #t "" ((2011 "12") (1992 "14") (1991 "15") (1990 "14"))))
    (cons 'N367 #(none "Sched F" "Custom hire expenses" 1 #t "" ((2011 "13") (1992 "15") (1991 "16") (1990 "15"))))
    (cons 'N364 #(none "Sched F" "Employee benefit programs" 1 #t "" ((2011 "15") (1992 "17") (1991 "18") (1990 "17"))))
    (cons 'N350 #(none "Sched F" "Feed purchased" 1 #t "" ((2011 "16") (1992 "18") (1991 "19") (1990 "18"))))
    (cons 'N352 #(none "Sched F" "Fertilizers and lime" 1 #t "" ((2011 "17") (1992 "19") (1991 "20") (1990 "19"))))
    (cons 'N361 #(none "Sched F" "Freight and trucking" 1 #t "" ((2011 "18") (1992 "20") (1991 "21") (1990 "20"))))
    (cons 'N356 #(none "Sched F" "Gasoline, fuel, and oil" 1 #t "" ((2011 "19") (1992 "21") (1991 "22") (1990 "21"))))
    (cons 'N359 #(none "Sched F" "Insurance, other than health" 1 #t "" ((2011 "20") (1992 "22") (1991 "23") (1990 "22"))))
    (cons 'N346 #(none "Sched F" "Interest expense, mortgage" 1 #t "" ((2011 "21a") (1992 "23a") (1991 "24a") (1990 "23a"))))
    (cons 'N347 #(none "Sched F" "Interest expense, other" 1 #t "" ((2011 "21b") (1992 "23b") (1991 "24b") (1990 "23b"))))
    (cons 'N344 #(none "Sched F" "Labor hired" 1 #t "" ((2011 "22") (1992 "24") (1991 "25") (1990 "24"))))
    (cons 'N363 #(none "Sched F" "Pension/profit sharing plans" 1 #t "" ((2011 "23") (1992 "25") (1991 "26") (1990 "25"))))
    (cons 'N349 #(none "Sched F" "Rent/lease vehicles, equip." 1 #t "" ((2011 "24a") (1992 "26a") (1991 "27a") (1990 "26a"))))
    (cons 'N348 #(none "Sched F" "Rent/lease land, animals" 1 #t "" ((2011 "24b") (1992 "26b") (1991 "27b") (1990 "26b"))))
    (cons 'N345 #(none "Sched F" "Repairs and maintenance" 1 #t "" ((2011 "25") (1992 "27") (1991 "28") (1990 "27"))))
    (cons 'N351 #(none "Sched F" "Seeds and plants purchased" 1 #t "" ((2011 "26") (1992 "28") (1991 "29") (1990 "28"))))
    (cons 'N357 #(none "Sched F" "Storage and warehousing" 1 #t "" ((2011 "27") (1992 "29") (1991 "30") (1990 "29"))))
    (cons 'N353 #(none "Sched F" "Supplies purchased" 1 #t "" ((2011 "28") (1992 "30") (1991 "31") (1990 "30"))))
    (cons 'N358 #(none "Sched F" "Taxes" 1 #t "" ((2011 "29") (1992 "31") (1991 "32") (1990 "31"))))
    (cons 'N360 #(none "Sched F" "Utilities" 1 #t "" ((2011 "30") (1992 "32") (1991 "33") (1990 "32"))))
    (cons 'N355 #(none "Sched F" "Vet, breeding, and medicine" 1 #t "" ((2011 "31") (1992 "33") (1991 "34") (1990 "33"))))
    (cons 'N365 #(current "Sched F" "Other farm expenses" 3 #t "" ((2011 "32") (1992 "34") (1991 "35") (1990 "34"))))

    (cons 'N418 #(not-impl "F4797" "Form 4797" 1 #f ""))
    (cons 'N419 #(not-impl "F4797" "Long Term dep. loss - business" 5 #f "" ((1990 "2"))))
    (cons 'N422 #(not-impl "F4797" "Long Term dep. loss - res. rent." 5 #f "" ((1990 "2"))))
    (cons 'N421 #(not-impl "F4797" "Short Term dep. prop. - business" 5 #f "" ((1997 "10") (1992 "11") (1990 "10"))))
    (cons 'N424 #(not-impl "F4797" "Short Term dep. prop. - res. rent." 5 #f "" ((1997 "10") (1992 "11") (1990 "10"))))

    (cons 'N1745 #(parent "F8825" "Advertising – A" 1 #t "" ((1990 "3A"))))
    (cons 'N1746 #(parent "F8825" "Auto and travel – A" 1 #t "" ((1990 "4A"))))
    (cons 'N1747 #(parent "F8825" "Cleaning and maintenance – A" 1 #t "" ((1990 "5A"))))
    (cons 'N1748 #(parent "F8825" "Commissions – A" 1 #t "" ((1990 "6A"))))
    (cons 'N1749 #(parent "F8825" "Insurance – A" 1 #t "" ((1990 "7A"))))
    (cons 'N1750 #(parent "F8825" "Legal and other prof. fees – A" 1 #t "" ((1990 "8A"))))
    (cons 'N1751 #(parent "F8825" "Interest expense – A" 1 #t "" ((1990 "9A"))))
    (cons 'N1752 #(parent "F8825" "Repairs – A" 1 #t "" ((1990 "10A"))))
    (cons 'N1753 #(parent "F8825" "Taxes – A" 1 #t "" ((1990 "11A"))))
    (cons 'N1754 #(parent "F8825" "Utilities – A" 1 #t "" ((1990 "12A"))))
    (cons 'N1755 #(parent "F8825" "Wages and salaries – A" 1 #t "" ((1990 "13A"))))
    (cons 'N1757 #(parent "F8825" "Other miscellaneous expenses – A" 1 #t "" ((1990 "15A"))))

    (cons 'N1760 #(parent "F8825" "Advertising – B" 1 #t "" ((1990 "3B"))))
    (cons 'N1761 #(parent "F8825" "Auto and travel – B" 1 #t "" ((1990 "4B"))))
    (cons 'N1762 #(parent "F8825" "Cleaning and maintenance – B" 1 #t "" ((1990 "5B"))))
    (cons 'N1763 #(parent "F8825" "Commissions – B" 1 #t "" ((1990 "6B"))))
    (cons 'N1764 #(parent "F8825" "Insurance – B" 1 #t "" ((1990 "7B"))))
    (cons 'N1765 #(parent "F8825" "Legal and other prof. fees – B" 1 #t "" ((1990 "8B"))))
    (cons 'N1766 #(parent "F8825" "Interest expense – B" 1 #t "" ((1990 "9B"))))
    (cons 'N1767 #(parent "F8825" "Repairs – B" 1 #t "" ((1990 "10B"))))
    (cons 'N1768 #(parent "F8825" "Taxes – B" 1 #t "" ((1990 "11B"))))
    (cons 'N1769 #(parent "F8825" "Utilities – B" 1 #t "" ((1990 "12B"))))
    (cons 'N1770 #(parent "F8825" "Wages and salaries – B" 1 #t "" ((1990 "13B"))))
    (cons 'N1771 #(parent "F8825" "Other miscellaneous expenses – B" 1 #t "" ((1990 "15B"))))

    (cons 'N1775 #(parent "F8825" "Advertising – C" 1 #t "" ((1990 "3C"))))
    (cons 'N1776 #(parent "F8825" "Auto and travel – C" 1 #t "" ((1990 "4C"))))
    (cons 'N1777 #(parent "F8825" "Cleaning and maintenance – C" 1 #t "" ((1990 "5C"))))
    (cons 'N1778 #(parent "F8825" "Commissions – C" 1 #t "" ((1990 "6C"))))
    (cons 'N1779 #(parent "F8825" "Insurance – C" 1 #t "" ((1990 "7C"))))
    (cons 'N1780 #(parent "F8825" "Legal and other prof. fees – C" 1 #t "" ((1990 "8C"))))
    (cons 'N1781 #(parent "F8825" "Interest expense – C" 1 #t "" ((1990 "9C"))))
    (cons 'N1782 #(parent "F8825" "Repairs – C" 1 #t "" ((1990 "10C"))))
    (cons 'N1783 #(parent "F8825" "Taxes – C" 1 #t "" ((1990 "11C"))))
    (cons 'N1784 #(parent "F8825" "Utilities – C" 1 #t "" ((1990 "12C"))))
    (cons 'N1785 #(parent "F8825" "Wages and salaries – C" 1 #t "" ((1990 "13C"))))
    (cons 'N1786 #(parent "F8825" "Other miscellaneous expenses – C" 1 #t "" ((1990 "15C"))))

    (cons 'N1790 #(parent "F8825" "Advertising – D" 1 #t "" ((1990 "3D"))))
    (cons 'N1791 #(parent "F8825" "Auto and travel – D" 1 #t "" ((1990 "4D"))))
    (cons 'N1792 #(parent "F8825" "Cleaning and maintenance – D" 1 #t "" ((1990 "5D"))))
    (cons 'N1793 #(parent "F8825" "Commissions – D" 1 #t "" ((1990 "6D"))))
    (cons 'N1794 #(parent "F8825" "Insurance – D" 1 #t "" ((1990 "7D"))))
    (cons 'N1795 #(parent "F8825" "Legal and other prof. fees – D" 1 #t "" ((1990 "8D"))))
    (cons 'N1796 #(parent "F8825" "Interest expense – D" 1 #t "" ((1990 "9D"))))
    (cons 'N1797 #(parent "F8825" "Repairs – D" 1 #t "" ((1990 "10D"))))
    (cons 'N1798 #(parent "F8825" "Taxes – D" 1 #t "" ((1990 "11D"))))
    (cons 'N1799 #(parent "F8825" "Utilities – D" 1 #t "" ((1990 "12D"))))
    (cons 'N1800 #(parent "F8825" "Wages and salaries – D" 1 #t "" ((1990 "13D"))))
    (cons 'N1801 #(parent "F8825" "Other miscellaneous expenses – D" 1 #t "" ((1990 "15D"))))

    (cons 'N446 #(not-impl "Sched K-1" "Schedule K-1 Worksheet" 1 #t ""))
    (cons 'N448 #(parent "Sched K-1" "Ordinary income or loss" 1 #t "" ((1990 "1"))))
    (cons 'N449 #(parent "Sched K-1" "Rental RE income or loss" 1 #t "" ((1990 "2"))))
    (cons 'N450 #(parent "Sched K-1" "Other rental income or loss" 1 #t "" ((1990 "3"))))
    (cons 'N453 #(parent "Sched K-1" "Net ST capital gain or loss" 1 #t "" ((2004 "8") (2003 "4d2") (1990 "4d"))))
    (cons 'N454 #(parent "Sched K-1" "Net LT capital gain or loss" 1 #t "" ((2004 "9a") (2003 "4e2") (2001 "4e1") (1997 "4e2") (1990 "4e"))))
    (cons 'N674 #(none "Sched K-1" "28% rate gain(loss)" 1 #t "" ((2004 "9b") (2003 "25") (2001 "4e2") (1997 "4e1"))))
    (cons 'N456 #(parent "Sched K-1" "Net Section 1231 gain or loss" 1 #t "" ((2004 "10") (2003 "6b") (1998 "6") (1997 "6b") (1990 "6"))))
    (cons 'N676 #(none "Sched K-1" "Other Income (loss)" 1 #t "" ((2004 "11") (1990 "7"))))
    (cons 'N679 #(none "Sched K-1" "Total Foreign Taxes" 1 #t "" ((2004 "16l") (2001 "17g") (2000 "17f") (1990 "17e"))))
   )
  )
  (cons 'F1120
   (list
    (cons 'N000 #(none "" "Tax Report Only - No TXF Export" 0 #f ""))

    (cons 'N1053 #(none "F1120" "Ded.-div. paid on pref. stk." 1 #f "" ((1990 "C18"))))

    (cons 'N1059 #(none "F1120" "Purchases" 1 #f "" ((2011 "Form 1125-A, 2") (1990 "A2"))))
    (cons 'N1061 #(none "F1120" "Cost of labor" 1 #f "" ((2011 "Form 1125-A, 3") (1990 "A3"))))
    (cons 'N1062 #(none "F1120" "Section 263A depreciation" 1 #f "" ((2011 "Form 1125-A, 4") (1992 "A5") (1990 "A4b"))))
    (cons 'N1063 #(none "F1120" "Add'l section 263A costs" 1 #f "" ((2011 "Form 1125-A, 4") (1992 "A4") (1990 "A4a"))))
    (cons 'N1075 #(none "F1120" "Other costs" 1 #f "" ((2011 "Form 1125-A, 5") (1992 "A5") (1990 "A4b"))))

    (cons 'N1080 #(none "F1120" "Comp. of Officers -Sch E" 1 #f "" ((2011 "Form 1125-E, 1") (1990 "E1"))))
    (cons 'N1085 #(none "F1120" "Salaries and wages" 1 #f "" ((1994 "13") (1990 "13a"))))
    (cons 'N1086 #(none "F1120" "Repairs and maintenance" 1 #f "" ((1990 "14"))))
    (cons 'N1087 #(none "F1120" "Bad debts" 1 #f "" ((1990 "15"))))
    (cons 'N1089 #(none "F1120" "Rents" 1 #f "" ((1990 "16"))))
    (cons 'N1091 #(none "F1120" "State taxes" 1 #f "" ((1990 "17"))))
    (cons 'N1093 #(none "F1120" "Local property taxes" 1 #f "" ((1990 "17"))))
    (cons 'N1095 #(none "F1120" "Payroll taxes" 1 #f "" ((1990 "17"))))
    (cons 'N1097 #(none "F1120" "Other miscellaneous taxes" 1 #f "" ((1990 "17"))))
    (cons 'N1098 #(none "F1120" "Licenses" 1 #f "" ((1990 "17"))))
    (cons 'N1099 #(none "F1120" "Interest expense" 1 #f "" ((1990 "18"))))
    (cons 'N1101 #(none "F1120" "Charitable contributions" 1 #f "" ((1990 "19"))))
    (cons 'N1103 #(none "F1120" "Depletion" 1 #f "" ((2005 "21") (1990 "22"))))
    (cons 'N1105 #(none "F1120" "Advertising" 1 #f "" ((2005 "22") (1990 "23"))))
    (cons 'N1107 #(none "F1120" "Pension/profit-sharing/etc." 1 #f "" ((2005 "23") (1990 "24"))))
    (cons 'N1109 #(none "F1120" "Employee benefit programs" 1 #f "" ((2005 "24") (1990 "25"))))
    (cons 'N1153 #(none "F1120" "Meals and entertainment" 1 #f "" ((1990 "26"))))
    (cons 'N1157 #(none "F1120" "Other deductions" 1 #f "" ((1990 "26"))))

    (cons 'N1161 #(none "F1120" "Estimated Qtrly Payment #1" 1 #f "" ((2011 "J-II, 13") (1990 "32b"))))
    (cons 'N1163 #(none "F1120" "Estimated Qtrly Payment #2" 1 #f "" ((2011 "J-II, 13") (1990 "32b"))))
    (cons 'N1165 #(none "F1120" "Estimated Qtrly Payment #3" 1 #f "" ((2011 "J-II, 13") (1990 "32b"))))
    (cons 'N1167 #(none "F1120" "Estimated Qtrly Payment #4" 1 #f "" ((2011 "J-II, 13") (1990 "32b"))))

    (cons 'N1234 #(none "F1120" "Depreciation per books" 1 #f "" ((1990 "M-1,5a&8a"))))
    (cons 'N1235 #(none "F1120" "Expenses on books not on return" 1 #f "" ((1990 "M-1,5"))))

    (cons 'N1240 #(none "F1120" "Cash distributions" 1 #f "" ((1990 "M-2,5a"))))
    (cons 'N1241 #(none "F1120" "Stock distributions" 1 #f "" ((1990 "M-2,5b"))))
    (cons 'N1242 #(none "F1120" "Property distributions" 1 #f "" ((1990 "M-2,5c"))))
    (cons 'N1243 #(none "F1120" "Other decreases" 1 #f "" ((1990 "M-2,6"))))

    (cons 'N418 #(not-impl "F4797" "Form 4797" 1 #f ""))
    (cons 'N419 #(not-impl "F4797" "Long Term dep. loss - business" 5 #f "" ((1990 "2"))))
    (cons 'N422 #(not-impl "F4797" "Long Term dep. loss - res. rent." 5 #f "" ((1990 "2"))))
    (cons 'N421 #(not-impl "F4797" "Short Term dep. prop. - business" 5 #f "" ((1997 "10") (1992 "11") (1990 "10"))))
    (cons 'N424 #(not-impl "F4797" "Short Term dep. prop. - res. rent." 5 #f "" ((1997 "10") (1992 "11") (1990 "10"))))
   )
  )
  (cons 'F1120S
   (list
    (cons 'N000 #(none "" "Tax Report Only - No TXF Export" 0 #f ""))

    (cons 'N1301 #(none "F1120S" "Compensation of shareholder/officers" 1 #f "" ((1990 "7"))))
    (cons 'N1302 #(none "F1120S" "Compensation of other officers" 1 #f "" ((1990 "7"))))
    (cons 'N1303 #(none "F1120S" "Salaries and wages" 1 #f "" ((2009 "8") (1990 "8a"))))
    (cons 'N1305 #(none "F1120S" "Repairs and maintenance" 1 #f "" ((1990 "9"))))
    (cons 'N1307 #(none "F1120S" "Bad debts" 1 #f "" ((1990 "10"))))
    (cons 'N1309 #(none "F1120S" "Rents" 1 #f "" ((1990 "11"))))
    (cons 'N1311 #(none "F1120S" "State franchise or inc. tax" 1 #f "" ((1990 "12"))))
    (cons 'N1313 #(none "F1120S" "Local property taxes" 1 #f "" ((1990 "12"))))
    (cons 'N1315 #(none "F1120S" "Payroll taxes" 1 #f "" ((1990 "12"))))
    (cons 'N1317 #(none "F1120S" "Other miscellaneous taxes" 1 #f "" ((1990 "12"))))
    (cons 'N1316 #(none "F1120S" "Licenses" 1 #f "" ((1990 "12"))))
    (cons 'N1318 #(none "F1120S" "Interest expense" 1 #f "" ((1990 "13"))))
    (cons 'N1319 #(none "F1120S" "Depletion" 1 #f "" ((1990 "15"))))
    (cons 'N1321 #(none "F1120S" "Advertising" 1 #f "" ((1990 "16"))))
    (cons 'N1323 #(none "F1120S" "Pension/profit-sharing/etc." 1 #f "" ((1990 "17"))))
    (cons 'N1325 #(none "F1120S" "Employee benefit programs" 1 #f "" ((1990 "18"))))
    (cons 'N1369 #(none "F1120S" "Meals and entertainment" 1 #f "" ((1990 "19"))))
    (cons 'N1373 #(none "F1120S" "Other deductions" 1 #f "" ((1990 "19"))))

    (cons 'N1275 #(none "F1120S" "Purchases" 1 #f "" ((2011 "Form 1125-A, 2") (1990 "A2"))))
    (cons 'N1277 #(none "F1120S" "Cost of labor" 1 #f "" ((2011 "Form 1125-A, 3") (1990 "A3"))))
    (cons 'N1279 #(none "F1120S" "Add'l section 263A costs" 1 #f "" ((2011 "Form 1125-A, 4") (1992 "A4") (1990 "A4a"))))
    (cons 'N1295 #(none "F1120S" "Other costs" 1 #f "" ((2011 "Form 1125-A, 5") (1992 "A5") (1990 "A4b"))))

    (cons 'N1494 #(none "F1120S" "Expenses-rental" 1 #f "" ((1990 "K3b"))))
    (cons 'N1510 #(none "F1120S" "Charitable contributions" 1 #f "" ((2004 "K12a") (1990 "K7"))))
    (cons 'N1512 #(none "F1120S" "Deds.-portfolio income (loss)" 1 #f "" ((2009 "K12d") (2004 "K12b") (1990 "K9"))))
    (cons 'N1519 #(none "F1120S" "Interest exp.-investment debts" 1 #f "" ((2005 "K12b") (2004 "K12c") (1990 "K11a"))))
    (cons 'N1526 #(none "F1120S" "Sec. 59(e) expenses" 1 #f "" ((2005 "K12c") (2004 "K12d") (1992 "K16") (1990 "K16a"))))
    (cons 'N1516 #(none "F1120S" "Other deductions" 1 #f "" ((2009 "K12d") (2004 "K12e") (1990 "K10"))))
    (cons 'N1523 #(none "F1120S" "Total foreign taxes" 1 #f "" ((2009 "K14l") (2004 "K14m") (2003 "K14g") (2000 "K15f") (1990 "K15e"))))
    (cons 'N1513 #(none "F1120S" "Reduction in available taxes" 1 #f "" ((2009 "K14m") (2004 "K14n") (2003 "K14h") (2000 "K15g") (1990 "K15f"))))
    (cons 'N1529 #(none "F1120S" "Nondeductible expenses" 1 #f "" ((2004 "K16c") (1992 "K19") (1990 "K18"))))
    (cons 'N1530 #(none "F1120S" "Prop. distribs.(incl.cash)" 1 #f "" ((2004 "K16d") (1992 "K20") (1990 "K17"))))

    (cons 'N1596 #(none "F1120S" "Depreciation per books" 1 #f "" ((1990 "M-1,3a&6a"))))
    (cons 'N1597 #(none "F1120S" "Expenses on books not on return" 1 #f "" ((1990 "M-1,3"))))

    (cons 'N418 #(not-impl "F4797" "Form 4797" 1 #f ""))
    (cons 'N419 #(not-impl "F4797" "Long Term dep. loss - business" 5 #f "" ((1990 "2"))))
    (cons 'N422 #(not-impl "F4797" "Long Term dep. loss - res. rent." 5 #f "" ((1990 "2"))))
    (cons 'N421 #(not-impl "F4797" "Short Term dep. prop. - business" 5 #f "" ((1997 "10") (1992 "11") (1990 "10"))))
    (cons 'N424 #(not-impl "F4797" "Short Term dep. prop. - res. rent." 5 #f "" ((1997 "10") (1992 "11") (1990 "10"))))

    (cons 'N1384 #(parent "F8825" "Advertising – A" 1 #t "" ((1990 "3A"))))
    (cons 'N1386 #(parent "F8825" "Auto and travel – A" 1 #t "" ((1990 "4A"))))
    (cons 'N1388 #(parent "F8825" "Cleaning and maintenance – A" 1 #t "" ((1990 "5A"))))
    (cons 'N1390 #(parent "F8825" "Commissions – A" 1 #t "" ((1990 "6A"))))
    (cons 'N1392 #(parent "F8825" "Insurance – A" 1 #t "" ((1990 "7A"))))
    (cons 'N1394 #(parent "F8825" "Legal and other prof. fees – A" 1 #t "" ((1990 "8A"))))
    (cons 'N1396 #(parent "F8825" "Interest expense – A" 1 #t "" ((1990 "9A"))))
    (cons 'N1398 #(parent "F8825" "Repairs – A" 1 #t "" ((1990 "10A"))))
    (cons 'N1400 #(parent "F8825" "Taxes – A" 1 #t "" ((1990 "11A"))))
    (cons 'N1402 #(parent "F8825" "Utilities – A" 1 #t "" ((1990 "12A"))))
    (cons 'N1404 #(parent "F8825" "Wages and salaries – A" 1 #t "" ((1990 "13A"))))
    (cons 'N1406 #(parent "F8825" "Other miscellaneous expenses – A" 1 #t "" ((1990 "15A"))))

    (cons 'N1412 #(parent "F8825" "Advertising – B" 1 #t "" ((1990 "3B"))))
    (cons 'N1414 #(parent "F8825" "Auto and travel – B" 1 #t "" ((1990 "4B"))))
    (cons 'N1416 #(parent "F8825" "Cleaning and maintenance – B" 1 #t "" ((1990 "5B"))))
    (cons 'N1418 #(parent "F8825" "Commissions – B" 1 #t "" ((1990 "6B"))))
    (cons 'N1420 #(parent "F8825" "Insurance – B" 1 #t "" ((1990 "7B"))))
    (cons 'N1422 #(parent "F8825" "Legal and other prof. fees – B" 1 #t "" ((1990 "8B"))))
    (cons 'N1424 #(parent "F8825" "Interest expense – B" 1 #t "" ((1990 "9B"))))
    (cons 'N1426 #(parent "F8825" "Repairs – B" 1 #t "" ((1990 "10B"))))
    (cons 'N1428 #(parent "F8825" "Taxes – B" 1 #t "" ((1990 "11B"))))
    (cons 'N1430 #(parent "F8825" "Utilities – B" 1 #t "" ((1990 "12B"))))
    (cons 'N1432 #(parent "F8825" "Wages and salaries – B" 1 #t "" ((1990 "13B"))))
    (cons 'N1434 #(parent "F8825" "Other miscellaneous expenses – B" 1 #t "" ((1990 "15B"))))

    (cons 'N1439 #(parent "F8825" "Advertising – C" 1 #t "" ((1990 "3C"))))
    (cons 'N1441 #(parent "F8825" "Auto and travel – C" 1 #t "" ((1990 "4C"))))
    (cons 'N1443 #(parent "F8825" "Cleaning and maintenance – C" 1 #t "" ((1990 "5C"))))
    (cons 'N1445 #(parent "F8825" "Commissions – C" 1 #t "" ((1990 "6C"))))
    (cons 'N1447 #(parent "F8825" "Insurance – C" 1 #t "" ((1990 "7C"))))
    (cons 'N1449 #(parent "F8825" "Legal and other prof. fees – C" 1 #t "" ((1990 "8C"))))
    (cons 'N1451 #(parent "F8825" "Interest expense – C" 1 #t "" ((1990 "9C"))))
    (cons 'N1453 #(parent "F8825" "Repairs – C" 1 #t "" ((1990 "10C"))))
    (cons 'N1455 #(parent "F8825" "Taxes – C" 1 #t "" ((1990 "11C"))))
    (cons 'N1457 #(parent "F8825" "Utilities – C" 1 #t "" ((1990 "12C"))))
    (cons 'N1459 #(parent "F8825" "Wages and salaries – C" 1 #t "" ((1990 "13C"))))
    (cons 'N1461 #(parent "F8825" "Other miscellaneous expenses – C" 1 #t "" ((1990 "15C"))))

    (cons 'N1466 #(parent "F8825" "Advertising – D" 1 #t "" ((1990 "3D"))))
    (cons 'N1468 #(parent "F8825" "Auto and travel – D" 1 #t "" ((1990 "4D"))))
    (cons 'N1470 #(parent "F8825" "Cleaning and maintenance – D" 1 #t "" ((1990 "5D"))))
    (cons 'N1472 #(parent "F8825" "Commissions – D" 1 #t "" ((1990 "6D"))))
    (cons 'N1474 #(parent "F8825" "Insurance – D" 1 #t "" ((1990 "7D"))))
    (cons 'N1476 #(parent "F8825" "Legal and other prof. fees – D" 1 #t "" ((1990 "8D"))))
    (cons 'N1478 #(parent "F8825" "Interest expense – D" 1 #t "" ((1990 "9D"))))
    (cons 'N1480 #(parent "F8825" "Repairs – D" 1 #t "" ((1990 "10D"))))
    (cons 'N1482 #(parent "F8825" "Taxes – D" 1 #t "" ((1990 "11D"))))
    (cons 'N1484 #(parent "F8825" "Utilities – D" 1 #t "" ((1990 "12D"))))
    (cons 'N1486 #(parent "F8825" "Wages and salaries – D" 1 #t "" ((1990 "13D"))))
    (cons 'N1488 #(parent "F8825" "Other miscellaneous expenses – D" 1 #t "" ((1990 "15D"))))

    (cons 'N446 #(not-impl "Sched K-1" "Schedule K-1 Worksheet" 1 #t ""))
    (cons 'N448 #(parent "Sched K-1" "Ordinary income or loss" 1 #t "" ((1990 "1"))))
    (cons 'N449 #(parent "Sched K-1" "Rental RE income or loss" 1 #t "" ((1990 "2"))))
    (cons 'N450 #(parent "Sched K-1" "Other rental income or loss" 1 #t "" ((1990 "3"))))
    (cons 'N453 #(parent "Sched K-1" "Net ST capital gain or loss" 1 #t "" ((2004 "7") (2003 "4d2") (1990 "4d"))))
    (cons 'N454 #(parent "Sched K-1" "Net LT capital gain or loss" 1 #t "" ((2004 "8a") (2003 "4e2") (2001 "4e1") (1997 "4e2") (1990 "4e"))))
    (cons 'N674 #(none "Sched K-1" "28% rate gain(loss)" 1 #t "" ((2004 "8b") (2003 "25") (2001 "4e2") (1997 "4e1"))))
    (cons 'N456 #(parent "Sched K-1" "Net Section 1231 gain or loss" 1 #t "" ((2004 "9") (2003 "6b") (1998 "6") (1997 "6b") (1990 "6"))))
    (cons 'N676 #(none "Sched K-1" "Other Income (loss)" 1 #t "" ((2004 "10") (1990 "7"))))
    (cons 'N679 #(none "Sched K-1" "Total Foreign Taxes" 1 #t "" ((2004 "14") (2001 "17g") (2000 "17f") (1990 "17e"))))
   )
  )
  (cons 'Other
   (list
    (cons 'N000 #(none "" "Tax Report Only - No TXF Export" 0 #f ""))
   )
  )
 )
)

(define txf-asset-categories
 (list
  (cons 'F1040
   (list
    (cons 'N000 #(none "" "Tax Report Only - No TXF Export" 0 #f ""))

    (cons 'N437 #(not-impl "F8606" "Form 8606" 1 #t ""))
    (cons 'N440 #(none "F8606" "IRA basis at beg of year" 1 #t "" ((1993 "2") (1988 "3"))))
    (cons 'N438 #(none "F8606" "IRAs value at end of year" 1 #t "" ((1993 "6") (1989 "1") (1988 "11") (1987 "8"))))

    (cons 'N392 #(not-impl "HomeWks" "Home Sale Worksheets" 1 #t ""))
    (cons 'N397 #(none "HomeWks" "Cost of new home" 1 #t "" ((2000 "11b")) 2000))
   )
  )
  (cons 'F1065
   (list
    (cons 'N000 #(none "" "Tax Report Only - No TXF Export" 0 #f ""))

    (cons 'N1864 #(none "F1065" "Cash" 1 #f "" ((1990 "L1"))))
    (cons 'N1865 #(none "F1065" "Accts. Rec. and trade notes" 1 #f "" ((1990 "L2a"))))
    (cons 'N1866 #(none "F1065" "Allowance for bad debts" 1 #f "" ((1990 "L2b"))))
    (cons 'N1868 #(none "F1065" "U.S. government obligations" 1 #f "" ((1990 "L4"))))
    (cons 'N1869 #(none "F1065" "Tax-exempt securities" 1 #f "" ((1990 "L5"))))
    (cons 'N1870 #(none "F1065" "Other current assets" 1 #f "" ((1990 "L6"))))
    (cons 'N1871 #(none "F1065" "Mortgage/real estate loans" 1 #f "" ((2011 "L7b")(1990 "L7"))))
    (cons 'N1872 #(none "F1065" "Other investments" 1 #f "" ((1990 "L8"))))
    (cons 'N1873 #(none "F1065" "Buildings/oth. depr. assets" 1 #f "" ((1990 "L9a"))))
    (cons 'N1874 #(none "F1065" "Accumulated depreciation" 1 #f "" ((1990 "L9b"))))
    (cons 'N1875 #(none "F1065" "Depletable assets" 1 #f "" ((1990 "L10a"))))
    (cons 'N1876 #(none "F1065" "Accumulated depletion" 1 #f "" ((1990 "L10b"))))
    (cons 'N1877 #(none "F1065" "Land" 1 #f "" ((1990 "L11"))))
    (cons 'N1878 #(none "F1065" "Intangible assets" 1 #f "" ((1990 "L12a"))))
    (cons 'N1879 #(none "F1065" "Accumulated amortization" 1 #f "" ((1990 "L12b"))))
    (cons 'N1880 #(none "F1065" "Other assets" 1 #f "" ((1990 "L13"))))
   )
  )
  (cons 'F1120
   (list
    (cons 'N000 #(none "" "Tax Report Only - No TXF Export" 0 #f ""))

    (cons 'N1172 #(none "F1120" "Cash" 1 #f "" ((1990 "L1"))))
    (cons 'N1174 #(none "F1120" "Accts. Rec. and trade notes." 1 #f "" ((1990 "L2a"))))
    (cons 'N1176 #(none "F1120" "Allowance for bad debts" 1 #f "" ((1990 "L2b"))))
    (cons 'N1180 #(none "F1120" "U.S. government obligations" 1 #f "" ((1990 "L4"))))
    (cons 'N1182 #(none "F1120" "Tax-exempt securities" 1 #f "" ((1990 "L5"))))
    (cons 'N1184 #(none "F1120" "Other current assets" 1 #f "" ((1990 "L6"))))
    (cons 'N1186 #(none "F1120" "Loans to stockholders" 1 #f "" ((1990 "L7"))))
    (cons 'N1188 #(none "F1120" "Mortgage/real estate loans" 1 #f "" ((1990 "L8"))))
    (cons 'N1190 #(none "F1120" "Other investments" 1 #f "" ((1990 "L9"))))
    (cons 'N1192 #(none "F1120" "Buildings/oth. depr. assets" 1 #f "" ((1990 "L10a"))))
    (cons 'N1194 #(none "F1120" "Accumulated depreciation" 1 #f "" ((1990 "L10b"))))
    (cons 'N1196 #(none "F1120" "Depletable assets" 1 #f "" ((1990 "L11a"))))
    (cons 'N1198 #(none "F1120" "Accumulated depletion" 1 #f "" ((1990 "L11b"))))
    (cons 'N1200 #(none "F1120" "Land" 1 #f "" ((1990 "L12"))))
    (cons 'N1202 #(none "F1120" "Intangible assets" 1 #f "" ((1990 "L13a"))))
    (cons 'N1204 #(none "F1120" "Accumulated amortization" 1 #f "" ((1990 "L13b"))))
    (cons 'N1206 #(none "F1120" "Other assets" 1 #f "" ((1990 "L14"))))
   )
  )
  (cons 'F1120S
   (list
    (cons 'N000 #(none "" "Tax Report Only - No TXF Export" 0 #f ""))

    (cons 'N1535 #(none "F1120S" "Cash" 1 #f "" ((1990 "L1"))))
    (cons 'N1537 #(none "F1120S" "Accts. Rec. and trade notes" 1 #f "" ((1990 "L2a"))))
    (cons 'N1539 #(none "F1120S" "Allowance for bad debts" 1 #f "" ((1990 "L2b"))))
    (cons 'N1543 #(none "F1120S" "U.S. government obligations" 1 #f "" ((1990 "L4"))))
    (cons 'N1545 #(none "F1120S" "Tax-exempt securities" 1 #f "" ((1990 "L5"))))
    (cons 'N1547 #(none "F1120S" "Other current assets" 1 #f "" ((1990 "L6"))))
    (cons 'N1549 #(none "F1120S" "Loans to shareholders" 1 #f "" ((1990 "L7"))))
    (cons 'N1551 #(none "F1120S" "Mortgage/real estate loans" 1 #f "" ((1990 "L8"))))
    (cons 'N1553 #(none "F1120S" "Other investments" 1 #f "" ((1990 "L9"))))
    (cons 'N1555 #(none "F1120S" "Buildings/oth. depr. assets" 1 #f "" ((1990 "L10a"))))
    (cons 'N1557 #(none "F1120S" "Accumulated depreciation" 1 #f "" ((1990 "L10b"))))
    (cons 'N1559 #(none "F1120S" "Depletable assets" 1 #f "" ((1990 "L11a"))))
    (cons 'N1561 #(none "F1120S" "Accumulated depletion" 1 #f "" ((1990 "L11b"))))
    (cons 'N1563 #(none "F1120S" "Land" 1 #f "" ((1990 "L12"))))
    (cons 'N1565 #(none "F1120S" "Intangible assets" 1 #f "" ((1990 "L13a"))))
    (cons 'N1567 #(none "F1120S" "Accumulated amortization" 1 #f "" ((1990 "L13b"))))
    (cons 'N1569 #(none "F1120S" "Other assets" 1 #f "" ((1990 "L14"))))
   )
  )
  (cons 'Other
   (list
    (cons 'N000 #(none "" "Tax Report Only - No TXF Export" 0 #f ""))
   )
  )
 )
)

(define txf-liab-eq-categories
 (list
  (cons 'F1040
   (list
    (cons 'N000 #(none "" "Tax Report Only - No TXF Export" 0 #f ""))
   )
  )
  (cons 'F1065
   (list
    (cons 'N000 #(none "" "Tax Report Only - No TXF Export" 0 #f ""))

    (cons 'N1884 #(none "F1065" "Accounts payable" 1 #f "" ((1990 "L15"))))
    (cons 'N1886 #(none "F1065" "S-T Mortgage/note/bonds pay." 1 #f "" ((1990 "L16"))))
    (cons 'N1888 #(none "F1065" "Other current liabilities" 1 #f "" ((1990 "L17"))))
    (cons 'N1890 #(none "F1065" "All nonrecourse loans" 1 #f "" ((1990 "L18"))))
    (cons 'N1892 #(none "F1065" "L-T Mortgage/note/bonds pay." 1 #f "" ((2011 "L19b") (1990 "L19"))))
    (cons 'N1894 #(none "F1065" "Other liabilities" 1 #f "" ((1990 "L20"))))
   )
  )
  (cons 'F1120
   (list
    (cons 'N000 #(none "" "Tax Report Only - No TXF Export" 0 #f ""))

    (cons 'N1209 #(none "F1120" "Accounts payable" 1 #f "" ((1990 "L16"))))
    (cons 'N1211 #(none "F1120" "S-T Mortgage/note/bonds pay." 1 #f "" ((1990 "L17"))))
    (cons 'N1213 #(none "F1120" "Other current liabilities" 1 #f "" ((1990 "L18"))))
    (cons 'N1215 #(none "F1120" "Loans from stockholders" 1 #f "" ((1990 "L19"))))
    (cons 'N1217 #(none "F1120" "L-T Mortgage/note/bonds pay." 1 #f "" ((1990 "L20"))))
    (cons 'N1219 #(none "F1120" "Other liabilities" 1 #f "" ((1990 "L21"))))
    (cons 'N1221 #(none "F1120" "Capital Stock - Preferred Stk." 1 #f "" ((1990 "L22a"))))
    (cons 'N1223 #(none "F1120" "Capital Stock - Common Stk." 1 #f "" ((1990 "L22b"))))
    (cons 'N1225 #(none "F1120" "Paid-in or capital surplus" 1 #f "" ((1990 "L23"))))
    (cons 'N1231 #(none "F1120" "Cost of Treasury stock" 1 #f "" ((1997 "L27") (1990 "L26"))))
   )
  )
  (cons 'F1120S
   (list
    (cons 'N000 #(none "" "Tax Report Only - No TXF Export" 0 #f ""))

    (cons 'N1573 #(none "F1120S" "Accounts payable" 1 #f "" ((1990 "L16"))))
    (cons 'N1575 #(none "F1120S" "S-T Mortgage/note/bonds pay." 1 #f "" ((1990 "L17"))))
    (cons 'N1577 #(none "F1120S" "Other current liabilities" 1 #f "" ((1990 "L18"))))
    (cons 'N1579 #(none "F1120S" "Loans from shareholders" 1 #f "" ((1990 "L19"))))
    (cons 'N1581 #(none "F1120S" "L-T Mortgage/note/bonds pay." 1 #f "" ((1990 "L20"))))
    (cons 'N1583 #(none "F1120S" "Other liabilities" 1 #f "" ((1990 "L21"))))
    (cons 'N1585 #(none "F1120S" "Capital stock" 1 #f "" ((1992 "L22") (1990 "L22a"))))
    (cons 'N1587 #(none "F1120S" "Paid-in or capital surplus" 1 #f "" ((1990 "L23"))))
    (cons 'N1591 #(none "F1120S" "Treasury stock" 1 #f "" ((1997 "L26") (1990 "L25"))))
   )
  )
  (cons 'Other
   (list
    (cons 'N000 #(none "" "Tax Report Only - No TXF Export" 0 #f ""))
   )
  )
 )
)
