# GNU Enterprise Application Server - Class Repository: Property
#
# Copyright 2003-2004 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it 
# and/or modify it under the terms of the GNU General Public 
# License as published by the Free Software Foundation; either 
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be 
# useful, but WITHOUT ANY WARRANTY; without even the implied 
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public 
# License along with program; see the file COPYING. If not, 
# write to the Free Software Foundation, Inc., 59 Temple Place 
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: Property.py 5448 2004-03-22 00:22:19Z reinhard $

from Base import *
from Namespace import createName


# =============================================================================
# Exceptions
# =============================================================================

class PropertyNotFoundError (ClassRepositoryError):
  def __init__ (self, classname, key):
    msg = _("Class '%(classname)s' has no property '%(property)s'") % \
            { "classname": classname, "property": key }
    ClassRepositoryError.__init__ (self, msg)


# =============================================================================
# Dictionary with all properties of a given class
# =============================================================================
class PropertyDict (BaseDictionary):

  # ---------------------------------------------------------------------------
  # Construct a Property-Dictionary for class aClass
  # ---------------------------------------------------------------------------
  def __init__ (self, session, aClass, predefs = None):
    BaseDictionary.__init__ (self, session, 'gnue_property')

    self.__class  = aClass
    self.__module = self.__class.module

    # predefs is a list of dictionaries, describing the properties
    if predefs is not None:
      for pDef in predefs:
        pMod = self.__module.modules.find (pDef ['gnue_module'])
        prop = Property (session, pMod, None, pDef)

        self._items [prop.fullName] = prop


  # ---------------------------------------------------------------------------
  # Create a new instance of a dictionary item
  # ---------------------------------------------------------------------------
  def _getNewItem (self, aObject):
    pMod = self.__module.modules.find (aObject.gnue_module.objectId)
    return Property (self._session, pMod, aObject, 
                     {"gnue_id": aObject.objectId})


  # ---------------------------------------------------------------------------
  # Restrict a reload () to the classes properties 
  # ---------------------------------------------------------------------------
  def _getReloadCondition (self):
    return [["eq", ""], ["field", u"gnue_class"], 
            ["const", self.__class.gnue_id]]


  # ---------------------------------------------------------------------------
  # Return a condition to match a given Property
  # ---------------------------------------------------------------------------
  def _getSingleCondition (self, key):
    return [["eq", ""], ["field", u"gnue_class"], 
            ["const", self.__class.gnue_id],
            ["eq", ""], ["field", u"gnue_name"], ["const", key]]


  # ---------------------------------------------------------------------------
  # Return a list of columns to be prepared by a find ()
  # ---------------------------------------------------------------------------
  def _getColumns (self):
    return [u"gnue_module", u"gnue_class", u"gnue_name", u"gnue_type",
            u"gnue_length", u"gnue_scale", u"gnue_comment"]


  # ---------------------------------------------------------------------------
  # Create a key-not-found-exception
  # ---------------------------------------------------------------------------
  def _itemNotFoundError (self, key):
    return PropertyNotFoundError (self.__class.fullName, key)


# =============================================================================
# A Business Object Property
# =============================================================================
class Property (BaseObject):

  _BASE_TYPES = ['boolean', 'date', 'datetime', 'number', 'string', 'time']
  _ID_TYPE    = "string"
  _ID_LENGTH  = 32
  _ID_SCALE   = 0

  # ---------------------------------------------------------------------------
  # Construct a new property from module, class and business object
  # ---------------------------------------------------------------------------
  def __init__ (self, session, module, object, pDefs = None):
    BaseObject.__init__ (self, session, 'gnue_property', object, pDefs)

    self.module   = module
    self.fullName = createName (self.module.gnue_name, self.gnue_name)
    self.column   = self.fullName

    # build appserver specific type information
    if self.gnue_type in ["string", "number"]:
      if self.gnue_type == "number" and self.gnue_length and self.gnue_scale:
        self.fullType = "%s(%d.%d)" % (self.gnue_type, self.gnue_length,
                                                       self.gnue_scale)
      elif self.gnue_length:
        self.fullType = "%s(%d)" % (self.gnue_type, self.gnue_length)
      else:
        self.fullType = self.gnue_type
    else:
      self.fullType = self.gnue_type

    # build database specific type information
    if self.gnue_type in self._BASE_TYPES:
      self.dbType   = self.gnue_type
      self.dbLength = self.gnue_length
      self.dbScale  = self.gnue_scale

    else:
      self.dbType   = self._ID_TYPE
      self.dbLength = self._ID_LENGTH
      self.dbScale  = self._ID_SCALE

    if self.dbType in ["string", "number"]:
      if self.dbType == "number" and self.dbLength and self.dbScale:
        self.dbFullType = "%s(%d.%d)" % (self.dbType, self.dbLength,
                                                      self.dbScale)
      elif self.dbLength:
        self.dbFullType = "%s(%d)" % (self.dbType, self.dbLength)
      else:
        self.dbFullType = self.dbType
    else:
      self.dbFullType = self.dbType

