# GNU Enterprise Application Server - Authentication Services
#
# Copyright 2001-2004 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: geasAuthentication.py 5636 2004-04-05 11:09:29Z reinhard $

import string

# =============================================================================
# Exceptions
# =============================================================================

class AuthError (gException):
  pass

# =============================================================================
# Basic Authentication Class ( abstract )
# =============================================================================

class geasAuthAgent:

  # ---------------------------------------------------------------------------
  # Initalize
  # ---------------------------------------------------------------------------

  def __init__ (self):
    pass

  # ---------------------------------------------------------------------------
  # authenticate an user
  # user contains the user name
  # auth contains authentication information (like password, fingerprint...)
  # return 1 on access ok  (0=access denied)
  # ---------------------------------------------------------------------------

  def authenticate(self, session, user, auth):
    gDebug (1, "Fake authentication for user %s" % user)
    return 1


  # ---------------------------------------------------------------------------
  # check if user x has access for class/table y
  # return 1 on access ok  (0=access denied)
  # ---------------------------------------------------------------------------

  def hasAccess (self, session, user, classname):
    gDebug (1, "Fake access check for class %s and user %s" % \
                         (classname,user))
    return 1

# =============================================================================
# Database Authentication Agent
# =============================================================================

class geasDBAuthAgent (geasAuthAgent):

  # ---------------------------------------------------------------------------
  # Initalize
  # ---------------------------------------------------------------------------

  def __init__ (self, internalSession):
    # creation and populating the user/password/tablelist list should be
    # moved here, when the management interface is implemented and an easy
    # way to update permissions is working
    self._intSess = internalSession

  # ---------------------------------------------------------------------------
  # authenticate an user
  # user contains the user name
  # auth contains authentication information (like password, fingerprint...)
  # return 1 on access ok  (0=access denied)
  # ---------------------------------------------------------------------------

  def authenticate (self, session, user, auth):
    authList = self._intSess.request ("gnue_useraccess",
                                     [['eq', ''],
                                      ['field', 'gnue_username'],
                                      ['const', user ]],
                                     [],
                                     ["gnue_username", "gnue_password",
                                      "gnue_accesslist"])

    authData = self._intSess.fetch (authList, 0, 10)

    if len (authData) == 0:
      raise AuthError, u_("User '%s' does not exist.") % user

    if len (authData) > 1:
      raise AuthError, u_("Internal Error: More than one (%(numrec)s) record "
                          "for user '%(username)s'." \
                       % {"numrec"  : len (authData), "username": user})

    if authData [0] [2] != auth ['password']:
      raise AuthError, u_("Invalid password for user '%s'") % user

    gDebug (1, "User '%s' logged in." % user)

    # possibly not the best solution
    session.tablelist = string.split (authData[0][3],',')

    return 1 # = has access

  # ---------------------------------------------------------------------------
  # check if user x has access for class/table y
  # ---------------------------------------------------------------------------

  def hasAccess (self, session, user, classname):
    # this should be changed to check for the user again.
    # it will only be secure if the protocol supports session-management too.
    tables = session.tablelist
    if hasattr (session, "tablelist") and ('all' in tables or classname in tables):
      return 1
    else:
      gDebug (1, "User '%(username)s' has no access to class %(classname)s." \
                 % {"username" : user, "classname": classname})
      return 0

# =============================================================================
# PAM Authentication Agent
# =============================================================================

class geasPAMAuthAgent (geasAuthAgent):

  # ---------------------------------------------------------------------------
  # Initalize
  # ---------------------------------------------------------------------------

  def __init__ (self):
    import PAM

  # ---------------------------------------------------------------------------
  # authenticate an user
  # user contains the user name
  # auth contains authentication information (like password, fingerprint...)
  # return 1 on access ok  (0=access denied)
  # ---------------------------------------------------------------------------

  def authenticate (self, session, user, auth):
    auth = PAM.pam ()
    auth.start (service)

    # TODO: add pam challenge invocation + parsing

    if 0:
      return 0 # = no access

    print _("User '%s' logged in.") % user

    return 1 # = has access

  # ---------------------------------------------------------------------------
  # check if user x has access for class/table y
  # ---------------------------------------------------------------------------

  def hasAccess (self, session, user, classname):
    # this should be changed to check for the user again.
    # it will only be secure if the protocol supports session-management too.
    if hasattr (session, "tablelist") and (classname in session.tablelist):
      return 1
    else:
      return 0
