# GNU Enterprise Application Server - Session Object
#
# Copyright 2001-2004 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: geasSession.py 5512 2004-03-25 23:17:47Z reinhard $

from gnue.appserver import data
import geasList, geasInstance
import whrandom

# =============================================================================
# Session class
# =============================================================================

class geasSession:

  # ---------------------------------------------------------------------------
  # Initalize
  # ---------------------------------------------------------------------------

  def __init__ (self, connections, authAdapter, sm, id):

    self.loggedIn = 0
    self.connections = connections
    try:
      self.database = gConfig('database')
    except:
      self.database = "gnue"

    self.__user = ""
    self.__lists = {}
    self.__listcount = 0
    self.__authAdapter = authAdapter
    self.__connection = None
    self.__dirtyInstances = {}

    self.sm = sm                        # The session manager
    self.id = id                        # The session id

  # ---------------------------------------------------------------------------
  # Get a class definition from a class name and check access
  # ---------------------------------------------------------------------------

  def __getClassdef (self, classname):

    # check if user has access rights for this class
    if not self.__authAdapter.hasAccess (self, self.__user, classname):
      raise gException, u_("Access to class '%s' denied") % classname

    return self.sm.classes [classname]

  # ---------------------------------------------------------------------------
  # Get a list of field names from a list of property names
  # ---------------------------------------------------------------------------

  def __getFieldlist (self, classdef, propertylist):

    return ([classdef.properties[p].column for p in propertylist])

  # ---------------------------------------------------------------------------
  # Log into the application server
  # ---------------------------------------------------------------------------

  def login (self, user, password):

    # This username/password is for the Application Server, not for the
    # database.
    self.__user = user
    self.loggedIn = self.__authAdapter.authenticate(self, user,
                                                   {'password':password} )
    if self.loggedIn:
      self.__connection = data.connection (self.connections, self.database)

    return self.loggedIn

  # ---------------------------------------------------------------------------
  # Logout from the application server
  # ---------------------------------------------------------------------------

  def logout (self):

    # FIXME: should the authAdapter be contacted?
    self.__connection.close ()
    self.loggedIn = 0

  # ---------------------------------------------------------------------------
  # Commit the active transaction
  # ---------------------------------------------------------------------------

  def commit (self):
    for instance in self.__dirtyInstances.values ():
      instance.validate ()

    self.__connection.commit ()
    self.__dirtyInstances = {}

  # ---------------------------------------------------------------------------
  # Rollback the active transaction
  # ---------------------------------------------------------------------------

  def rollback (self):

    self.__connection.rollback ()
    self.__dirtyInstances = {}

  # ---------------------------------------------------------------------------
  # Create a new list of business objects of a given class
  # ---------------------------------------------------------------------------

  def request (self, classname, conditions, sortorder, propertylist):

    classdef = self.__getClassdef (classname)

    # TODO: translate property names to column names in conditions
    sortlist = self.__getFieldlist (classdef, sortorder)
    fieldlist = self.__getFieldlist (classdef, [u'gnue_id'] + propertylist)

    recordset = self.__connection.query (classdef.table, fieldlist, conditions,
                                         sortlist)

    list = geasList.geasList (self, classdef, recordset,
                              [u'gnue_id'] + propertylist)

    self.__listcount += 1
    self.__lists [self.__listcount] = list
    list_id = self.__listcount
    return list_id;

  # ---------------------------------------------------------------------------
  # Check list id and raise exception if invalid, return list otherwise
  # ---------------------------------------------------------------------------

  def __getList (self, list_id):

    if self.__lists.has_key (list_id):
      return self.__lists [list_id]
    else:
      raise gException, u_("Can't find a list with ID '%s'") % list_id

  # ---------------------------------------------------------------------------
  # Count the number of objects in the list
  # ---------------------------------------------------------------------------

  def count (self, list_id):

    list = self.__getList (list_id)
    return list.count ();

  # ---------------------------------------------------------------------------
  # Fetch data from the database backend
  # ---------------------------------------------------------------------------

  def fetch (self, list_id, start, count):

    list = self.__getList (list_id)
    return list.fetch (start, count)

  # ---------------------------------------------------------------------------
  # Create a single geasInstance object from an existing record
  # ---------------------------------------------------------------------------

  def __findInstance (self, classdef, object_id, propertylist):

    table = classdef.table
    fieldlist = self.__getFieldlist (classdef, propertylist)
    record = self.__connection.findRecord (table, object_id, fieldlist)
    return geasInstance.geasInstance (self, record, classdef)

  # ---------------------------------------------------------------------------
  # Create a single geasInstance object for a new record
  # ---------------------------------------------------------------------------

  def __newInstance (self, classdef):

    table = classdef.table
    record = self.__connection.insertRecord (table)
    return geasInstance.geasInstance (self, record, classdef)

  # ---------------------------------------------------------------------------
  # Load data from the database backend
  # ---------------------------------------------------------------------------

  def load (self, classname, obj_id_list, propertylist):

    classdef = self.__getClassdef (classname)

    result = []
    for object_id in obj_id_list:
      if object_id:
        instance = self.__findInstance (classdef, object_id, propertylist)
        result.append (instance.get (propertylist))
      else:
        result.append ([classdef.properties[p].fullType for p in propertylist])

    return result

  # ---------------------------------------------------------------------------
  # Store data in the database backend
  # ---------------------------------------------------------------------------

  def store (self, classname, obj_id_list, propertylist, data):

    classdef = self.__getClassdef (classname)

    result = []
    i = 0
    for object_id in obj_id_list:
      if object_id:
        instance = self.__findInstance (classdef, object_id, [])
        new_object_id = object_id
      else:
        instance = self.__newInstance (classdef)
        new_object_id = instance.get ([u'gnue_id']) [0]
      instance.put (propertylist, data [i])
      i += 1
      result.append (new_object_id)
      self.__dirtyInstances [new_object_id] = instance

    return result

  # ---------------------------------------------------------------------------
  # Delete business objects
  # ---------------------------------------------------------------------------

  def delete (self, classname, obj_id_list):

    classdef = self.__getClassdef (classname)

    for object_id in obj_id_list:
      self.__connection.deleteRecord (classdef.table, object_id)
      if self.__dirtyInstances.has_key (object_id):
        del self.__dirtyInstances [object_id]

  # ---------------------------------------------------------------------------
  # Call a procedure of business objects
  # ---------------------------------------------------------------------------

  def call (self, classname, obj_id_list, procedurename, parameters):

    classdef = self.__getClassdef (classname)

    result = []
    for object_id in obj_id_list:
      instance = self.__findInstance (classdef, object_id, [])
      result.append (instance.call (procedurename, parameters))

    return result
