# GNU Enterprise Application Server - Language interface session
#
# Copyright 2003-2004 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it 
# and/or modify it under the terms of the GNU General Public 
# License as published by the Free Software Foundation; either 
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be 
# useful, but WITHOUT ANY WARRANTY; without even the implied 
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public 
# License along with program; see the file COPYING. If not, 
# write to the Free Software Foundation, Inc., 59 Temple Place 
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: Session.py 5499 2004-03-24 08:33:58Z reinhard $

from ObjectList import ObjectList
from Object import Object


# ===========================================================================
# Exceptions
# ===========================================================================

class NoContextError (gException):
  def __init__ (self, name):
    msg = u_("Cannot qualifiy name '%s', no context specified") % name
    gException.__init__ (self, name)


# ===========================================================================
# CLASS session: implement a session of the language interface
# ===========================================================================
class Session:

  # -------------------------------------------------------------------------
  # Constructor
  # -------------------------------------------------------------------------
  def __init__ (self, sessionManager, sessionId):
    self.__sm         = sessionManager
    self.__session_id = sessionId
    self.__context    = None

  # -------------------------------------------------------------------------
  # Get the session's sessionManager
  # -------------------------------------------------------------------------
  def getSessionManager (self):
    return self.__sm

  # -------------------------------------------------------------------------
  # Get the session's id
  # -------------------------------------------------------------------------
  def getSessionId (self):
    return self.__session_id

  # -------------------------------------------------------------------------
  # Set the default context
  # -------------------------------------------------------------------------
  def setcontext (self, context):
    self.__context = context

  # -------------------------------------------------------------------------
  # Ensure the given name is fully qualified using the current context
  # -------------------------------------------------------------------------
  def qualify (self, name):
    if '_' in name:
      return name
    else:
      if self.__context is None:
        raise NoContextError (name)
      else:
        return "%s_%s" % (self.__context, name)


  # -------------------------------------------------------------------------
  # Close the session 
  # -------------------------------------------------------------------------
  def close (self):
    if self.__sm is not None:
      self.__sm.close (self.__session_id, None)
      self.__sm = self.__session_id = None

  # -------------------------------------------------------------------------
  # Commit the current transaction
  # -------------------------------------------------------------------------
  def commit (self):
    if self.__session_id is not None:
      self.__sm.commit (self.__session_id)

  # -------------------------------------------------------------------------
  # Revoke the current transaction
  # -------------------------------------------------------------------------
  def rollback (self):
    if self.__session_id is not None:
      self.__sm.rollback (self.__session_id)

  # -------------------------------------------------------------------------
  # Return a collection of 'classname' matching the given arguments
  # -------------------------------------------------------------------------
  def find (self, classname, conditions = None, sortorder = ['gnue_id'],
            properties = []):
    return ObjectList (self, self.qualify (classname), conditions, sortorder,
                       properties)

  # -------------------------------------------------------------------------
  # Create a new instance of classname
  # -------------------------------------------------------------------------
  def new (self, classname):
    return Object (self, self.qualify (classname))



# =============================================================================
# Login-Session creates a new Session-Id using 'user' and 'password'
# =============================================================================
class LoginSession (Session):

  # -------------------------------------------------------------------------
  # Constructor
  # -------------------------------------------------------------------------
  def __init__ (self, sessionManager, user, password):
    Session.__init__ (self, sessionManager,
                  sessionManager.open ({'user': user, 'password': password}))


# =============================================================================
# InternalSession uses 0 as Session-Id
# =============================================================================
class InternalSession(Session):

  # -------------------------------------------------------------------------
  # Constructor
  # -------------------------------------------------------------------------
  def __init__ (self, sessionManager):
    Session.__init__ (self, sessionManager, 0)
