# GNU Enterprise Common - Base DB Driver - Connection
#
# Copyright 2001-2005 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: Connection.py 7057 2005-02-23 16:53:58Z johannes $

import copy
import string

__all__ = ['Connection']


# System Init:
#   +-- GConnections initializes
#   +-- GParser initializes:
#     +-- GDataSource loaded
#   +-- GDataSources Intialized
#     +-- GConnection.getDataObject()
#       +--
#
class Connection:
  def __init__(self, connections, name, parameters):
    self.manager = connections
    self.parameters = parameters
    self.name = name

    # Text encoding used by the database.
    # (Not all databases support this concept)
    try:
      self._encoding = parameters['encoding']
    except KeyError:
      try:
         self._encoding = gConfig('textEncoding')
      except:
        # TODO: Is this a safe default?
        self._encoding = 'iso8859-1'


  # Commit changes to the database
  def commit(self):
    pass

  # Rollback changes to the database
  def rollback(self):
    pass

  # Close the connection to the database backend
  def close (self):

    # Make sure to release the reference to the connection manager as well as
    # the introspector's instance (if available). This makes garbage collection
    # behave nice :)
    self.manager = None
    if hasattr (self, 'introspector'):
      self.introspector = None


  def connect(self, connectData):
    pass


  # ---------------------------------------------------------------------------
  # Create a new instance of the associated schema creator
  # ---------------------------------------------------------------------------

  def getSchemaCreator (self):
    """
    This function creates a new instance of the schema creator associated with
    this driver or None if no creator is available.
    """

    if hasattr (self, 'defaultCreator') and self.defaultCreator:
      return self.defaultCreator (self)
    else:
      return None


  # ---------------------------------------------------------------------------
  # update the schema definition 
  # ---------------------------------------------------------------------------

  def updateSchema (self, definition, codeOnly = False):
    """
    This function modifies the database schema according to the given
    definition, which is a dictionary of table defininitions (@see:
    Schema.Creation.Creation), where the tablenames are the keys.

    @param definition: sequence of table definitions
    @param codeOnly: if TRUE, only the code will be generated to modify the
        schema, no actions take place.
    @return: a tuple of three sequences (prologue, body, epilogue) holding the
        code to perform all schema modifications needed. These sequences should
        be executed in this order to successfully create the schema.
    """

    result = ([], [], [])

    schemaCreator = self.getSchemaCreator ()

    if schemaCreator is None:
      return result

    try:
      workingSet    = copy.copy (definition)
      constraintSet = {}

      # before any actions are performed, validate the given definitions
      for table in definition:
        schemaCreator.validate (table)

      # in a first run we remove all constraint definitions from the tables, so
      # we can add or alter all tables without having troubles with order of
      # occurence.
      for table in workingSet:
        # Do we have already a table with that name?
        res = self.introspector.find (name = table ['name'])

        if res is not None:
          method = schemaCreator.modifyTable
          # Please note: we keep existingFields sequence in all lowercase
          existingFields = [f.name.lower () for f in res [0].fields ()]

          # keep only new fields
          keep = []
          for field in table ['fields']:
            if not field ['name'].lower () in existingFields:
              keep.append (field)

          table ['fields'] = keep

          # on updates of a table we cannot use a primary key
          if table.has_key ('primarykey'):
            del table ['primarykey']
  
          # keep modified or new indices only
          if table.has_key ('indices'):
            keep = []

            # if driver supports index-introspection we'll use it
            if hasattr (res [0], 'indices'):
              for index in table ['indices']:
                oldIndex = None
                for (name, value) in res [0].indices.items ():
                  if name.lower () == index ['name'].lower ():
                    oldIndex = value

                if oldIndex is None:
                  keep.append (index)
                else:
                  old = [f.lower () for f in oldIndex ['fields']]
                  new = [f.lower () for f in index ['fields']]

                  if oldIndex ['unique'] != index ['unique'] or old != new:

                    # make sure the backend has a possibility to remove the old
                    # index before changing it
                    if not table.has_key ('old_indices'):
                      table ['old_indices'] = []
                    table ['old_indices'].append (index ['name'])

                    keep.append (index)
            else:
              # if no index-introspection available we only keep an index, if it
              # has new fields
              for index in table ['indices']:
                for field in index ['fields']:
                  if not field.lower () in existingFields:
                    # make sure the backend has a possibility to remove the old
                    # index before changing it
                    if not table.has_key ('old_indices'):
                      table ['old_indices'] = []
                    table ['old_indices'].append (index ['name'])

                    keep.append (index)
                    break

            table ['indices'] = keep

          # we create a constraint on a table update only if it contains new
          # fields
          if table.has_key ('constraints'):
            keep = []

            for constraint in table ['constraints']:
              for field in constraint ['fields']:
                if not field.lower () in existingFields:
                  keep.append (constraint)
                  break

            table ['constraints'] = keep

        else:
          method = schemaCreator.createTable

        if table.has_key ('constraints'):
          constraintSet [table ['name']] = {'name':        table ['name'],
                                          'constraints': table ['constraints']}
          del table ['constraints']

        # before we execute the planned action, have a look if there's still
        # work to be done
        perform = table.has_key ('fields') and len (table ['fields'])
        perform = perform or table.has_key ('primarykey')
        perform = perform or (table.has_key ('indices') and \
                              len (table ['indices']))

        if perform:
          schemaCreator.mergeTuple (result, method (table, codeOnly))

      # on the second run we process all constraints
      for table in constraintSet.values ():
        schemaCreator.mergeTuple (result,
                                schemaCreator.createTable (table, codeOnly))
      return result

    finally:
      schemaCreator.close ()
