# GNU Enterprise Common - Interbase/Firebird DB Driver - Schema Creation
#
# Copyright 2001-2005 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: Creation.py 6851 2005-01-03 20:59:28Z jcater $

import os
from gnue.common.datasources.GLoginHandler import BasicLoginHandler
from gnue.common.datasources.drivers.DBSIG2.Schema.Creation import \
    Creation as Base


# =============================================================================
# Class implementing schema creation for Interbase (5.x/6.x), Firebird (1.x)
# =============================================================================

class Creation (Base.Creation):

  MAX_NAME_LENGTH = 31
  ALTER_MULTIPLE  = False
  MAX_VARCHAR_LEN = 10921

  # ---------------------------------------------------------------------------
  # Create a new database
  # ---------------------------------------------------------------------------

  def createDatabase (self):
    """
    This function creates a new database as specified by the given connection's
    parameters. The password for the SYSDBA will be queried.
    """
    dbname   = self.connection.parameters.get ('dbname', None)
    username = self.connection.parameters.get ('username', 'gnue')
    password = self.connection.parameters.get ('password', 'gnue')
    host     = self.connection.parameters.get ('host', None)
    gsecbin  = self.connection.parameters.get ('gsecbin', 'gsec')

    res = {'_password': ''}
    while not res ['_password']:
      res = BasicLoginHandler ().getLogin (['as SYSDBA', 'Security Database',
                                        [['_password', 'Password', True]]], '')
    syspw = res ['_password']

    if host:
      dburl = "%s:%s" % (host, dbname)
    else:
      dburl = dbname

    code = u"%s -user sysdba -password %s -delete %s" % \
        (gsecbin, syspw, username)

    try:
      os.system (code)
    except:
      pass

    code = u"%s -user sysdba -password %s -add %s -pw %s" % \
        (gsecbin, syspw, username, password)

    try:
      # if creating the user fails we try to create the db anyway. Maybe this
      # is done from a remote system where no gsec is available, but the given
      # credentials are valid on the given server.
      os.system (code)
    except:
      pass

    self.connection._driver.create_database (\
       u"create database '%s' user '%s' password '%s' " \
        "default character set UNICODE_FSS" % (dburl, username, password))

    self.connection.manager.loginToConnection (self.connection)

    code = u"CREATE DOMAIN boolean AS smallint CHECK (value IN (0,1));"
    self.connection.makecursor (code)
    self.connection.commit ()


  # ---------------------------------------------------------------------------
  # Process a defaultwith attribute
  # ---------------------------------------------------------------------------

  def _defaultwith (self, code, tableName, fieldDef, forAlter):
    if fieldDef ['defaultwith'] == 'serial':
      seq = self._getSequenceName (tableName, fieldDef)
      code [0].append (u"CREATE GENERATOR %s%s" % (seq, self.END_COMMAND))
      fName = fieldDef ['name']

# TODO: this code works with options -f -o only
#      code [2].append (u"SET TERM ^ ;")
#      code [2].append (u"CREATE TRIGGER trg_%s FOR %s ACTIVE "
#          "BEFORE INSERT POSITION 0 AS BEGIN "
#          "IF (NEW.%s IS NULL) THEN NEW.%s = GEN_ID (%s,1); "
#          "END ^" % (fName, tableName, fName, fName, seq))
#      code [2].append (u"SET TERM ; ^")

      code [2].append (u"CREATE TRIGGER trg_%s FOR %s ACTIVE "
          "BEFORE INSERT POSITION 0 AS "
          "BEGIN IF (NEW.%s IS NULL) THEN NEW.%s = GEN_ID (%s,1); END"
           % (fName, tableName, fName, fName, seq))

    elif fieldDef ['defaultwith'] == 'timestamp':
      fieldDef ['default'] = "'NOW'"



  # ---------------------------------------------------------------------------
  # A key is an integer
  # ---------------------------------------------------------------------------

  def key (self, fieldDefinition):
    """
    Native datatype for a 'key'-field is 'integer'

    @param fieldDefinition: dictionary describing the field
    @return: 'integer'
    """
    return "integer"


  # ---------------------------------------------------------------------------
  # Create a native type representation for strings
  # ---------------------------------------------------------------------------

  def string (self, fieldDefinition):
    """
    This function returns an apropriate type for strings according to their
    lenght.

    @param fieldDefinition: dictionary describing the field
    @return: string with the native datatype
    """

    if fieldDefinition.has_key ('length') and \
       fieldDefinition ['length'] <= self.MAX_VARCHAR_LEN:
      return "varchar (%s)" % fieldDefinition ['length']

    elif not fieldDefinition.has_key ('length'):

      return "varchar (%s)" % self.MAX_VARCHAR_LEN
    else:
      return "blob"


  # ---------------------------------------------------------------------------
  # Create an apropriate type for a number
  # ---------------------------------------------------------------------------

  def number (self, fieldDefinition):
    """
    This function returns an apropriate type for a number according to the
    given length and precision.

    @param fieldDefinition: dictionary describing the field
    @return: string with the native datatype
    """
    scale  = 0
    length = 0

    if fieldDefinition.has_key ('precision'):
      scale = fieldDefinition ['precision']
    if fieldDefinition.has_key ('length'):
      length = fieldDefinition ['length']

    if scale == 0:
      if length <= 4:
        return "smallint"
      elif length <= 9:
        return "integer"
      else:
        return "numeric (%s,0)" % length
    else:
      return "numeric (%s,%s)" % (length, scale)


  # ---------------------------------------------------------------------------
  # Interbase has no native boolean type
  # ---------------------------------------------------------------------------

  def boolean (self, fieldDefinition):
    """
    This funciton returns a boolean, since this domain is created using the
    createDatabase () function.

    @param fieldDefinition: dictionary describing the field
    @return: 'boolean'
    """
    return "boolean"


  # ---------------------------------------------------------------------------
  # Native datatype for datetime
  # ---------------------------------------------------------------------------

  def datetime (self, fieldDefinition):
    """
    This function returns the native type for a datetime value, which is
    'timestamp' for interbase.

    @param fieldDefinition: dictionary describing the field
    @return: 'timestamp'
    """
    return "timestamp"

