#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2004 Free Software Foundation
#
# FILE:
# _dbsig/DBdriver.py
#
# DESCRIPTION:
# Generic implementation of dbdriver using Python DB-SIG v2
# specification.
#
# NOTES:
# The classes below are meant to be extended
#
# HISTORY:
#

__all__ = ['Connection']

from types import *

from gnue.common.datasources import Exceptions
from gnue.common.datasources.drivers.Base.Connection import Connection as BaseConnection
from gnue.common.apps import GDebug
import string
import sys
import mx.DateTime

class Connection (BaseConnection):
  """
  The base class for all drivers that use DBSIG2 compatible modules. All
  these drivers I{must} subclass this class.

  Descendants I{must} override the following class variables:

  @param _driver: the loaded Python module of the DBSIG2 driver

  Descendants I{may} override the following class variables:

  @param _boolean_false: Value to post to the database for boolean FALSE
      (defaults to '0')
  @param _boolean_true: Value to post to the database for boolean TRUE
      (defaults to '1')
  """
  _driver = None                        # DBSIG2 compatible driver module
  _boolean_false = '0'                  # value to pass for boolean FALSE
  _boolean_true  = '1'                  # value to pass for boolean TRUE
  _broken_fetchmany = False             # Does fetchmany () raise an exception
                                        # when no records are left?

  # This should be over-ridden only if driver needs more than user/pass
  def getLoginFields(self):
    return [['_username', _('User Name'),0],['_password', _('Password'),1]]

  def commit(self):
    GDebug.printMesg (5,"DB-SIG database driver: commit()")

    try:
      self.native.commit()
    except self._DatabaseError, value:
      raise Exceptions.ConnectionError, value

    self._beginTransaction()

  def rollback(self):
    GDebug.printMesg (5,"DB-SIG database driver: rollback()")

    try:
      self.native.rollback()
    except:
      pass	# I'm SURE this isn't right (jcater)
                # But not all db's support transactions

    self._beginTransaction()

  def close(self):
    """
    Close the DBSIG2 Connection.
    """
    self.native.close()
    
    
  def _beginTransaction(self):
    """
    Code necessary to force the connection into transaction mode...
    this is usually not necessary (MySQL is one of few DBs that must force)
    """
    pass


  # ===========================================================================
  # SQL statement handling
  # ===========================================================================

  # ---------------------------------------------------------------------------
  # Convert type into what the DBSIG2 driver wants as parameter
  # ---------------------------------------------------------------------------

  def _makeparam (self, value):
    """
    Convert any given value into the datatype that must be passed as parameter
    to the DBSIG2 cursor.execute() function.

    Descendants may override this function to to different type conversions.
    """
    if isinstance (value, UnicodeType):
      # Unicode: return encoded string
      return value.encode (self._encoding)

    elif isinstance (value, IntType):
      # Sometimes (sigh), IntType is used for booleans. Some datbases (e.g.
      # postgres) want boolean 0 and 1 values as strings.
      # Can be removed as soon as we require Python 2.3
      return str (value)

    elif isinstance (value, FloatType):
      # Some databases (especially postgres) want floating point numbers
      # passed as strings
      return str (value)

    elif sys.hexversion >= 0x02030000 and isinstance (value, BooleanType):
      # Booleans (Python 2.3 and later)
      if value:
        return self._boolean_true
      else:
        return self._boolean_false

    elif isinstance (value, mx.DateTime.DateTimeType):
      # mx.DateTime
      return self._driver.Timestamp (value.year, value.month, value.day,
                                     value.hour, value.minute, value.second)
    else:
      # Strings, Integers
      return value

  # ---------------------------------------------------------------------------
  # Change SQL statement and parameters to different paramstyles
  # ---------------------------------------------------------------------------

  # All these functions receive the statement and the params in pyformat style
  # and convert to whatever style is needed (according to function name).
  # One of these functions get called in makecursor.

  def __param_qmark (self, statement, parameters):
    s = statement
    l = []
    while True:
      start = string.find (s, '%(')
      if start == -1:
        break
      end = string.find (s, ')s', start)
      if end == -1:
        break
      key = s [start+2:end]
      s = s [:start] + '?' + s [end+2:]
      l.append (parameters [key])
    return (s, l)

  def __param_numeric (self, statement, parameters):
    s = statement
    l = []
    i = 0
    while True:
      start = string.find (s, '%(')
      if start == -1:
        break
      end = string.find (s, ')s', start)
      if end == -1:
        break
      i += 1
      key = s [start+2:end]
      s = s [:start] + (':%d' % i) + s [end+2:]
      l.append (parameters [key])
    return (s, l)

  def __param_named (self, statement, parameters):
    s = statement
    while True:
      start = string.find (s, '%(')
      if start == -1:
        break
      end = string.find (s, ')s', start)
      if end == -1:
        break
      key = s [start+2:end]
      s = s [:start] + (':%s' % key) + s [end+2:]
    return (s, parameters)

  def __param_format (self, statement, parameters):
    s = statement
    l = []
    while True:
      start = string.find (s, '%(')
      if start == -1:
        break
      end = string.find (s, ')s', start)
      if end == -1:
        break
      key = s [start+2:end]
      s = s [:start] + '%s' + s [end+2:]
      l.append (parameters [key])
    return (s, l)

  # ---------------------------------------------------------------------------
  # Create a new DBSIG2 cursor object and execute the given SQL statement
  # ---------------------------------------------------------------------------

  def makecursor (self, statement, parameters = None):
    """
    Create a new DBSIG2 cursor object and execute the given SQL statement.

    @param statement: The SQL statement as either 8-bit or unicode string. Can
        contain %(name)s style placeholders for parameters.
    @param parameters: A dictionary with the parameter values. The values of
        the dictionary can be 8-bit strings, unicode strings, integer or
        floating point numbers, booleans or mx.DateTime values.
    """
    checktype (statement, [StringType, UnicodeType])
    checktype (parameters, [DictionaryType, NoneType])
    if parameters:
      for (parameters_key, parameters_value) in parameters.items ():
        checktype (parameters_key, [StringType, UnicodeType])
        # checktype (parameters_value, .....) -- too many valid types :-)

    # Convert to encoded string for database
    if isinstance (statement, UnicodeType):
      s = statement.encode (self._encoding)
    else:
      s = statement

    if parameters:
      # convert parameter dictionary to encoded strings
      p = {}
      for (key, value) in parameters.items ():
        if isinstance (key, UnicodeType):
          k = key.encode (self._encoding)
        else:
          k = key
        p [k] = self._makeparam (value)

      # Convert parameters into correct style
      paramstyle = self._driver.paramstyle
      if paramstyle != 'pyformat':
        (s, p) = getattr (self, '_Connection__param_' + paramstyle) (s, p)

    else:
      p = None

    gDebug (1, "DBSIG2 Statement: %s" % s)
    gDebug (1, "DBSIG2 Parameters: %s" % p)
    # Create DBSIG2 cursor and execute statement
    try:
      cursor = self.native.cursor ()
      if p:
        cursor.execute (s, p)
      else:
        cursor.execute (s)
    except:
      (exc_type, exc_value, exc_traceback) = sys.exc_info ()
      try:
        cursor.close ()
      except:
        pass
      raise exc_type, exc_value, exc_traceback
    return cursor

  # ---------------------------------------------------------------------------
  # Execute the given SQL statement and return the result matrix
  # ---------------------------------------------------------------------------

  def sql (self, statement, parameters = None):
    """
    Execute the given SQL statement and return the result matrix.

    @param statement: The SQL statement as either 8-bit or unicode string. Can
        contain %(name)s style placeholders for parameters.
    @param parameters: A dictionary with the parameter values. The values of
        the dictionary can be 8-bit strings, unicode strings, integer or
        floating point numbers, booleans or mx.DateTime values.
    """
    result = None
    cursor = self.makecursor (statement, parameters)
    try:
      # This generates an exception if the statement didn't generate a
      # resultset
      result = cursor.fetchall ()
    except:
      pass
    cursor.close ()

    if result:
      return result
    else:
      return []

  # ---------------------------------------------------------------------------
  # Execute the given SQL statement that is expected to return a single value
  # ---------------------------------------------------------------------------

  def sql1 (self, statement, parameters = None):
    """
    Execute the given SQL statement that is expected to return a single value.
    If the query returns nothing, None is returned

    @param statement: The SQL statement as either 8-bit or unicode string. Can
        contain %(name)s style placeholders for parameters.
    @param parameters: A dictionary with the parameter values. The values of
        the dictionary can be 8-bit strings, unicode strings, integer or
        floating point numbers, booleans or mx.DateTime values.
    """
    cursor = self.makecursor (statement, parameters)
    try:
      result = cursor.fetchone ()
    finally:
      cursor.close ()

    if result:
      return result [0]
    else:
      return None
