#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2004 Free Software Foundation
#
# FILE:
# dbf/DBdriver.py
#
# DESCRIPTION:
# Virtual database driver for loading data from a DBF file
#
# NOTES:
# Used whenever a data has to be imported from dbase III+
#

__all__ = ['RecordSet']

from gnue.common.apps import GDebug
import string
from gnue.common.datasources.GDataObjects import *
from gnue.common.drivers.special.static.Driver import *
import dbf


###########################################################
#
# This is an static data driver for connectionless clients
# 
###########################################################
class DBF_DataObject (STATIC_DataObject): 

  def __init__(self):
    DataObject.__init__(self)
    self.triggerExtensions = TriggerExtensions(self)
    self._DatabaseError = Error
  
  def _createResultSet(self, conditions={}, readOnly=0, masterRecordSet=None, sql=""):
    return DBF_ResultSet(self, masterRecordSet=masterRecordSet)


  # We don't do connections (we are connectionless)
  def connect(self, connectData={}):
    try:
      GDebug.printMesg(3,'Open file: %s' % (connectData['directory']+connectData['dbname']))
      self._dataConnection = dbf.dbf(connectData['directory']+\
                                     connectData['dbname'])
    except IOError:
      tmsg = _('DBF file not found.')
      raise self._DatabaseError, tmsg
    except TypeError:
      tmsg = _('Wrong file format.')
      raise self._DatabaseError, tmsg

    # build field list
    self._fieldReferences=[]
    for f in self._dataConnection.fields:
      self._fieldReferences.append(string.lower(f[0]))
   
    self._postConnect()

  #
  # Schema (metadata) functions
  #

  # Return a list of the types of Schema objects this driver provides
  def getSchemaTypes(self):
    return [('table',_('Tables'),1)]

  # Return a list of Schema objects
  def getSchemaList(self, type=None):
    tablename=self._dataConnection.fname
    if tablename[-4:]=='.dbf':
        tablename=tablename[:-4]

    list = [Schema(attrs={'id':tablename,\
                          'name':tablename, \
                          'type':1},\
                   getChildSchema=self.__getFieldSchema)]
    return list


  # Find a schema object with specified name
  def getSchemaByName(self, name, type=None):
    tablename=self._dataConnection.fname
    if tablename[-4:]=='.dbf':
        tablename=tablename[:-4]

    if name==tablename:
      return self.getSchemaList()
    else:
      return None

  # Get fields for a table
  def __getFieldSchema(self, parent):

    list = []
    for field in self._dataConnection.fields:
      
      fname=string.lower(field[0])
      
      attrs={'id': "%s.%s" % (parent.id,fname), 'name': fname,
             'type':'field', 'nativetype': field[1],
             'required': 0}
      if field[1] == 'C':
        attrs['datatype']='text'
        attrs['length']='%s' % field[2]
      elif field[1] == 'N':
        attrs['datatype']='number'
      elif field[1] == 'D':
        attrs['datatype']='date'
      elif field[1] == 'L':
        attrs['datatype']='boolean'
      else:
        GDebug.printMesg(1,'WARNING: dbf native type error: %s' % field[1])

      list.append(Schema(attrs=attrs))

    return list




###########################################################
#
#
#
###########################################################
class DBF_ResultSet(STATIC_ResultSet):

  def __init__(self, dataObject, cursor=None, defaultValues={}, masterRecordSet=None):
    ResultSet.__init__(self, dataObject, \
            cursor, defaultValues, masterRecordSet)
    
    self._recordSetClass = STATIC_RecordSet

  # Returns 1=DataObject has uncommitted changes
  def isPending(self):
    return 0    # Static DataObjects cannot have pending changes :)

  # Post changes to the database
  def post(self):
    # Leave this here in case (for some bizarro reason)
    # a bound dataobject uses us as a master
    for record in (self._cachedRecords):
      record.post()

  # Load cacheCount number of new records
  def _loadNextRecord(self):
    if hasattr(self,"_alldataloaded"):
      return 0
    
    # Load static data
    for row in self._dataObject._dataConnection:
      dict = {}
      c=0
      for f in self._dataObject._dataConnection.fields:
        dict[string.lower(f[0])] = row[c]
        c+=1

      record=self._recordSetClass(parent=self,initialData=dict)
      
      self._cachedRecords.append (record)
      
      self._recordCount=self._recordCount+1

    self._alldataloaded = 1
      
    return 1



######################################
#
#  The following hashes describe
#  this driver's characteristings.
#
######################################

#
#  All datasouce "types" and corresponding DataObject class
#
supportedDataObjects = {
  'static': DBF_DataObject,
  'object': DBF_DataObject
}


