#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2004 Free Software Foundation
#
# FILE:
# sapdb/DBdriver.py
#
# DESCRIPTION:
# Driver to provide access to data via SAP's SAP-DB/Python Driver
# Requires SAP-DB (http://www.sapdb.org/)
#
# NOTES:
#
#   Supported attributes (via connections.conf or <database> tag)
#
#     host=      This is the SAP-DB host for your connection (optional)
#     dbname=    This is the SAP-DB database to use (required)
#     timeout=   Command timeout in seconds (optional)
#     isolation= Isolation level (options)
#     sqlmode=   INTERNAl or ORACLE (optional)
#     sqlsubmode= ODBC or empty (optional)
#

_exampleConfig = """
# This connection uses the SAP DB  driver
# We will be connecting to the SAP DB server on
# "localhost" to a database called "TST".
[sapdb]
comment = XYZ Development Database
provider = sapdb
dbname = TST
# host = localhost   # (optional)
# sqlmode = INTERNAL # (default) or ORACLE
# sqlsubmode = ODBC  # (for compatibility with the SAP DB ODBC driver)
# timeout = 900      # (command timeout in seconds)
# isolation = 1      # 0, 1 (default), 10, 15, 2, 20, 3, 30
"""
#### THIS IS AN UNTESTED DRIVER ####
####      Any volunteers?       ####

import string
from string import lower
import sys
from gnue.common.datasources import GDataObjects, GConditions, GConnections
from gnue.common.apps import GDebug
from gnue.common.datasources.drivers import DBSIG2

raise "This data driver has not been upgraded to the new format."

try:
  import sapdbapi as SIG2api
except ImportError, message:
  tmsg = _("Driver not installed: sapdbapi for SAP-DB 7.x \n[%s]") % message
  raise GConnections.AdapterNotInstalled, tmsg

class SAP_RecordSet(DBSIG2.RecordSet):
  pass


class SAP_ResultSet(DBSIG2.ResultSet):
  def __init__(self, dataObject, cursor=None, defaultValues={}, masterRecordSet=None):
    DBSIG2.ResultSet.__init__(self, dataObject, \
            cursor, defaultValues, masterRecordSet)
    self._recordSetClass = SAP_RecordSet



class SAP_DataObject(DBSIG2.DataObject):
  def __init__(self):
    DBSIG2.DataObject.__init__(self)
    self._DatabaseError = SIG2api.DatabaseError
    self._resultSetClass = SAP_ResultSet


  def connect(self, connectData={}):
    GDebug.printMesg(1,"SAP database driver initializing")
    try:
      options = {'autocommit': 'off'}
      for gnueName, sapdbName in [('sqlmode', 'sqlmode'),
                                  ('timeout', 'timeout'),
                                  ('isolation', 'isolation'),
                                  ('sqlsubmode', 'component')]:
          if connectData.has_key (gnueName):
              options [sapdbName] = connectData [gnueName]
      self._dataConnection = apply (SIG2api.connect,
        (connectData['_username'], connectData['_password'],
        connectData['dbname'], connectData.get ('host', '')),
        options)
      #self._dataConnection = SIG2api.connect( \
      #             user=connectData['_username'], \
      #             password=connectData['_password'], \
      #             database=connectData['dbname'], \
      #             host=connectData.get ('host', ''), \
      #             autocommit="off")
    except self._DatabaseError, value:
      raise GDataObjects.LoginError, value

    self._postConnect()

  def _postConnect(self):
    self.triggerExtensions = TriggerExtensions(self._dataConnection)


  #
  # Schema (metadata) functions
  #

  # Return a list of the types of Schema objects this driver provides
  def getSchemaTypes(self):
    return [ ('table',    _('Tables'),1),
             ('view',     _('Views'), 1),
             ('synonym',  _('Synonyms'),1),
             ('result',   _('Result Table'),1) ]


  # Return a list of Schema objects
  def getSchemaList(self, type=None):

    where_user = ""
    if type == None:
      where_type = "where TYPE <> 'SYSTEM' and TYPE <> 'SYNONYM' "
    else:
      where_type = "where TYPE='%s'" % string.upper(type)


    statement = \
      "select owner||'.'||tablename||'.'||type, " + \
        "owner||'.'||tablename table_name, " + \
        "type table_type " + \
        "from domain.tables %s" \
              % (where_type) + \
          "order by tablename "

    GDebug.printMesg(5,statement)

    cursor = self._dataConnection.cursor()
    cursor.execute(statement)

    list = []
    for rs in cursor.fetchall():
      list.append(GDataObjects.Schema(attrs={'id':rs[0], 'name':rs[1],
                         'type':string.lower(rs[2])},
                         getChildSchema=self.__getFieldSchema))

    cursor.close()
    return list


  # Find a schema object with specified name
  def getSchemaByName(self, name, type=None):

    where_user = ""
    parts = string.split(string.upper(name),'.')
    name = parts[-1]
    if len(parts) > 1:
      schema = " and owner='%s'" % parts[-2]
    else:
      schema = ""

    statement = \
      "select owner||'.'||tablename||'.'||type, " + \
        "owner||'.'||tablename table_name, " + \
        "type table_type, " + \
        "owner, tablename " + \
        "from domain.tables where tablename='%s'%s" \
              % (name, schema) + \
          "order by tablename "

    GDebug.printMesg(5,statement)

    cursor = self._dataConnection.cursor()
    cursor.execute(statement)

    list = []
    for rs in cursor.fetchall():
      list.append(GDataObjects.Schema(attrs={'id':string.lower(rs[0]), 'name':rs[1],
                         'type':rs[2], 'sapdbId': (rs [3], rs [4])},
                         getChildSchema=self.__getFieldSchema))

    cursor.close()

    try:
      return list[0]
    except:
      return None



  # Get fields for a table
  def __getFieldSchema(self, parent):

    # TODO: This does not support user-defined datatypes...
    # TODO: it will always report such as TEXT-like fields.

    schema, name, type = string.split(parent.id,'.')
    owner, basename = parent.sapdbId
    cursor = self._dataConnection.cursor()

#    if type == 'synonym':
#      statement = "select base_tabschema, base_tabname " + \
#                  "from syscat.tables " + \
#                  "where tabschema = '%s' and tabname='%s'" % (schema, name)
#
#      GDebug.printMesg(5,statement)
#
#      cursor.execute(statement)
#      rs = cursor.fetchone()
#      schema, name = rs

    statement = \
       "select owner||'.'||tablename||'.'||columnname, " + \
       "columnname, datatype, 'Y', len, dec " + \
       "from domain.columns " + \
       "where owner = '%s' and tablename = '%s' " % (owner, basename) + \
       'order by "POS"'

    GDebug.printMesg(5,statement)

    cursor.execute(statement)

    list = []
    for rs in cursor.fetchall():

      attrs={'id': rs[0], 'name': rs[1],
             'type':'field', 'nativetype': rs[2],
             'required': 'N'}

      if rs[2] in ('BOOLEAN','FIXED','FLOAT','INTEGER','LONG','SMALLINT'):
        attrs['precision'] = rs[5]
        attrs['datatype'] = 'number'
      elif rs[2] in ('DATE','TIME','TIMESTAMP'):
        attrs['datatype'] = 'date'
      else:
        attrs['datatype'] = 'text'

      if rs[5] != 0:
        attrs['length'] = rs[4]

      list.append(GDataObjects.Schema(attrs=attrs))

    cursor.close()
    return tuple(list)


class SAP_DataObject_Object(SAP_DataObject, \
      DBSIG2.DataObject_Object):

  def __init__(self):
    SAP_DataObject.__init__(self)

  def _buildQuery(self, conditions={},forDetail=None,additionalSQL=""):
    return DBSIG2.DataObject_Object._buildQuery(self, conditions,forDetail,additionalSQL)


class SAP_DataObject_SQL(SAP_DataObject, \
      DBSIG2.DataObject_SQL):
  def __init__(self):
    # Call DBSIG init first because SAP_DataObject needs to overwrite
    # some of its values
    DBSIG2.DataObject_SQL.__init__(self)
    SAP_DataObject.__init__(self)

  def _buildQuery(self, conditions={}):
    return DBSIG2.DataObject_SQL._buildQuery(self, conditions)


#
#  Extensions to Trigger Namespaces
#
class TriggerExtensions:

  def __init__(self, connection):
    self.__connection = connection





######################################
#
#  The following hashes describe
#  this driver's characteristings.
#
######################################

#
#  All datasource "types" and corresponding DataObject class
#
supportedDataObjects = {
  'object': SAP_DataObject_Object,
  'sql':    SAP_DataObject_SQL
}

def createConnection (conn, **overrides):
    from gnue.common.datasources.GConnections import GConnections
    connections = GConnections (r'D:\Python22\etc\connections.conf')
    parameters = connections.getConnectionParameters (conn).copy ()
    dataObject = connections.getDataObject (conn, 'object')
    parameters.update (overrides)
    dataObject.connect (parameters)
    return dataObject

def testConnection ():
    conn, user, pwd = sys.argv [1:4]
    connection = createConnection (conn, _username = user, _password = pwd)
    connection.getSchemaList ()

if __name__ == "__main__":
    testConnection ()
