#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2004 Free Software Foundation
#
# $Id: python.py 5663 2004-04-07 09:51:56Z reinhard $


import string
import re
import copy

from gnue.common.logic import language
from gnue.common.logic.adapters import Base

# =============================================================================
# Implementation of a language adapter for python
# =============================================================================

class LanguageAdapter (Base.BaseLanguageAdapter):
  """
  Implementation of a language engine for python
  """

  # ---------------------------------------------------------------------------
  # Create a new execution context
  # ---------------------------------------------------------------------------

  def createNewContext (self):
    """
    Create a python execution context
    """
    return PythonExecutionContext ()


# =============================================================================
# Python Execution Context
# =============================================================================

class PythonExecutionContext (Base.BaseExecutionContext):
  """
  This class implements an ExecutionContext for Python.
  """

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, runtime = None):
    Base.BaseExecutionContext.__init__ (self, runtime)

    self._globalNS = {}
    self._localNS  = {}


  # ---------------------------------------------------------------------------
  # Replace the local or global namespace
  # ---------------------------------------------------------------------------

  def defineNamespace (self, addNS, asGlobal = False):
    """
    This function replaces the local or global namespace with @addNS.
    """
    if asGlobal:
      self._globalNS = addNS
    else:
      self._localNS = addNS


  # ---------------------------------------------------------------------------
  # Add an object to the namespace
  # ---------------------------------------------------------------------------

  def bindObject (self, name, aObject, asGlobal = False):
    """
    Add @aObject as @name to the local or global namespace.
    """
    if asGlobal:
      self._globalNS [name] = aObject
    else:
      self._localNS [name] = aObject


  # ---------------------------------------------------------------------------
  # Add a function to the namespace
  # ---------------------------------------------------------------------------

  def bindFunction (self, name, aFunction, asGlobal = False):
    """
    Add @aFunction as @name to the local or global namespace.
    """
    if asGlobal:
      self._globalNS [name] = aFunction
    else:
      self._localNS [name] = aFunction


  # ---------------------------------------------------------------------------
  # Create a function
  # ---------------------------------------------------------------------------

  def buildFunction (self, name, code, parameters = {}):
    return PythonFunction (self, name, code, parameters)


# =============================================================================
# This class implements a virtual function using python
# =============================================================================

class PythonFunction (Base.VirtualFunction):
  """
  Implementation of a virtual function using Python.
  """

  # ---------------------------------------------------------------------------
  # Prepare a python function
  # ---------------------------------------------------------------------------

  def _prepare (self):
    """
    Preparing a sourcecode for python means compiling. This function compiles
    the code.
    """
    # The function name may only contain ascii characters and underscores.
    self.funcName = self.__identifier (self._name)
    if not self.funcName:
      self.funcName = 'unnamed'

    paramlist = string.join (['__namespace'] + \
           [self.__identifier (key) for key in self._parameters.keys ()], ", ")
    text      = self._code.rstrip ()

    try:
      tabPos = text.find ('\t')
      if tabPos > -1:
        raise gException, \
           u_("Sourcecode contains tab character at position %d") % tabPos


      # get the indentation level of the first line
      indent = 0
      for line in text.splitlines ():
        if len (line.strip ()) and line.lstrip () [0] != "#":
          indent = len (line) - len (line.lstrip ())
          break

      # add the functions header
      revisedCode  = "# -*- coding: utf-8 -*-\n"
      revisedCode += "def %s (%s):\n" % (self.funcName, paramlist)

      # add the namespace transformation loop
      revisedCode += "  for __add in __namespace.keys ():\n"
      revisedCode += "    exec '%s = __namespace [\"%s\"]' % (__add, __add)\n"
      revisedCode += "  del __add, __namespace\n\n"

      # add the original code
      for line in text.splitlines ():
        revisedCode += "  %s\n" % line [indent:]

      # and finalize the function by a call to it
      revisedCode += "  pass\n\n"
      revisedCode += "__result = %s (%s)\n" % (self.funcName, paramlist)

      self._compiled = compile (revisedCode.encode ('utf-8'),
                     '<%s>' % self._context.shortname.encode ('utf-8'), 'exec')
    except:
      raise language.CompileError, self._traceback (1)
          

  # ---------------------------------------------------------------------------
  # Execute the function
  # ---------------------------------------------------------------------------

  def execute (self, *args, **params):
    """
    This function creates a local namespace as a copy from the execution
    context's local namespace, adds all parameters to this namespace and
    executes the code.
    """
    try:
      localNS = copy.copy (self._context._localNS)
      localNS.update (params)
      localNS ['__namespace'] = localNS
      localNS ['abort'] = self.requestAbort

      # make sure we only use safe identifiers in our namespace
      self.__makeSafeNamespace (localNS)

      exec self._compiled in self._context._globalNS, localNS

      if localNS.has_key ('__result'):
        return localNS ['__result']
      else:
        return None
    except language.AbortRequest:
      # Pass through AbortRequests unchanged
      raise
    except:
      # All others raise a RuntimeError
      raise language.RuntimeError, self._traceback (2)


  # ---------------------------------------------------------------------------
  # Make sure we use proper identifiers
  # ---------------------------------------------------------------------------

  def __identifier (self, name):
    """
    This function translates @name to a valid python identifier, which means
    all non-letter characters are replaced by an underscore.
    """
    return re.sub ('[^a-zA-Z0-9]', '_', name)


  # ---------------------------------------------------------------------------
  # Make sure the given Namespace has no invalid identifiers
  # ---------------------------------------------------------------------------

  def __makeSafeNamespace (self, namespace):
    """
    This function replaces all invalid keys in the dict. @namespace by
    appropriate identifiers.
    """
    for key in namespace.keys ():
      safeId = self.__identifier (key)
      if safeId != key:
        namespace [safeId] = namespace [key]
        del namespace [key]
