# GNU Enterprise RPC interface - Base for all drivers
#
# Copyright 2001-2004 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: Base.py 5533 2004-03-26 16:19:16Z reinhard $

from types import *

import string
import sys
import thread
import traceback
import urlparse

from gnue.common.apps import GDebug

# Indicate that this is not a valid plugin
__noplugin__ = True

# =============================================================================
# Client adapter
# =============================================================================

class Client:
  """
  Basic client adapter
  """
  _default_transport = None
  _default_host      = 'localhost'
  _default_port      = None
  _default_path      = '/'

  # ---------------------------------------------------------------------------
  # Initialize object
  # ---------------------------------------------------------------------------

  def __init__ (self, params):

    checktype (params, DictionaryType)

    self._url       = None
    self._transport = None
    self._host      = None
    self._port      = None
    self._path      = None

    if params.has_key ('url'):

      # Connection defined as URL
      (self._transport, netloc, self._path, params, query, fragment) \
        = urlparse.urlparse (params ['url'])
      (self._host, self._port) = split (netloc, ':', 1)

    else:

      # Connection defined as transport/host/port/path info
      if params.has_key ('transport'): self._transport = params ['transport']
      if params.has_key ('host'     ): self._host      = params ['host'     ]
      if params.has_key ('port'     ): self._port      = params ['port'     ]
      if params.has_key ('path'     ): self._path      = params ['path'     ]

    # If some info isn't given, fall back to default values
    if not self._transport: self._transport = self._default_transport
    if not self._host:      self._host      = self._default_host
    if not self._port:      self._port      = self._default_port
    if not self._path:      self._path      = self._default_path

    # Make sure port is an integer
    self._port = int (self._port)

    # Now build the full URL
    self._url = '%s://%s:%d%s' % (self._transport, self._host, self._port,
                                  self._path)

    if params.has_key ('timeout'):
      self._timeout = params ['timeout']
    else:
      self._timeout = 1.0

    self.__open = True

  # ---------------------------------------------------------------------------
  # Set timeout
  # ---------------------------------------------------------------------------

  def setTimeout (self, timeout):

    self._timeout = timeout

  # ---------------------------------------------------------------------------
  # Request a (static) proxy object
  # ---------------------------------------------------------------------------

  def request (self, service):

    return ProxyObject (self, service, False)

  # ---------------------------------------------------------------------------
  # Close the connection
  # ---------------------------------------------------------------------------

  def close ():
    """
    Close the connection to the server
    """
    if self.__open:
      self._close ()

  # ---------------------------------------------------------------------------
  # Clean up
  # ---------------------------------------------------------------------------

  def __del__ (self):

    if self.__open:
      self._close ()

  # ---------------------------------------------------------------------------
  # Run a procedure on the server (abstract)
  # ---------------------------------------------------------------------------

  def _runMethod (self, method, *args, **params):
    """
    execute a remote method
    """
    pass

  # ---------------------------------------------------------------------------
  # Close the server (virtual)
  # ---------------------------------------------------------------------------

  def _close (self):

    pass

  # ---------------------------------------------------------------------------
  # Create a dynamic proxy object
  # ---------------------------------------------------------------------------

  def _createproxy (self, service):

    return ProxyObject (self, service, True)

# =============================================================================
# Proxy object for clients
# =============================================================================

class ProxyObject:

  # ---------------------------------------------------------------------------
  # Initialize proxy object
  # ---------------------------------------------------------------------------

  def __init__ (self, adapter, service, dynamic):

    self.__adapter = adapter
    self.__service = service
    self.__dynamic = dynamic

  # ---------------------------------------------------------------------------
  # Get a (proxy) method
  # ---------------------------------------------------------------------------

  def __getattr__ (self, attr):

    if attr[0] == '_':
      raise AttributeError, attr

    method = ProxyMethod (self.__adapter, self.__service + '.' + attr)
    self.__dict__ [attr] = method
    return method

  # ---------------------------------------------------------------------------
  # Set an object attribute (must start with '_')
  # ---------------------------------------------------------------------------

  def __setattr__ (self, attr, value):

    # FIXME: what do we need this for?

    if attr[0] == '_':
      self.__dict__[attr] = value
    else:
      raise AttributeError, attr

  # ---------------------------------------------------------------------------
  # Clean up
  # ---------------------------------------------------------------------------

  def __del__ (self):

    if self.__dynamic:
      self.__adapter._runMethod (self.__service + '._close')

# =============================================================================
# Proxy method for clients
# =============================================================================

class ProxyMethod:

  # ---------------------------------------------------------------------------
  # Initialize proxy method
  # ---------------------------------------------------------------------------

  def __init__ (self, adapter, methodname):

    self._adapter    = adapter
    self._methodname = methodname

  # ---------------------------------------------------------------------------
  # Run the method
  # ---------------------------------------------------------------------------

  def __call__ (self, *args, **params):
    return self._adapter._runMethod (self._methodname, *args, **params)

# =============================================================================
# Server adapter
# =============================================================================

class Server:
  """
  Basic server adapter
  """
  _default_transport = None
  _default_port      = None

  # ---------------------------------------------------------------------------
  # Initialize server object
  # ---------------------------------------------------------------------------

  def __init__ (self, rpcdef, bindings, params):
    self._bindings = bindings
    self._rpcdef = rpcdef

    checktype (params, DictionaryType)

    self._transport = None
    self._port      = None

    if params.has_key ('transport'): self._transport = params ['transport']
    if params.has_key ('port'     ): self._port      = params ['port'     ]

    # If some info isn't given, fall back to default values
    if not self._transport: self._transport = self._default_transport
    if not self._port:      self._port      = self._default_port

    # Make sure port is an integer
    self._port = int (self._port)

  # ---------------------------------------------------------------------------
  # Start server
  # ---------------------------------------------------------------------------

  def serve (self):
    pass

  # ---------------------------------------------------------------------------
  # Start server as new thread
  # ---------------------------------------------------------------------------

  def serveAsNewThread (self):
    thread.start_new_thread (self.serve, ())

  # ---------------------------------------------------------------------------
  # Return an exception
  # ---------------------------------------------------------------------------

  def raiseException (self, exception, message, event=None):
    raise exception, message

  # ---------------------------------------------------------------------------
  # Put info for current exception into a string to transfer it over RPC
  # ---------------------------------------------------------------------------

  def _traceback (self, count):
    """
    Returns the complete traceback of the exception as a string, where the
    first count lines of the traceback are hidden.

    This method can be used in descendant classes to encapsulate a traceback
    that happened on the server and send it over to the client.
    """
    lines = traceback.format_exception (*sys.exc_info ())
    del lines [1:count+1]
    return string.join (lines, '')
