#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2004 Free Software Foundation
#
# FILE:
# ObjectLibrarian.py
#
# DESCRIPTION:
# Helper module that keeps track of object references for use
# with non-Object interfaces
#
# NOTES:
#

import sha, time, thread

class InvalidObjectReference(StandardError):
  pass

_objects = {}
_index = 0.0


def getObjectReference(object):
  try:
    return object._ObjectLibrarian__refid
  except AttributeError:

    # This is to introduce some obscurity to the id handles...
    global _index
    _index += time.clock()
    object._ObjectLibrarian__refid = \
        sha.new("%s%s%s"%(id(object), object, _index)).hexdigest()
    
    # add creation time
    now = time.time()
    object._ObjectLibrarian__creationtime = now
    object._ObjectLibrarian__lastusage = now
    
    return object._ObjectLibrarian__refid


def archiveObject(object):
  _objects[getObjectReference(object)] = object


def retrieveObject(handle):
  try:
    object = _objects[handle]

    if hasattr(object,"_ObjectLibrarian__myGC"):
      # call private Garbage Collector (f.e. for sessions)
      object._ObjectLibrarian__myGC.beingRetrieved(object)
    else:
      # just store last used time
      object._ObjectLibrarian__lastusage = time.time()
    
    return object
  except KeyError:
    raise InvalidObjectReference


def deferenceObject(object):
  try:
    del _objects[object._ObjectLibrarian__refid]
  except KeyError:
    pass
  except AttributeError:
    pass

def getObjectLibrarianStatus():
  status={}
  status["objcount"]=len(_objects)
  return status

class GarbageCollector:
  def __init__(self):
    self.defaultTimeout=3600  # in seconds
    self.loopLength=20        # in seconds
    self.running=0
    
  def setDefaultTimeout(self,timespan):
    self.defaultTimeout=timespan

  def getDefaultTimeout(self):
    return self.defaultTimeout

  def setLoopLength(self,length):
    self.loopLength=length
    
  def getLoopLength(self):
    return self.loopLength
  
  def cleanUp(self):
    # is this threadsafe ?
    try:
      keys=_objects.keys()
      now=time.time()
      for handle in keys:
        try:
          object=_objects[handle]
          if not hasattr(object,"_ObjectLibrarian__myGC"):
            if object._ObjectLibrarian__lastusage + self.defaultTimeout < now:
              deferenceObject(object)
              GDebug.printMesg(4,"Timeout reached for object %s" & object)
              
        except:
          pass

    except:
      pass
      
  def loop(self):
    while 1==1:
      try:
        self.cleanUp()
        time.sleep(self.loopLength)
      except:
        pass
      
  def start_in_new_thread(self):
    if self.running:
      return
    self.running=1
    thread.start_new_thread(self.loop,())   


_global_GarbageCollector = GarbageCollector()


