# GNU Enterprise RPC interface - Py-XMLRPC client adapter
#
# Copyright 2001-2004 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: ClientAdapter.py 5533 2004-03-26 16:19:16Z reinhard $

# Requires py-xmlrpc from  http://sourceforge.net/Projects/py-xmlrpc
#
# Client Parameters:
#
#    url         The complete URI address of the XML-RPC service.
#                e.g, url="https://www.domain.com:9876/service.php"
#    ..or..
#    transport   The service transport (either "http" or "https")
#    host        The hostname or IP address of the service
#    port        The port that the service is located on
#

from types import *

import string
import sys
import types
import mx.DateTime

from gnue.common.rpc import client
from gnue.common.rpc.drivers import Base

try:
  import xmlrpc
except ImportError:
  tmsg = _("\nUnable to load xmlrpc.  To use the XML-RPC interface, \n"
           "please install xmlrpc from:\n"
           "    http://sourceforge.net/projects/py-xmlrpc\n"
           "the appropriate DEBIAN package is python-xmlrpc.deb")
  raise client.AdapterInitializationError, tmsg

import typeconv

# =============================================================================
# Client driver
# =============================================================================

class ClientAdapter(Base.Client):

  _default_transport = 'http'
  _default_port      = 8765

  # ---------------------------------------------------------------------------
  # Initialize object
  # ---------------------------------------------------------------------------

  def __init__ (self, params):

    Base.Client.__init__ (self, params)

    if (self._transport != 'http'):
      msg = _("py-xmlrpc only supports http as a transport.  If you need "
              "https, please have a look at the 'pw_xmlrpc' adapter.")
      raise client.AdapterConfigurationError, msg

    # setLoglevel
    if hasattr (params, 'loglevel'):
      xmlrpc.setLogLevel (params ['loglevel'])
    else:
      xmlrpc.setLogLevel (0)

    # create the real xmlrpc client
    self._client = xmlrpc.client (self._host, self._port, self._path)

  # ---------------------------------------------------------------------------
  # Run a procedure on the server
  # ---------------------------------------------------------------------------

  def _runMethod (self, method, *args, **params):

    __args = [typeconv.python_to_rpc (arg, client.InvalidParameter)
              for arg in args]

    try:
      result = self._client.execute (method, __args, self._timeout)
    except:
      # Bah! py_xmlrpc uses string exceptions!!
      if sys.exc_info () [0] == 'xmlrpc.error':
        if isinstance (sys.exc_info () [1], StringType):
          parts = string.split (sys.exc_info () [1], ':', 2)
          if parts [0] == '1':
            raise client.DistantError (parts [2])
      raise

    # check, if an object handle is sent
    # TODO: make a better check
    if type(result)==type("String") and len(result)==40 :

      # create an proxy for this handle
      return self._createproxy ("[%s]" % result)

    else:
      return typeconv.rpc_to_python (result, client.InvalidParameter)
