#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2002-2004 Free Software Foundation
#
# FILE:
# Objects.py
#
# DESCRIPTION:
# GObjects for the Schema definitions
#
# NOTES:
#
# $Id: Objects.py 5670 2004-04-07 13:27:12Z reinhard $

from gnue.common.definitions.GObjects import GObj
from gnue.common.definitions.GRootObj import GRootObj
from gnue.common.schema.GSData import verifyDataType, valueToNative
import GSParser
import types

class EFoundErrors (Exception):
  def __init__ (self):
    text = _("Errors found while processing GSD file.")
    Exception.__init__ (self, text)

class GSObject(GObj):
  pass

class GSSchema(GRootObj, GSObject):
  def __init__(self, parent=None):
    GRootObj.__init__(self, 'schema', GSParser.getXMLelements, GSParser)
    GSObject.__init__(self, parent, type='GSSchema')
    self.foundErrors = False

    self._inits.extend ([None, self._checkErrors])

  def _checkErrors (self):
    if self.foundErrors:
      raise EFoundErrors ()

class GSTables(GSObject):
  def __init__(self, parent):
    GSObject.__init__(self, parent, type='GSTables')

class GSTable(GSObject):
  def __init__(self, parent):
    GSObject.__init__(self, parent, type='GSTable')

class GSFields(GSObject):
  def __init__(self, parent):
    GSObject.__init__(self, parent, type='GSFields')

class GSField(GSObject):
  def __init__(self, parent):
    GSObject.__init__(self, parent, type='GSField')

  def _buildObject(self):
    # TODO: Added with Common 0.5.0; deprecate at some point
    if hasattr(self,'auto') and self.auto:
      self.defaultwith="serial"
      self.auto = 0

    return GSObject._buildObject(self)
  
class GSPrimaryKey(GSObject):
  def __init__(self, parent):
    GSObject.__init__(self, parent, type='GSPrimaryKey')

class GSPKField(GSObject):
  def __init__(self, parent):
    GSObject.__init__(self, parent, type='GSPKField')

class GSConstraints(GSObject):
  def __init__(self, parent):
    GSObject.__init__(self, parent, type='GSConstraints')

class GSConstraint(GSObject):
  def __init__(self, parent):
    GSObject.__init__(self, parent, type='GSConstraint')
    self._inits.append (self._validate)
    self.__tables = None


  # ---------------------------------------------------------------------------
  # Check a constraint definition
  # ---------------------------------------------------------------------------

  def _validate (self):
    self.type = self.type.lower ()

    try:
      if not self.type in ["unique", "foreignkey"]:
        raise Exception (_("Invalid constraint type '%s'.") % self.type)

      csFields = self.findChildrenOfType ('GSConstraintField')
      self.__checkFields (None, csFields)

      if self.type == "foreignkey":
        refFields = self.findChildrenOfType ('GSConstraintRef')
        if refFields is None:
          raise Exception (_("Constraint '%s' has no reference fields.") % \
                             self.name)
        self.__checkFields (refFields [0].table, refFields)

        if len (refFields) <> len (csFields):
          raise Exception (_("Constraint '%s' has unbalanced fields.") % \
                            self.name)

        self.__typeCheck (csFields, refFields)


    except Exception, message:
      print message
      setErrorFlag (self)


  # ---------------------------------------------------------------------------
  # find a table definition in the object hierachy for @tablename
  # ---------------------------------------------------------------------------

  def __findTable (self, tablename = None):
    # if no tablename is given we're looking for our parent table
    if tablename is None:
      return self.findParentOfType ('GSTable')

    if self.__tables is None:
      self.__tables = self.findParentOfType ('GSTables')

    if self.__tables is not None:
      for table in self.__tables.findChildrenOfType ('GSTable'):
        if table.name == tablename:
          return table

    return None


  # ---------------------------------------------------------------------------
  # Check if the table 'tablename' has all fields listed in 'cFields'
  # ---------------------------------------------------------------------------

  def __checkFields (self, tablename, cFields):
    """
    This function raises an exception if the table @tablename has not all
    fields listed in @cFields.
    """
    table = self.__findTable (tablename)
    if table is None:
      raise Exception (_("Cannot find table '%s'") % tablename)

    tbFields = table.findChildrenOfType ('GSField', True, True)

    if len (cFields) > len (tbFields):
      raise gException, u_("Constraint '%s' has more fields than the "
                           "table '%s'") % (self.name, table.name)

    for check in cFields:
      try:
        for field in tbFields:
          if field.name == check.name:
            raise Exception ('found')

      except:
        pass

      else:
        raise Exception (_("Table '%s' has no field '%s'.") % \
                           (table.name, check.name))


  # ---------------------------------------------------------------------------
  # Check if both sides of a reference matches in type
  # ---------------------------------------------------------------------------

  def __typeCheck (self, csFields, refFields):
    csTable = self.__findTable ()
    rfTable = self.__findTable (refFields [0].table)

    rfFields = {}
    myFields = {}

    for item in csTable.findChildrenOfType ('GSField', True, True):
      myFields [item.name] = item

    for item in rfTable.findChildrenOfType ('GSField', True, True):
      rfFields [item.name] = item


    for ix in range (0, len (csFields)):
      if myFields [csFields [ix].name].type != \
         rfFields [refFields [ix].name].type:
        raise gException, u_("Constraint '%s': typemismatch in reference "
                             "field '%s'.") % (self.name, csFields [ix].name)


class GSConstraintField(GSObject):
  def __init__(self, parent):
    GSObject.__init__(self, parent, type='GSConstraintField')

class GSConstraintRef(GSObject):
  def __init__(self, parent):
    GSObject.__init__(self, parent, type='GSConstraintRef')

class GSIndexes(GSObject):
  def __init__(self, parent):
    GSObject.__init__(self, parent, type='GSIndexes')

class GSIndex(GSObject):
  def __init__(self, parent):
    GSObject.__init__(self, parent, type='GSIndex')

class GSIndexField(GSObject):
  def __init__(self, parent):
    GSObject.__init__(self, parent, type='GSIndexField')

class GSData(GSObject):
  def __init__(self, parent):
    GSObject.__init__(self, parent, type='GSData')

class GSTableData(GSObject):
  def __init__(self, parent):
    GSObject.__init__(self, parent, type='GSTableData')

class GSRows(GSObject):
  def __init__(self, parent):
    GSObject.__init__(self, parent, type='GSRows')

class GSRow(GSObject):
  def __init__(self, parent):
    GSObject.__init__(self, parent, type='GSRow')

# =============================================================================
# Field-Values
# =============================================================================

class GSValue(GSObject):
  """
  This class implements a single data value of a row-collection. On
  Phase-I-init the values' datatype is determined - either by a matching
  column-definition, or by a direct type-argument - and a native python object
  is created for it.
  """
  def __init__(self, parent):
    GSObject.__init__(self, parent, type='GSValue')
    self.dataType  = None
    self.length    = None
    self.precision = None
    self.value     = None

    self._inits.append (self._validate)


  def _validate (self):
    if hasattr (self, "field"):
      column = self._findColumn (self.field)
    else:
      column = self._findColumn (self._getIndex ())

    try:
      self.value = valueToNative (self, column)

    except Exception, message:
      self.value = None

      print message
      setErrorFlag (self)


  # ---------------------------------------------------------------------------
  # Find a column definition either by it's index or by it's fieldname
  # ---------------------------------------------------------------------------

  def _findColumn (self, fieldSpec):
    """
    This function searches for a given field in a GSTableData's column
    definition. If @fieldSpec is an integer this number will be used as index
    to the column-definition collection. Ohterwise @fieldSpec is treated as the
    field name of the column to be returned.
    """
    tableData = self.findParentOfType ('GSTableData')
    if tableData is not None:
      colDefs = tableData.findChildOfType ('GSDefinition')
      if colDefs is not None:
        return colDefs.getColumn (fieldSpec)

    return None


  # ---------------------------------------------------------------------------
  # get the index of an instance in it's parent's child-collection
  # ---------------------------------------------------------------------------

  def _getIndex (self):
    """
    This function get's the objects index in it's parents child-collection
    """
    if self._parent is not None:
      for index in range (0, len (self._parent._children)):
        if self._parent._children [index] == self:
          return index

    raise Exception (_("GSD-Error: can't find myself in the XML tree?!"))



class GSDescription(GSObject):
  def __init__(self, parent):
    GSObject.__init__(self, parent, type='GSDescription')



# =============================================================================
# Column definitions
# =============================================================================

class GSDefinition (GSObject):
  """
  GSDefinition holds a collection of column definition instances. In
  Phase-I-init a dictionary with all columns is created as well as a sequence
  of columns (for index-based access).
  """
  def __init__ (self, parent):
    GSObject.__init__ (self, parent, type = "GSDefinition")
    self.columns = {}
    self.collist = []
    self._inits.append (self.__buildColumnDict)

  def __buildColumnDict (self):
    self.collist = self.findChildrenOfType ('GSColumn')
    self.columns = {}

    for col in self.collist:
      self.columns [col.field] = col


  def getColumn (self, fieldSpec):
    """
    If @fieldSpec is an integer type '@fieldSpec' is treated as an index to the
    columns-sequence, otherwise @fieldSpec is used as key into the
    columns-dictionary.
    """
    if isinstance (fieldSpec, types.IntType) and fieldSpec < len (self.collist):
      return self.collist [fieldSpec]

    elif self.columns.has_key (fieldSpec):
      return self.columns [fieldSpec]

    return None


# =============================================================================
# Column definitions
# =============================================================================

class GSColumn (GSObject):
  """
  This class implements a column definition. Phase-I-init verifies the datatype
  of the column definition and set's the properties 'typename', 'length' and
  'precision'.
  """
  def __init__ (self, parent):
    GSObject.__init__ (self, parent, type = "GSColumn")
    self.typename  = None
    self.length    = None
    self.precision = None
    self._inits.append (self._validate)

  def _validate (self):
    (self.typename, self.length, self.precision) = verifyDataType (self)


# -----------------------------------------------------------------------------
# recursively set an error flag in a object hierarchy
# -----------------------------------------------------------------------------

def setErrorFlag (aObject):
  """
  This function sets the property 'foundErrors' in a object and all its
  parents.
  """
  if aObject is not None:
    aObject.foundErrors = True
    setErrorFlag (aObject._parent)
