#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2002-2004 Free Software Foundation
#
# $Id: Importer.py 5212 2004-03-03 16:47:06Z johannes $
#

from gnue.common import VERSION
from gnue.common.schema import GSParser
from gnue.common.schema.importer.Definition import TableDefinition
from gnue.common.utils.FileUtils import openResource
from gnue.common.apps.GClientApp import GClientApp
from gnue.common.datasources import GDataSource, GConditions

from types import UnicodeType


class ENoCondition (Exception):
  def __init__ (self, tablename, rowIndex):
    msg = _("Row %d of table %s has no key fields.") % (rowIndex, tablename)
    Exception.__init__ (self, msg)


# =============================================================================
# Import data from a GNUe Schema Definition file into connection
# =============================================================================

class Importer (GClientApp):

  VERSION         = VERSION
  COMMAND         = "gnue-import"
  NAME            = "GNUe Schema Importer"
  USAGE           = "[options] file"
  COMMAND_OPTIONS = [
      [ 'connection', 'c', 'connection', True, 'gnue', 'connectionname',
        _("Use the connection <connectionname> for importing data.")],
      ]

  SUMMARY = _("GNUe Schema Importer reads data from a GNUe Schema ") + \
            _("Definition and imports it into a connection.")



  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------
  def __init__ (self, connections = None):
    ConfigOptions = {}
    GClientApp.__init__ (self, connections, 'schema', ConfigOptions)


  # ---------------------------------------------------------------------------
  # Walk through all command line options 
  # ---------------------------------------------------------------------------
  def __check_options (self):
    
    # do we have an accessible input file
    if not len (self.ARGUMENTS):
      self.handleStartupError (_("No input file specified."))

    try:
      self.__input = openResource (self.ARGUMENTS [0])

    except IOError:
      self.handleStartupError (_("Unable to open input file %s.") % \
                                 self.ARGUMENTS [0])



  # ---------------------------------------------------------------------------
  # iteration over all schema objects in the document tree
  # ---------------------------------------------------------------------------

  def __iterate_objects (self, sObject):
    if sObject._type == "GSTableData":
      self.__data_table (sObject)

    return 


  # ---------------------------------------------------------------------------
  # Process a tabledata node
  # ---------------------------------------------------------------------------
  def __data_table (self, sObject):
    data = TableDefinition (sObject.tablename)

    sObject.walk (self.__data_rows, dataDef = data)

    if len (data.rows) == 0:
      return 

    if isinstance (data.name, UnicodeType):
      tablename = data.name.encode ('utf-8')
    else:
      tablename = data.name

    flist = [item.field.encode ('utf-8') for item in data.rows [0].values]

    attributes = {}
    attributes ['name']     = "dts_%s" % tablename
    attributes ['database'] = self.OPTIONS ['connection']
    attributes ['table']    = tablename

    datasource = GDataSource.DataSourceWrapper (
      connections = self.connections,
      attributes  = attributes,
      fields      = flist)

    datasource._dataObject._unicodeMode = 1

    updCount = 0
    insCount = 0

    print _("Importing into %s") % tablename

    for rowIndex in range (0, len (data.rows)):
      row  = data.rows [rowIndex]
      cond = []

      for item in row.values:
        if hasattr (item, "key") and item.key:
          if isinstance (item.field, UnicodeType):
            fieldname = item.field.encode ('utf-8')
          else:
            fieldname = item.field

          if isinstance (item.value, UnicodeType):
            fvalue = item.value.encode ('utf-8')
          else:
            fvalue = item.value

          cond.extend ([["eq", ""], ["field", fieldname], 
                       ["const", fvalue]])

      if len (cond) == 0:
        raise ENoCondition (data.name, rowIndex)

      condition_tree = GConditions.buildTreeFromPrefix (cond)
      resultset = datasource.createResultSet (condition_tree)

      if resultset.firstRecord () is None:
        resultset.insertRecord ()
        insCount += 1
      else:
        updCount += 1

      for item in row.values:
        fieldname = item.field
        if isinstance (fieldname, UnicodeType):
          fieldname = fieldname.encode ('utf-8')

        resultset.current.setField (fieldname, item.value)

      resultset.post ()
    
    self.connections.commitAll ()

    print _("  Inserted %d row(s), updated %d row(s)") % (insCount, updCount)



  # ---------------------------------------------------------------------------
  # Iterate over all rows of a tabledata definition
  # ---------------------------------------------------------------------------
  def __data_rows (self, sObject, dataDef):
    if sObject._type == "GSRow":
      row = dataDef.addRow ()
      sObject.walk (self.__data_values, rowDef = row)


  # ---------------------------------------------------------------------------
  # Iterate over all values of a row definition
  # ---------------------------------------------------------------------------
  def __data_values (self, sObject, rowDef):
    if sObject._type == "GSValue":
      rowDef.values.append (sObject)



  # ---------------------------------------------------------------------------
  # Main program
  # ---------------------------------------------------------------------------
  def run (self):

    self.__check_options ()

    self.schema = GSParser.loadFile (self.__input)

    self.schema.walk (self.__iterate_objects)



# =============================================================================
# If executed directly, start the scripter
# =============================================================================
if __name__ == '__main__':
  Importer ().run ()
