#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2004 Free Software Foundation
#
# $Id: Definition.py 5614 2004-04-02 10:53:54Z johannes $
#

# =============================================================================
# The basic definition class
# =============================================================================

class Definition:
  """
  This class implements a basic definition. Each definition has a name, and a
  list of sequences: a prologue, a header, a body, a footer and finally an
  epilogue. A dump of a definition is done by concatenating all these sequences
  line by line.
  """

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, name = None):
    self.name     = name

    self.prologue = []
    self.header   = []
    self.body     = []
    self.footer   = []
    self.epilogue = []


  # ---------------------------------------------------------------------------
  # Return all sequences as one single list 
  # ---------------------------------------------------------------------------

  def merge (self):
    """
    This function merges all sequences into a single list
    """
    return self.prologue + self.header + self.body + self.footer + \
           self.epilogue

  # ---------------------------------------------------------------------------
  # Dump a definition to a given destination
  # ---------------------------------------------------------------------------

  def writeDefinition (self, destination, encoding = "UTF-8"):
    """
    This method writes all sequences to the given destination using 'encoding',
    which defaults to 'UTF-8'. The definition is written in the following
    order: prologue, header, body, footer, epilogue
    """
    for line in self.merge ():
      destination.write (line.encode (encoding) + "\n")



# =============================================================================
# Basic class for schema definitions
# =============================================================================

class SchemaDefinition (Definition):
  """
  This class introduces another sequence "fields" to the definition. This list
  holds GSField/GSIndexField instances.
  """
  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, name = None):
    Definition.__init__ (self, name)
    self.fields = []



# =============================================================================
# IndexDefinition introduces unique-flag
# =============================================================================

class IndexDefinition (SchemaDefinition):
  """
  This class has another public property 'unique', which describes wether a
  given index has the unique-flag set or not.
  """
  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, name = None, unique = False):
    SchemaDefinition.__init__ (self, name)
    self.unique = unique


# =============================================================================
# Constraint definition
# =============================================================================

class ConstraintDefinition (SchemaDefinition):
  """
  This class implements reference-constraints
  """
  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------
  def __init__ (self, name = None, kind = None):
    SchemaDefinition.__init__ (self, name)

    self.kind      = kind
    self.reftable  = None
    self.reffields = []


# =============================================================================
# PhaseDefinition
# =============================================================================

class PhaseDefinition (Definition):

  def __init__ (self, name = None, phase = 0):
    Definition.__init__ (self, name)
    self.phase = phase


# =============================================================================
# TableDefinition 
# =============================================================================
class TableDefinition:
  """
  """

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------
  def __init__ (self, name = None, action = 'create'):
    self.name        = name
    self.action      = action
    self.fields      = []
    self.primaryKey  = None
    self.indices     = {}
    self.constraints = {}

    self.phases      = {}


  # ---------------------------------------------------------------------------
  # Create a new IndexDefinition and add it to our index-sequence
  # ---------------------------------------------------------------------------

  def newIndex (self, name, unique = False):
    """
    This function creates a new IndexDefinition instance, adds it into the
    table definitions' index-dictionary and returns it as a function result.
    """
    index = IndexDefinition (name, unique)
    self.indices [index.name] = index
    return index


  # ---------------------------------------------------------------------------
  # Create a new ConstraintDefinition
  # ---------------------------------------------------------------------------

  def newConstraint (self, name, kind):
    """
    This function creates a new ConstraintDefinition instance, adds it into the
    table definitions' constraint-dictionary and returns it.
    """
    constraint = ConstraintDefinition (name, kind)
    self.constraints [constraint.name] = constraint
    return constraint


  # ---------------------------------------------------------------------------
  # Create a new primary key definition
  # ---------------------------------------------------------------------------

  def addPrimaryKey (self, name):
    """
    This function returns a new primary key definition.
    """
    self.primaryKey = IndexDefinition (name, True)
    return self.primaryKey


  # ---------------------------------------------------------------------------
  # Returns a field by name from the fields collection
  # ---------------------------------------------------------------------------
  def getField (self, fieldName):
    """
    This function searches a field with the name 'fieldName' in it's fields
    list. None is returned if no field with the given name was found.
    """
    for field in self.fields:
      if field.name == fieldName:
        return field

    return None


  # ---------------------------------------------------------------------------
  # Add a given phasedefinition and return it
  # ---------------------------------------------------------------------------

  def getPhase (self, phase):
    """
    This function returns the PhaseDefinition instance of @phase; if it's not
    defined it will be created
    """
    if not self.phases.has_key (phase):
      self.phases [phase] = PhaseDefinition (self.name, phase)
      
    return self.phases [phase]


# =============================================================================
# Definition class for data rows
# =============================================================================
class DataDefinition (Definition):
  """
  This class encapsulates data to be inserted into the table 'name'. The
  collection rows is a list of RowDefinition objects. These objects hold the
  actual insertion data.
  """

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------
  def __init__ (self, name = None):
    Definition.__init__ (self, name)

    self.rows     = []
    self.tableDef = None

  # ---------------------------------------------------------------------------
  # Add a new row to the collection
  # ---------------------------------------------------------------------------
  """
  This method extends the rows collection and returns a new RowDefinition
  instance.
  """
  def addRow (self):
    row = RowDefinition ()
    self.rows.append (row)
    return row


# =============================================================================
# Definition of a single data row
# =============================================================================
class RowDefinition:
  """
  This class encapsulates a single data row by providing three sequences: 
  columns, values and types. Columns is a list with column-names, values a list
  of corresponding values and types holds the columns datatypes.
  """

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------
  def __init__ (self):
    self.columns = []
    self.values  = []
    self.types   = []
