# GNU Enterprise Common Library - MySQL database driver using MySQLdb
#
# Copyright 2000-2005 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: mysqldbdrv.py 7695 2005-07-07 11:46:55Z johannes $

"""
Database driver plugin for MySQL backends using the MySQLdb DBSIG2 module.
"""

__all__ = ['Connection']

from gnue.common.datasources.drivers import DBSIG2
from gnue.common.datasources.drivers.sql.mysql import Behavior


# =============================================================================
# Test if plugin is functional
# =============================================================================

def __initplugin__ ():
  from gnue.common.datasources import GConnections
  try:
    import MySQLdb
  except ImportError:
    raise GConnections.DependencyError ('mysql-python',
        'http://sourceforge.net/projects/mysql-python')


# =============================================================================
# Driver info
# =============================================================================

class DriverInfo:

  name = "MySQLdb"

  url = "http://sourceforge.net/projects/mysql-python"

  doc = """
Description
-----------
Mysql-python, written by Andy Dustman, supports MySQL 3.22, 3.23, and
4.x.

Support
-------
Supported Platforms:

  - Linux/BSD
  - Solaris
  - MS Windows 98/NT/2000/XP (Installer available)

Connection Properties
----------------------
* host       -- This is the host for your connection (required)
* dbname     -- This is the name of the database to use (required)
* port       -- This is the port where the server is running (optional)

Examples
--------
 [myconn]
 provider=mysql          # Use the MySQLdb adapter
 host=localhost          # The hostname/IP of the MySQL host
 dbname=mydb             # The name of the MySQL database

Notes
-----
1. Transactions are supported if MySQL is compiled with proper
   transactional support (3.x series does NOT do this by default!).

2. Windows installer available at http://www.cs.fhm.edu/~ifw00065/.

3. Other than that, the driver is fully functional with no known serious
   problems.
"""


# =============================================================================
# Connection class
# =============================================================================

class Connection (DBSIG2.Connection):
  """
  Connection class for MySQL backends using the MySQLdb DBSIG2 module.
  """

  _drivername_    = 'MySQLdb'
  _behavior_      = Behavior.Behavior

  _boolean_true_  = 1
  _boolean_false_ = 0
  _std_datetime_  = True


  # ---------------------------------------------------------------------------
  # Get connection parameters
  # ---------------------------------------------------------------------------

  def _getConnectParams_ (self, connectData):

    # mandatory parameters
    kwargs = {'db'    : connectData ['dbname'],
              'user'  : connectData ['_username'],
              'passwd': connectData ['_password'] or ''}

    # optional parameters
    for gnueName, dbName in [('host', 'host'), ('port', 'port'),
                             ('encoding', 'unicode')]:
      if gnueName in connectData:
        if gnueName == 'port':
          kwargs [dbName] = int (connectData [gnueName])
        else:
          kwargs [dbName] = connectData [gnueName]

    return ([], kwargs)


  # ---------------------------------------------------------------------------
  # Start a new transaction
  # ---------------------------------------------------------------------------

  def _beginTransaction_ (self):
    # only available if MySQL is compiled with transaction support
    try:
      self._native.begin ()
    except:
      pass


  # ---------------------------------------------------------------------------
  # Return the current date, according to database
  # ---------------------------------------------------------------------------

  def getTimeStamp (self):
    return self.sql1 ("select current_timestamp")


  # ---------------------------------------------------------------------------
  # Return a sequence number from sequence 'name'
  # ---------------------------------------------------------------------------

  # def getSequence (self, name):
  #   (not available in MySQL)
