#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2005 Free Software Foundation
#
# FILE:
# GParser.py
#
# DESCRIPTION:
"""
Class that contains a SAX2-based XML processor for GNUe
"""
# NOTES:
#

import sys, copy, types
from gnue.common.definitions.GObjects import GObj
from gnue.common.definitions.GRootObj import GRootObj
from gnue.common.logic.GTrigger import GTrigger
from gnue.common.utils.FileUtils import openResource

try:
  from xml.sax import saxutils
  import xml.sax
except ImportError:
  print """
   This GNUe tool requires pythons XML module be installed.
   Typically this is the case, however some GNU/Linux distro's
   like Debian distribute this as a seperate package

   To install this package...
     On Debian: apt-get install python-xml

"""


import string
import os

from gnue.common.apps import errors
from gnue.common.formatting import GTypecast
from gnue.common.definitions.GParserHelpers import GContent


#######################################################
#
# Error classed raised for markup errors
#
class MarkupError(errors.ApplicationError):
  def __init__ (self, message, url = None, line = None):
    errors.ApplicationError.__init__ (self, message)
    text = [message]
    if url is not None:
      if line is not None:
        msg = u_("XML markup error in '%(url)s' at line %(line)s:") \
              % {'url': url, 'line': line}
      else:
        msg = u_("XML markup error in '%(url)s':") % {'url': url}

      text.insert (0, msg)

    self.detail = string.join (text, os.linesep)


#######################################################
#
# loadXMLObject
#
# This method loads an object from an XML file and
# returns that object.  If initialize is 1 (default),
# then the object is initialized and ready to go.
# Setting initialize to 0 is useful for a design
# environment where the object is not actually
# being used, but simply loaded as a document.
#
# "attributes" is a dictionary containing extra
# attributes that should be attached to this object.
#
#  e.g., if attributes={myproperty:[0,1,2]}, then
#    before the object is initialized or returned,
#    object.myproperty == [0,1,2].
#
#######################################################

def loadXMLObject(stream, handler, rootType, xmlFileType,
    initialize=True, attributes={}, initParameters={}, url = None):

  # Create a parser
  parser = xml.sax.make_parser()

  # Set up some namespace-related stuff for the parsers
  parser.setFeature(xml.sax.handler.feature_namespaces, 1)

  # Allow for parameter external entities
  ## Does not work with expat!!! ##
  ##parser.setFeature(xml.sax.handler.feature_external_pes, 1)

  # Create a stack for the parsing routine
  object = None

  # Create the handler
  dh = handler ()

  # pass the url of the stream and a pointer to the parser instance down to the
  # handler, so it's able to do better error reporting
  if url is None:
    url = hasattr (stream, 'url') and stream.url or '[unknown]'

  dh.url = url
  dh.parser = parser

  dh.initValidation()

  # Tell the parser to use our handler
  parser.setContentHandler(dh)
  
  try:
    parser.parse (stream)

  except xml.sax.SAXParseException, e:
    raise MarkupError, (errors.getException () [2], url, e.getLineNumber ())

  object = dh.getRoot()

  if not object:
    tmsg = u_("Error loading %s: empty definition file") % (xmlFileType)
    raise MarkupError, (tmsg, url)

  elif object._type != rootType:
    tmsg = u_("Error loading %(filetype)s: not a valid %(filetype)s definition "
              "(expected: %(expected)s, got: %(got)s)") \
           % {'filetype': xmlFileType,
              'expected': rootType,
              'got'     : object._type}
    raise MarkupError, (tmsg, url)

  dh.finalValidation ()

  # Set the root object's attributes
  #
  # There should only be 1 root object but GNUe Forms
  # allows for nested forms so we have to walk the tree
  #
  #
  #object.__dict__.update(attributes)

  object.walk(addAttributesWalker,attributes=attributes)

  if initialize:
    assert gDebug (7, "Initializing the object tree starting at %s" %(object))
    object.phaseInit(dh._phaseInitCount)

  # Break the cyclic reference in the content handler, so garbage collection is
  # able to collect it
  dh.parser = None


  return object
#######################################################
#
# addAttributesWalker
#
#######################################################
def addAttributesWalker(object, attributes={}):
  if isinstance(object,GRootObj):
    object.__dict__.update(attributes)



#######################################################
#
# normalise_whitespace
#
# Remove redundant whitespace from a string ( from xml-howto )
#
#######################################################
def normalise_whitespace(text):
  return string.join( string.split(text), ' ')


def default(attrs, key, default):
  try:
    return attrs[key]
  except KeyError:
    return default


#######################################################
#
# xmlHandler
#
# This class is called by the XML parser to
# process the xml file.
#
#######################################################
class xmlHandler(xml.sax.ContentHandler):

  # Set to default namespace (which would be dropped)
  # i.e., if default namespace is "Foo", then <Foo:test> would
  # just be processed as <test>.
  default_namespace = None

  # Drop any elements that have a namespaces we don't know about?
  ignore_unknown_namespaces = False

  def __init__(self):

    self.xmlElements = {}

    # This is a hack, currently used only by GRPassThru
    self.xmlMasqueradeNamespaceElements = None

    # Use namespace as a prefix in GObjects
    self.xmlNamespaceAttributesAsPrefixes = False

    # Internal stuff
    self.xmlStack = []
    self.nameStack = []
    self.bootstrapflag = False
    self.uniqueIDs = {}
    self.root = None
    self._phaseInitCount = 0

    self._requiredTags = []
    self._singleInstanceTags = []
    self._tagCounts = {}
    self.url = None

  #
  # Called by client code to get the "root" node
  #
  def getRoot(self):
    return self.root

  #
  # Builds structures need to verify requirements in the file
  #
  def initValidation(self):
    #
    # Build list of tags along with a list of
    # require tags
    #
    for element in self.xmlElements.keys():
      self._tagCounts[element] = 0
      try:
        if self.xmlElements[element]['Required'] == 1:
          self._requiredTags.append(element)
      except KeyError:
        pass

      try:
        if self.xmlElements[element]['SingleInstance'] == 1:
          self._singleInstanceTags.append(element)
      except KeyError:
        pass

  def finalValidation(self):
    # TODO: too simple a validation need to be per object instance
    #for element in self._singleInstanceTags:
    #  if self._tagCounts[element] > 1:
    #    raise MarkupError, _("File has multiple instances of <%s> when only one allowed") % (element)

    for element in self._requiredTags:
      if self._tagCounts[element] < 1:
        tmsg = u_("File is missing required tag <%s>") % (element)
        raise MarkupError, (tmsg, self.url)


  #
  # Called by the internal SAX parser whenever
  # a starting XML element/tag is encountered.
  #
  def startElementNS(self, qtag, qname, saxattrs):

    ns, name = qtag
    attrs = {}
    loadedxmlattrs = {}
    baseAttrs = {}

    if not ns or ns == self.default_namespace:
      #
      # No namespace qualifier
      #
      assert gDebug (7, "<%s>" % name)

      try:
        baseAttrs = self.xmlElements[name].get('Attributes',{})

      except KeyError:
        raise MarkupError, \
            (u_("Error processing <%(tagname)s> tag [I do not know what a "
                "<%(tagname)s> tag does]") % {'tagname': name}, self.url,
                self.parser.getLineNumber ())

      xmlns = {}

      for qattr in saxattrs.keys():
        attrns, attr = qattr

        if attrns:
          if not self.xmlNamespaceAttributesAsPrefixes:
            tmsg = _("Unexpected namespace on attribute")
            raise MarkupError, (tmsg, self.url, self.parser.getLineNumber ())
          prefix = attrns.split(':')[-1]
          attrs[prefix + '__' + attr] = saxattrs[qattr]
          xmlns[prefix] = attrns

        else:

          # Typecasting, anyone?  If attribute should be int, make it an int
          try:
            attrs [attr] = baseAttrs [attr].get ('Typecast', GTypecast.text)(saxattrs[qattr])
            loadedxmlattrs[attr] = attrs[attr]
          except KeyError:
            raise MarkupError, \
                (u_('Error processing <%(tagname)s> tag [I do not '
                   'recognize the "%(attribute)s" attribute]') \
                % {'tagname': name,
                   'attribute': attr}, self.url, self.parser.getLineNumber ())

          # this error shouldn't occure anymore
          #except UnicodeError:
          #  tmsg = _('Error processing <%s> tag [Encoding error: invalid character in "%s" attribute;]')\
          #      % (name, attr)
          #  raise MarkupError, tmsg

          #except StandardError, msg:
          #  raise
          #  tmsg = _('Error processing <%s> tag [invalid type for "%s" attribute; value is "%s"]')\
          #      % (name, attr, saxattrs[qattr])
          #  raise MarkupError, tmsg

          # If this attribute must be unique, check for duplicates
          if baseAttrs[attr].get('Unique',0): # default (baseAttrs[attr],'Unique',0):
            if self.uniqueIDs.has_key('%s' % (saxattrs[qattr])):
              tmsg = u_('Error processing <%(tag)s> tag ["%(attribute)s" '
                        'attribute should be unique; duplicate value is '
                        '"%(duplicate)s"]') \
                     % {'tag'      : name,
                        'attribute': attr,
                        'duplicate': saxattrs [qattr]}
              raise MarkupError, (tmsg, self.url, self.parser.getLineNumber ())

            # FIXME: If we really want to have a working 'Unique'-Attribute, we
            # would add the following line. But this would break forms atm.
            # self.uniqueIDs ["%s" % saxattrs [qattr]] = True


      for attr in baseAttrs.keys():
        try:
          if not attrs.has_key(attr):
  
            # Pull default values for missing attributes
            if baseAttrs[attr].has_key ('Default'):
              attrs[attr] = baseAttrs[attr].get('Typecast', GTypecast.text) (baseAttrs[attr]['Default'])
  
            # Check for missing required attributes
            elif baseAttrs[attr].get('Required', False): 
              tmsg = u_('Error processing <%(tagname)s> tag [required attribute '
                        '"%(attribute)s" not present]') \
                    % {'tagname'  : name,
                        'attribute': attr}
              raise MarkupError, (tmsg, self.url, self.parser.getLineNumber ())
        except (AttributeError, KeyError), msg: 
          raise errors.SystemError, _( 
            'Error in GParser xmlElement definition for %s/%s'
                ) % (name, attr) + '\n' + str(msg)

      attrs['_xmlnamespaces'] = xmlns

      if self.bootstrapflag:
        if self.xmlStack[0] != None:
          object = self.xmlElements[name]['BaseClass'](self.xmlStack[0])
      else:
        object = self.xmlElements[name]['BaseClass']()
        self.root = object
        self.bootstrapflag = 1

      self._tagCounts[name] += 1

      object._xmltag = name

    elif self.xmlMasqueradeNamespaceElements:
      #
      # namespace qualifier and we are masquerading
      #

      assert gDebug (7, "<%s:%s>" % (ns,name))

      for qattr in saxattrs.keys():
        attrns, attr = qattr

        attrs[attr] = saxattrs[qattr]
        loadedxmlattrs[attr] = saxattrs[qattr]

      try:
        object = self.xmlMasqueradeNamespaceElements(self.xmlStack[0])
      except IndexError:
        tmsg = u_("Error processing <%(namespace)s:%(name)s> tag: root "
                  "element needs to be in default namespace") \
              % {'namespace': ns,
                  'name'     : name}
        raise MarkupError, (tmsg, self.url, self.parser.getLineNumber ())

      object._xmltag = name
      object._xmlnamespace = ns
      object._listedAttributes = loadedxmlattrs.keys()

    elif self.ignore_unknown_namespaces:
      self.xmlStack.insert(0, None)
      self.nameStack.insert(0, None)
      return
    else:
      #
      # namespace qualifier and we are not masquerading
      #
      tmsg =  u_("WARNING: Markup includes unsupported namespace '%s'." ) % (ns)
      raise MarkupError, (tmsg, self.url, self.parser.getLineNumber ())


    # Save the attributes loaded from XML file
    # (i.e., attributes that were not defaulted)
    object._loadedxmlattrs = loadedxmlattrs
    
    # We make the url of the xml-stream and the line number of the element
    # available to the instance (for later error handling)
    object._lineNumber     = self.parser.getLineNumber ()
    object._url            = self.url

    # Set the attributes
    object.__dict__.update(attrs)

    self.xmlStack.insert(0, object)
    self.nameStack.insert(0, name)

    # processing trigger/procedure code from external files
    for qattr in saxattrs.keys():
      attrns, attr = qattr

      if baseAttrs.has_key ('file') and attr == 'file':
        textEncoding=gConfig('textEncoding')
        handle = openResource(attrs[attr])
        text = handle.read().decode(textEncoding)
        handle.close()
        
        if self.xmlStack[0] != None:
          GContent(self.xmlStack[0], text)


  #
  # Called by the internal SAX parser whenever
  # text (not part of a tag) is encountered.
  #
  def characters(self, text):

    if self.xmlStack[0] != None:

      # Masqueraging namespace elements, then keep content
      xmlns = self.xmlMasqueradeNamespaceElements and \
          isinstance(self.xmlStack[0],self.xmlMasqueradeNamespaceElements)

      # Should we keep the text?
      if xmlns or self.xmlElements[self.nameStack[0]].get('MixedContent',0):

        if xmlns or self.xmlElements[self.nameStack[0]].get('KeepWhitespace',0):
          GContent(self.xmlStack[0], text)
        else:
          # Normalize
          if len(string.replace(string.replace(string.replace(text,' ',''),'\n',''),'\t','')):
            text = normalise_whitespace (text)
          else:
            text = ""
          if len(text):
            GContent(self.xmlStack[0], text)


  #
  # Called by the internal SAX parser whenever
  # an ending XML tag/element is encountered.
  #
  def endElementNS(self, qtag, qname):
    ns, name = qtag
    self.nameStack.pop(0)
    child = self.xmlStack.pop(0)

    if not child:
      return

    inits = child._buildObject()
    self._phaseInitCount = (inits != None and inits > self._phaseInitCount \
                            and inits or self._phaseInitCount)
    assert gDebug (7, "</%s>" % name)



class GImportItem(GObj):
  def __init__(self, parent=None, type="GCImport-Item"):
    GObj.__init__(self, parent, type=type)
    self._loadedxmlattrs = {} # Set by parser
    self._inits = [self.primaryInit]
    self._xmlParser = self.findParentOfType(None)._xmlParser

  def _buildObject(self):
    if hasattr(self,'_xmltag'):
      self._type = 'GC%s' % self._xmltag
    if not hasattr(self,'_importclass'):
      self._importclass = self._xmlParser\
         .getXMLelements()[string.lower(self._type[9:])]['BaseClass']
    return GObj._buildObject(self)

  def primaryInit(self):
     #
     # Open the library and convert it into objects
     #
     handle = openResource(self.library)
     parent = self.findParentOfType (None)

     # Let the parent provide it's instance either as _app or _instance
     if hasattr (parent, '_instance'):
       instance = parent._instance
     elif hasattr (parent, '_app'):
       instance = parent._app
     else:
       instance = None

     form = self._xmlParser.loadFile (handle, instance, initialize = 0)
     handle.close ()
     id = 'id'
     if hasattr(self,'name'):
         id = 'name'
     #
     # Configure the imported object, assign as a child of self
     #
     rv = self.__findImportItem(self, form, id)
     if rv != None:
       rv.setParent (self)
       rv._IMPORTED = 1
       self._children.append(rv)
       #
       # transfer attributes reassigned during the import
       #
       for key in self._loadedxmlattrs.keys():
         if key[0] != '_':
           rv.__dict__[key] = self._loadedxmlattrs[key]
           assert gDebug (7, ">>> Moving %s" % key)
       rv._buildObject()
     else:
         raise MarkupError, \
             (u_("Unable to find an importable object named %(name)s in "
                 "%(library)s") \
              % {'name'   : self.name, 'library': self.library},
              self.url)

  #
  # __findImportItem
  #
  # finds the item in the object tree with the
  # same name and instance type
  #
  def __findImportItem(self, find, object, id):
     if isinstance(object, find._importclass) and \
        hasattr(object, id) and \
        object.__dict__[id] == find.__dict__[id]:
       return object
     elif hasattr(object,'_children'):
       rv = None
       for child in object._children:
         rv = self.__findImportItem(find, child, id)
         if rv:
           break
       return rv
     else:
       return None


class GImport(GObj):
  def __init__(self, parent=None):
    GObj.__init__(self, parent, type="GCImport")
    self.library = ""
    self._form = None
    self._inits = [self.primaryInit]
    self._xmlParser = self.findParentOfType(None)._xmlParser

  def primaryInit(self):
    handle = openResource(self.library)
    form = self._xmlParser.loadFile(handle, self.findParentOfType(None)._app, initialize=0)
    handle.close()

    for attribute in self._loadedxmlattrs.keys():
      if attribute != 'library':
        importAll =  self._loadedxmlattrs[attribute] == "*"
        importNames = string.split(string.replace(self._loadedxmlattrs[attribute],' ',''),',')

        instanceType = self._xmlParser.getXMLelements()[string.lower(attribute)]['BaseClass']

        if importAll or len(importNames):
          for child in form._children:
            if isinstance(child,instanceType) and \
               (importAll or child.name in importNames):
              child.setParent (self)
              child._IMPORTED = 1
              self._children.append(child)
              child._buildObject()

def buildImportableTags(rootTag, elements):
    #
    # Scans xml elements and looks for Importable = 1
    # Items with this set can be imported
    # If an object needs to be importable,
    # simply add its tag name to the tuple below
    # and make sure it has a "name" attribute
    # (otherwise we don't know how to reference
    # it in the imported file).
    #
    importElement = {'BaseClass': GImport,
                     'Attributes': {'library': {
                                      'Required': 1,
                                      'Typecast': GTypecast.name },
                                   },
                     'ParentTags': rootTag,
                     }

    for key in elements.keys():
     if elements[key].has_key('Importable') and elements[key]['Importable']:
       name = "import-%s" % key
       copy._deepcopy_dispatch[types.FunctionType] = copy._deepcopy_atomic
       copy._deepcopy_dispatch[types.ClassType] = copy._deepcopy_atomic
       copy._deepcopy_dispatch[type(int)] = copy._deepcopy_atomic

       p = copy.deepcopy(elements[key])
       p['BaseClass'] = GImportItem

       if not p.has_key('Attributes'):
         p['Attributes'] = {}

       p['Attributes']['library'] = {
          'Required': 1,
          'Typecast': GTypecast.name }
       p['MixedContent'] = 0
       p['Required'] = 0
       elements[name] = p

       importElement['Attributes'][key] =  {
         'Typecast': GTypecast.name,
         'Default': ""  }

    if len(importElement['Attributes'].keys()):
      elements['import'] = importElement
    return elements
