#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2004 Free Software Foundation
#
# FILE:
# PropertyEditor.py
#
# DESCRIPTION:
#
# NOTES:
#

__all__ = ['PropertyEditor']

import sys, os, string
from wxPython.wx import *
from wxPython.grid import *
from gnue.common.apps import GDebug
from gnue.common.formatting import GTypecast
from gnue.designer.base.ToolBase import *

class PropertyEditor (ToolBase):

  runtime_section = 'PropertyEditor'
  default_dock = 'left-1'

  def init(self):
    self.panel = wxPanel(self,-1,style=wxSIMPLE_BORDER, pos=wxPoint(6,6))
    self.notebook = wxNotebook(self.panel, -1, style=wxNB_BOTTOM)

    self.object = None

    # EventAware provided by ToolBase
    self.registerEventListeners({
                       'ObjectSelected'      : self.onSetCurrentObject,
                       'ObjectCreated'       : self.onCreateObject,
                       'ObjectModified'      : self.onModifyObject,
                       'ObjectDeleted'       : self.onDeleteObject,
                      })

    self.supplemental = []
    EVT_SIZE(self, self.__onSize)
    self.main = self.createMainPage()
    self.notebook.AddPage(self.main,'Properties')


  def createMainPage(self):
    return InspectorPanel(self, self.notebook)

  def onSetCurrentObject (self, event):
    object = event.object
    handler = event.originator
    if object == None:
      return
    if object != self.object:
      self.setCurrent(object)

  def setCurrent(self, object):
      self.object = object
      self.reset()
      self.main.setCurrent(object)
      self.notebook.SetPageText(0, self.main.getPageText())
      i = 1
      for page in self.supplemental:
        page.setCurrent(object)
        self.notebook.SetPageText(i, page.getPageText())
        i += 1

  def addPage(self, inspector, label):
    self.supplemental.append(inspector)
    self.notebook.AddPage(inspector, label)

  def reset(self):
    for i in range(len(self.supplemental)):
      self.notebook.DeletePage(1)
    self.supplemental = []

  def onCreateObject (self, event):
    object = event.object
    handler = event.originator

  def onModifyObject (self, event):
    object = event.object
    if object == None:
      return

    if event.originator != __name__:
      for page in [self.main] + self.supplemental:
        page.resetValues(object)

  def onDeleteObject (self, event):
    object = event.object
    handler = event.originator
    if object == None:
      return
    if handler != __name__:
      pass


  def __onSize(self, event):
    w,h = self.GetClientSizeTuple()
    self.panel.SetSize((w-12,h-12))
    self.notebook.SetSize(self.panel.GetClientSizeTuple())


#
#
#
class InspectorPanel(wxScrolledWindow):

  NAMESPACE = ""

  def __init__(self, editor, parent):
    wxScrolledWindow.__init__(self, parent, -1, style = wxHSCROLL)
    self.splitter = splitter = wxSplitterWindow(self, -1, style=wxSP_FULLSASH | wxSP_NOBORDER)
    self.object = None
    self.editor = editor
    self.fields = []
    self.labels = []

    # These will be set in _setCurrent
    self.maxy = 0
    self.maxLabel = 0

    self.labelPanel = wxPanel(splitter, -1)
    self.fieldPanel = wxPanel(splitter, -1)
    splitter.SplitVertically(self.labelPanel, self.fieldPanel)
    EVT_SPLITTER_SASH_POS_CHANGED(self, self.splitter.GetId(),self.__onSize)
    EVT_SIZE(self, self.__onSize)


  def getAttributes(self, object):
    elements = self.editor.instance.incubator.elements
    try:
      return elements[(object._type[2:]).lower()]['Attributes']
    except KeyError:
      return {}


  def setCurrent(self, object):
      self._setCurrent(object)

  def _setCurrent(self, object):
      self.object = object
      self.attributes = self.getAttributes(object)

      # Define our rows
      self.rowList = self.attributes.keys()
      self.rowList.sort()

      # Get rid of any old fields
      for i in range(len(self.fields)):
        field = self.fields.pop()
        field.Destroy()
        label = self.labels.pop()
        label.Destroy()

      # Only show properties for nondeprecated values (unless set)
      i = 0
      while i < len(self.rowList):
        key = self.rowList[i]
        xkey = key.replace(':','__')
        if self.attributes[key].has_key ('Deprecated') and \
           self.attributes[key]['Deprecated'] and \
           (not hasattr(object, xkey) or \
              (self.attributes[key].has_key('Default') and \
               object.__dict__[xkey] == self.attributes[key]['Default'] ) ):
          self.rowList.pop(i)
        else:
          i = i + 1

      y = 2
      maxLabel = 0
      for key in self.rowList:

        xkey = key.replace(':','__')

        # If an attribute has a "Label", use it;
        # otherwise create one from its name.
        try:
          text = self.attributes[key]['Label']
        except KeyError:
          # This little tidbit does mixed case w/'_' as separators
          words = string.split(key.split(':',1)[-1],'_')
          for j in range(len(words)):
            words[j] = words[j].capitalize()
          text = string.join(words)

        if self.attributes[key]['Typecast'] == GTypecast.boolean:
          text += _('?')
        else:
          text += ':'

        label = wxStaticText(self.labelPanel, -1, text)

        # Determine the type of Cell Editor we want
        # (Integer, Boolean, Dropdown, Char)
        if self.attributes[key].has_key('ValueSet'):
          field = LimitedSetEditor(self.fieldPanel,self.attributes[key])
        elif self.attributes[key].has_key('References'):
          tag, attr = self.attributes[key]['References'].split('.')
          objectList = self.editor.instance.getObjectList(tag)
          field = LinkedTextEditor(self.fieldPanel, self.attributes[key], objectList, attr)
        elif self.attributes[key]['Typecast'] == GTypecast.boolean:
          field = BoolEditor(self.fieldPanel,self.attributes[key])
        elif self.attributes[key]['Typecast'] in (GTypecast.integer,
                GTypecast.number, GTypecast.whole):
          field = IntEditor(self.fieldPanel,self.attributes[key])
        else:
          field = TextEditor(self.fieldPanel,self.attributes[key])

        EVT_KILL_FOCUS(field, self.__valueModified)
        EVT_KEY_UP(field, self.__enterPressed)

        # Align the centers of the field and label
        fx, fy = field.GetSizeTuple()
        lx, ly = label.GetSizeTuple()
        maxy = max(fy, ly)
        maxLabel = max(maxLabel, lx+4)
        field.SetPosition((2, int((maxy - fy)/2) + y))
        label.SetPosition((4, int((maxy - ly)/2) + y))

        # Next y...
        y += maxy


        # Set the initial value of the cells to the current property values
        if hasattr(object, xkey):
          field.SetValue(object.__dict__[xkey])

        self.fields.append(field)
        self.labels.append(label)

      self.splitter.SetSashPosition(maxLabel+2)
      self.maxy = y
      self.maxLabel = maxLabel
      self.__onSize(None)

  # Reset the values displayed based on the values on the object
  def resetValues(self, object):
    if object != self.object:
      return 
    for i in range(len(self.fields)):
      xkey = self.rowList[i].replace(':','__')
      field = self.fields[i]
      # Set the initial value of the cells to the current property values
      if hasattr(object, xkey):
        field.SetValue(object.__dict__[xkey])
      else:
        field.Clear()

  def getPageText(self):
    return self.object._type[2:]

  def __onSize(self, event):
    x, y = self.GetClientSizeTuple()
    my = max(y, self.maxy)
    self.splitter.SetSize((x, my ))
    self.splitter.SetSashPosition(self.maxLabel+2)
    self.SetVirtualSize((x, my))
    if my > y:
      self.SetScrollRate(0,10)
    fw = self.fieldPanel.GetSize().x - 4
    for field in self.fields:
      field.SetSize((fw, field.GetSize().y))

  # If user presses enter, create an ObjectModified event
  def __enterPressed(self, event):
    if event.GetKeyCode() == WXK_RETURN:
      self.__valueModified(event)
    event.Skip()

  # Notify system that the object has been modified
  def __valueModified(self, event):
    field = event.GetEventObject()
    attr = self.rowList[self.fields.index(field)]
    value = field.GetValue()

    xkey = attr.replace(':','__')
    try:
      try:
        ov = self.object.__dict__[xkey]
      except KeyError:
        ov = None

      oldVal = {attr: ov}

      # We only care if user actually changed the current value...
      if ov != value and not (ov == None and value in (0,'')):
        # If value is None, then user basically selected "No value"
        if value != None:
          # Typecast the value, then store in the object's dictionary
          value = self.attributes[attr]['Typecast'](value)
        else:
          # No value... try to set value as "default" value
          try:
            value = self.attributes[attr]['Default']
          except KeyError:
            # Otherwise leave as None
            pass

        self.object.__dict__[xkey] = value
        newVal = {attr: value}

        self.editor.dispatchEvent('ObjectModified',
                         object=self.object,
                         originator=__name__,
                         old=oldVal,
                         new=newVal)

    except ValueError:
      wxBell()

    event.Skip()


#
# Property Editor derived from an attribute's ValueSet
#
class LimitedSetEditor(wxComboBox):
  def __init__(self, parent, attributes):
    wxComboBox.__init__(self, parent, -1,
       style=wxTE_PROCESS_ENTER|wxCB_DROPDOWN|wxCB_READONLY)
    self.attributes = attributes
    self.selectionList  = []

    try:
      default = self.attributes['Default']
    except KeyError:
      default = None


    if not (attributes.has_key('Required') and attributes['Required'] ):
      self.selectionList.append(None)
      self.Append('')

    # Sort the attribute names...
    keys = attributes['ValueSet'].keys()
    keys.sort()

    # .. but make the default always come first
    if default:
      keys.remove(default)
      keys.insert(0, default)

    # Add attributes to the combo box
    for v in keys:
      self.selectionList.append(v)
      try:
        text = attributes['ValueSet'][v]['Label']
      except KeyError:
        text = v
      self.Append("%s%s" % (text, v == default and '*' or '' ))

  def GetValue(self):
    return self.selectionList[self.GetSelection()]

  def SetValue(self, value):
    self.SetSelection(self.selectionList.index(value))


#
# Property Editor whose selections are linked to lists of other objects
#
class LinkedTextEditor(wxComboBox):
  def __init__(self, parent, attributes, objectList, attr):
    wxComboBox.__init__(self, parent, -1,
       style=wxTE_PROCESS_ENTER|wxCB_DROPDOWN|wxCB_SORT)
    self.attributes = attributes
    self.objectList = objectList
    self.attr = attr
    self.updateList()
    objectList.addListener(self.updateList)

  def Destroy(self):
    self.objectList.removeListener(self.updateList)
    wxComboBox.Destroy(self)

  def GetValue(self):
    return wxComboBox.GetValue(self) or None

  def SetValue(self, value):
    if value in self.list:
      self.SetSelection(self.list.index(value))
    elif value is None:
      wxComboBox.SetValue(self, "")
    else:
      wxComboBox.SetValue(self, value)

  def updateList(self):
    try:
      self.Clear()
      self.list = []
      if not (self.attributes.has_key('Required') and self.attributes['Required'] ):
        self.list.append(None)
        self.Append('')
      for v in self.objectList:
        self.list.append(v[self.attr])
        self.Append("%s" % v[self.attr])
    except wxPyDeadObjectError:
      # TODO: why are we getting this exception?
      # TODO: Destroy() is being explicitly called. :(
      pass

#
# Property editor for boolean types
#
class BoolEditor(wxComboBox):
  def __init__(self, parent, attributes):
    wxComboBox.__init__(self, parent, -1,
       style=wxTE_PROCESS_ENTER|wxCB_DROPDOWN|wxCB_READONLY|wxCB_SORT)
    self.attributes = attributes

    if not (self.attributes.has_key('Required') and self.attributes['Required'] ):
      self.Append('')

    try:
      default = self.attributes['Default']
    except KeyError:
      default = None

    self.true = _('Yes')  + (default and '*' or '')
    self.false = _('No')  + (not default and '*' or '')

    self.Append(self.true)
    self.Append(self.false)


  def GetValue(self):
    v = wxComboBox.GetValue(self)
    if v == '':
      return None
    else:
      return v == self.true

  def SetValue(self, value):
    if value == None:
      wxComboBox.SetValue(self,'')
    elif value:
      wxComboBox.SetValue(self,self.true)
    else:
      wxComboBox.SetValue(self,self.false)

#
#
#
class IntEditor(wxSpinCtrl):
  def __init__(self, parent, attributes):
    wxSpinCtrl.__init__(self, parent, -1, style=wxTE_PROCESS_ENTER)
    self.attributes = attributes

#
#
#
class TextEditor(wxTextCtrl):
  def __init__(self, parent, attributes):
    wxTextCtrl.__init__(self, parent, -1, style=wxTE_PROCESS_ENTER)
    self.attributes = attributes

  def SetValue(self, value):
    if value == None:
      value = ""
    wxTextCtrl.SetValue(self, value)
