#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2004 Free Software Foundation
#
# FILE:
# Introspection.py
#
# DESCRIPTION:
# Implements a basic schema template using GNUe's database introspection
#
# NOTES:
# While functional, the primary purpose of this wizard is
# as a "learning-by-example" tool.


from gnue.designer.schema.TemplateSupport import *
import string

# NOTE: It is VERY important that in any references to a
#    "<insert type here> wizard", the word "wizard" must be
#    in lower case, as many proper names such as "Foobar Wizard"
#    have been trademarked. (No, I'm not kidding :)
#


class SchemaIntrospectionTemplate(SchemaTemplate):


  # The first step in our wizard.
  # The template parser will initially
  # call GetStep(FIRST_STEP).
  FIRST_STEP = '0'


  ###############
  #
  # Initialize any runtime variables
  #
  def Start(self, root, current):
    self.root = root



  ###############
  #
  # Return the markup for a specific page
  #
  def GetStep(self, stepCode):

    #
    # Step #1 / Get Title, et al
    #
    if stepCode == '0':
      return   { 'title': 'Basic Form Information',
                 'content': (WizardText('Welcome to the introspection schema wizard.'),
                             WizardText('To create your schema, I need to know some basic information.\n\n'
                                        'First, what shall I call this schema? This name is for '
                                        'informational use only.'),
                             WizardInput('title', label='Schema Name:', required=1,
                                         size=40),
                             WizardText('What connection should we use to connect to the database for introspection?'),
                             WizardInput('connection',label='Connection:', required=1,
                                         set=self.GetAvailableConnections()),
                             WizardText('You may be asked to login to this connection.'),
                            ),
                 'prev': None,
                 'next': '1' }


    #
    # Step #2 / Get Tables to Include
    #
    elif stepCode == '1':

      return   { 'title': 'Select Tables to Include',
                 'content': (WizardText('Which tables shall I include in your schema?'),
                             WizardInput('tables', label='Tables:', required=1,
                                         maxSelections=-1, orderable=1,
                                         set=self.GetAvailableSources(self.variables['connection']))),
                 'prev': '1',
                 'next': None }



  ###############
  #
  # Verify contents of current step
  # Return None if no problems, otherwise
  # return a tuple of error message strings
  #
  def ValidateStep(self, stepCode):

    # The Simple wizard uses basic "required"
    # settings in the page markup, so doesn't
    # need any special validation.
    return None



  ###############
  #
  # We have all the data, so generate our schema. This
  # is called after the user clicks the "Finish" button.
  # No more user input is allowed at this point.
  #
  def Finalize(self):

    # Set the basic attributes of the schema
    self.ModifyElement(self.root, title=self.variables['title'])

    conn = self.variables['connection']

    # Sort the table names to create a prettier file.
    tablesorted = self.variables['tables']
    tablesorted.sort()


    # Create the tables and fields...
    print "Adding <tables>"
    tables = self.root.findChildOfType('GSTables') #self.AddElement('tables', self.root, {})



    # Create each table
    for tablename in tablesorted:
      print "Adding <table> %s" % tablename
      schema = self.GetSourceSchema(conn, tablename)
      table = self.AddElement( 'table', tables,
                               { 'name': tablename } )

      try:
        primarykeys = schema.primarykey
      except:
        primarykeys = []

      fields = self.AddElement('fields', table, {})

      # Add each field
      for field in schema.fields():
        print "  Adding <field> %s" % field.name
        attrs = { 'name': field.name,
                  'type': field.datatype }

        # Add length if appropriate for this datatype
        try:
          if field.datatype != 'date':
            attrs['length'] = int(field.length)
        except AttributeError:
          if field.datatype == 'number':
            attrs['length'] = 12
          elif field.datatype == 'text':
            attrs['length'] = 200

        # ... precision
        try:
          attrs['precision'] = int(field.precision)
        except AttributeError:
          pass

        # Add length if appropriate for this datatype
        try:
          attrs['nullable'] = not field.required
        except AttributeError:
          attrs['nullable'] = 1

        # ... default values?
        if hasattr(field,'defaulttype'):
          if field.defaulttype == 'serial':
            attrs['defaultwith'] = 'serial'
            if field.name in primarykeys and field.datatype == 'number':
              attrs['type'] = 'key'
          elif field.defaulttype == 'sequence':
            attrs['defaultwith'] = 'sequence'
            attrs['default'] = field.defaultval
          elif field.defaulttype == 'constant':
            attrs['defaultwith'] = 'constant'
            attrs['default'] = field.defaultval



        # Create the field element
        self.AddElement('field', fields, attrs)

      # Primary keys
      if primarykeys:
        pk = self.AddElement ('primarykey', table, {})
        pk.name='pk_%s' % tablename
        for field in primarykeys:
          self.AddElement('pkfield', pk, {'name': field})

      # Add misc tags
      self.AddElement ('constraints', table, {})
      self.AddElement ('indexes', table, {})


    # That's it... we're done.
    return 1



############
#
# Basic information about this template
#
TemplateInformation = {
    'Product': 'schema',
    'BaseID' : 'Introspection',
    'BaseClass' : SchemaIntrospectionTemplate,
    'Name' : 'Introspection-based Schema Builder',
    'Description' : 'Creates a schema definition based on an existing database.',
    'Version' : '0.0.1',
    'Author' : 'The GNUe Designer Team',
    'Behavior': WIZARD
}

