#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2005 Free Software Foundation
#
# FILE:
# Workspace.py
#
# DESCRIPTION:
#
# NOTES:
#

__all__ = ['Workspace']

import sys, os
from wxPython.wx import *
from gnue.common.apps import GDebug
from gnue.forms import GFObjects
from gnue.forms.uidrivers.wx import UIdriver as UIwxpython
from gnue.designer.base.PopupMenu import PageMenu
from gnue.designer.base.TemplateParser import TemplateParser
from gnue.designer.base.ToolBase import *
from gnue.common.events import Event

# My support files
from Utils import *
from LayoutEditorTools import LayoutEditorTools
from PropertyBar import PropertyBar
from DisplayDropTarget import DisplayDropTarget
from ReorderFocus import ReorderFocus

from renderers.native.WidgetHandler import WidgetHandler
from renderers.Base.GridPane import GridPane

panelColor = wxWHITE

class Workspace(wxPanel):

  def __init__(self, parent, layouteditor, page):

    self.scroller = scroller = wxScrolledWindow(parent, -1,
       pos=wxPoint(0,0),
       style=wxCLIP_CHILDREN|wxHSCROLL|wxVSCROLL|wxSUNKEN_BORDER)

    wxPanel.__init__(self, scroller, -1)

    EVT_SIZE(self.scroller, self.__OnGridSize)
    self.SetBackgroundColour(panelColor)


    self.layouteditor = layouteditor
    self.page = page

    # TODO: blah.... historical reasons
    self._instance = self.instance = layouteditor.instance

    self._app = self.instance._app

    self.block = None
    self.lastBlock = None
    self.blockMap = {}
    self.panel = None
    self._notebookMap = {}
    self.positionMappings = {}
    self.widgetList = []
    self._currentSelection = []

    # Create our own WX GFUserInterface instance
    self.uidriver = uidriver = UIwxpython.GFUserInterface(self.instance, 0)

    # Create a list of all UI widgets
    self.widgets = {}
    self.widgets.update(uidriver._supportedWidgets)

    # But we don't want a page to act like normal objects...
    # we have special plans for it (bwahahaha)
    del self.widgets['GFPage']


    self.notebook = wxNotebook(self, -1, pos=wxPoint(4, 4), size=wxSize(32,32))
    self.backcolor = wxWHITE


    self._currentObject = None
    self.layouteditor.mode = 'move'
    self.reorderfocus = ReorderFocus(self.instance)

    # Internal mouse states... used by OnMotion, etc
    self.__drawing = 0
    self.__x = 0
    self.__y = 0
    self.__ox = 0
    self.__oy = 0

    # EventAware provided by ToolBase
##    self.registerEventListeners({
##                       'ObjectSelected'      : self.onSetCurrentObject,
##                       'ObjectCreated'       : self.onCreateObject,
##                       'ObjectModified'      : self.onModifyObject,
##                       'ObjectDeleted'       : self.onDeleteObject,
##                       'LayoutEditor:ZoomIn' : self.zoomIn,
##                       'LayoutEditor:ZoomOut': self.zoomOut,
##                       'LayoutEditor:FocusOrder': self.beginFocusOrder,
##                       'Cancel:LayoutEditor:FocusOrder': self.endFocusOrder,
##                       'LayoutEditor:Prepositioning': self.beginPrePositioningTemplate,
##                       'Cancel:LayoutEditor:Prepositioning': self.cancelPrePositioningTemplate,
##                       'LayoutEditor:Select': self.beginSelectMode,
##                       'Cancel:LayoutEditor:Select': self.cancelSelectMode,
##                       'EndWizard': self.endWizard,
##                      })


    ## Stuff needed by UIwxpython
    self._pageList = []  # Needed by UIwxpython

    # Create our grid
    self.panel = GridPane(self.layouteditor, self, wxPoint(10,12))

    # Automatically resize workspace
    EVT_SIZE(self.panel, self.__OnGridSize)

    EVT_CHAR(self.panel, self.keyTrap)
    EVT_LEFT_DOWN(self.panel, self.OnLeftDown)
    EVT_MOTION(self.panel, self.OnMotion)
    EVT_LEFT_UP(self.panel, self.OnLeftUp)
    EVT_RIGHT_UP(self.panel, self.OnRightUp)


    # We are a drop target for Drag-and-Drop
    self.panel.SetDropTarget(DisplayDropTarget(self))
#    self.panel.SetDropTarget(SchemaDropTarget(self))


    self.panelColor = self.panel.GetBackgroundColour()
    self.panelGridColor = wxColour(min(255,self.panelColor.Red()+16),
                 min(255,self.panelColor.Green()+16),
                 min(255,self.panelColor.Blue()+16))

    page.__pointSize = UIwxpython.getPointSize()
    self.calcGridSpacing()



    # Draw the page
    self.drawPage()


  # Recalculate the scrollbar sizes
  def __OnGridSize(self, event):
    w, h = self.panel.GetSizeTuple()
    w2, h2 = self.scroller.GetClientSizeTuple()
    w = int(max(w, w2)*1.1+30)
    h = int(max(h, h2)*1.1+30)
    self.SetSize((w,h))
    self.scroller.SetScrollbars(w/30-1,h/30-1,30,30)

  def _setFeedback(self):

    object = self._currentObject

    ft = ""

    # This sets the feedback text at the top of the Layout Editor window
    if object:
      try:      ft += 'Name: % (%s)' % (object.name, object._type[2:])
      except:   pass

      try:      ft += '   Col: %s' % object.Char__x
      except:   pass

      try:      ft += '   Row: %s' % object.Char__y
      except:   pass

      try:      ft += '   Width: %s' % object.Char__width
      except:   pass

      try:
        if object.Char__height > 1:
          ft += '   Height: %s' % object.Char__height
      except:
        pass

    if ft:
     self.layouteditor.setFeedback(ft)


  def _setCurrentPage(self, object):
    if not object:
      return
    page = isinstance(object, GFObjects.GFPage) and object or \
           object.findParentOfType('GFPage')

    if page != None and page != self.page:
      self.page = page

      self.notebook.SetSelection(self._notebookMap[page])
      try:
        self.workspace = page.__workspace
        self.panel = self.page.__panel
        UIwxpython.setPointSize(self.page.__pointSize)
        self.calcGridSpacing()
      except AttributeError:
        # This simply means we are setting up
        # our notebook for the first time.
        pass


  def _setSelection(self, focus, objects):

    # Unhighlight any previously selected items
    # that are no longer selected.
    for key in self._currentSelection:
      if key not in objects:
        try:
          key._widgetHandler.setSelected(0)
        except AttributeError:
          pass

    # and now highlight any new items
    for object in objects:
      if hasattr(object, '_widgetHandler') and object not in self._currentSelection:
        self._currentSelection.append(object)
        object._widgetHandler.setSelected(1, object == focus)

    self._currentSelection = objects[:]


  def onCreateObject (self, event):
    object = event.object
    handler = event.originator
##    self._currentSelection = []
    if object == None:
      return

    if handler != "Forms::LayoutEditor":
      origpage = self.page

      if object._type != 'GFPage':
        page = object.findParentOfType('GFPage')
        if page:
          self._setCurrentPage(page)
          item = self.layouteditor.renderer.addWidget(object)

      self.inventoryObject(object)
      self._setCurrentPage(origpage)
    if object._type == 'GFBlock':
      self.blockMap[object.name.lower()] = object
      self._rebuildBlockCombo()


  def onModifyObject (self, event):
    object = event.object
    handler = event.originator

    # TODO: adjust label width to match new length...this shouldn't be here
    if object._type == 'GFLabel':
      mods = []
      for field in event.new.keys():
        value = event.new[field]
        if field=='text':
          if (not hasattr(object,'alignment')) or ( object.alignment == 'left'):
            if not hasattr(object,'Char__width') or object.Char__width != len(value):
              object.Char__width=len(value)
              # TODO: This should probably trigger a new ObjectModified event
              event.new['Char:width'] = object.Char__width

##    self.propBar.onModifyObject(object, event)

    if handler != "Forms::LayoutEditor":
      if object._type == 'GFPage':
        self.notebook.SetPageText(self._notebookMap[object],object.name)
        self.notebook.SetSelection(self.notebook.GetSelection())
      else:
        page = object.findParentOfType('GFPage')
        if page:
          self._setCurrentPage(page)

      if object in (self.layouteditor.rootObject, self.page, self.layouteditor.rootObject._layout) or object in self.widgetList:
        self.refreshPage(self.page)

    if object._type == 'GFBlock' and event.old.has_key('name'):
      del self.blockMap[event.old['name'].lower()]
      self.blockMap[object.name.lower()] = object
      self._rebuildBlockCombo()

    self._setFeedback()


  def onDeleteObject (self, event):

    object = event.object
    handler = event.originator

    self._currentObject = None

    if 1: ## handler != "Forms::LayoutEditor":
      if object._type == 'GFPage':
        index = self._notebookMap[object]
        for i in range(index+1, len(self._notebookMap.keys())/2):
          self._notebookMap[i-1] = self._notebookMap[i]
        del self._notebookMap[int(len(self._notebookMap.keys())/2)-1]
        del self._notebookMap[object]
        self.notebook.DeletePage(index)

    if hasattr(object, '_widgetHandler') and object._widgetHandler != None:
      if object in self._currentSelection:
        object._widgetHandler.setSelected(0)
        del self._currentSelection[self._currentSelection.index(object)]

      self.widgetList.remove(object)
      ##for i in range(len(self.widgetList)):
      ##  if self.widgetList[i] == object:
      ##    self.widgetList.pop(i)
      ##    break

##      object._widget.cleanup(object)
      object._widget._object = None
      object._widget = None
      object._widgetHandler.Destroy()
      object._widgetHandler = None


  def refreshPage(self, page):
    try:
      page.__panel.Destroy()
      del page.__panel
    except AttributeError:
      pass

    self.drawPage()


  def drawPage(self):
    self._currentSelection = []
    page = self.page



  def calcGridSpacing(self):
      UIwxpython.initFont(self.panel)

      maxWidth, maxHeight, maxDescent, maxLeading = [0,0,0,0]

      checkchars = string.letters+string.digits+"-\|" + string.punctuation
      for letter in checkchars:
        width,height,descent,leading = self.panel.GetFullTextExtent(letter)
        maxWidth = maxWidth > width and maxWidth or width
        maxHeight = maxHeight > height and maxHeight or height
        maxDescent = maxDescent > descent and maxDescent or descent
        maxLeading = maxLeading > leading and maxLeading or leading

      self.charWidth = maxWidth+maxLeading
      self.charHeight = maxHeight+maxDescent

      self.borderPercentage = (float(gConfig('borderPercentage', section='designer')) / 100)
      self.textPercentage = (float(gConfig('textPercentage', section='designer')) / 100)

      self.widgetWidth = int(self.charWidth * self.borderPercentage)
      self.widgetHeight = int(self.charHeight * self.borderPercentage) + 3
      self.textWidth = self.charWidth * self.textPercentage
      self.textHeight = self.charHeight * self.textPercentage

      self.gridWidth = self.widgetWidth
      self.gridHeight = self.widgetHeight

      self.menu_sb_space=0 # the extra spaces needed by the menu, toolbar and statusbar

      width = self.layouteditor.rootObject._layout.Char__width
      height = self.layouteditor.rootObject._layout.Char__height
      self.panel.SetClientSize(wxSize(int(width)*self.widgetWidth,
                       int(height)*self.widgetHeight))
      self.panel.Refresh()



  def OnLeftDown(self, event):
    x, y = event.GetPositionTuple()
    self.handleLeftDown(x,y)
    event.Skip()


  def handleLeftDown(self, x, y):
    self.__drawing = 1

    # Save starting coordinates for cross-selection box
    self.__cx = cx = int(x / self.gridWidth)
    self.__cy = cy = int(y / self.gridHeight)

    self.layouteditor.setFeedback ('Col: %s   Row: %s' % (cx, cy))

    self.__brush = wxBrush(wxWHITE, style=wxTRANSPARENT)

    self.__x = x
    self.__y = y
    self.__ox = None
    self.__oy = None


  # Used by the dragging routines to draw a selection box
  def xorBox(self, x1, y1, x2, y2, x3=None, y3=None):
    dc = wxClientDC(self.panel)
    dc.SetBrush(self.__brush)
    dc.SetPen(wxGREY_PEN)
    dc.SetLogicalFunction(wxXOR)

    if x3 != None:
      dc.DrawRectangle(x1,y1,x3-x1,y3-y1)

    dc.DrawRectangle(x1,y1,x2-x1,y2-y1)


  def OnMotion(self, event):
    # draw cross-selection box
    if self.__drawing and event.LeftIsDown():

      if self.__drawing == 1:
        self.panel.SetCursor(wxCROSS_CURSOR)
        self.__drawing = 2

      x, y = event.GetPositionTuple()
      cx = int(x / self.gridWidth)
      cy = int(y / self.gridHeight)

      cx1, cx2 = order(self.__cx, cx)
      cy1, cy2 = order(self.__cy, cy)

      self.layouteditor.setFeedback('Col: %s   Row: %s   Width: %s   Height: %s'% (cx1, cy1, cx2 - cx1 + 1, cy2 - cy1 + 1))



      self.xorBox(self.__x, self.__y, x, y, self.__ox, self.__oy)

      self.__ox = x
      self.__oy = y

    event.Skip()


  def OnLeftUp(self, event):
    x, y = event.GetPositionTuple()
    self.handleLeftUp(x,y)
    event.Skip()


  def handleLeftUp(self, x, y):
    selection = []

    self.layouteditor.setFeedback('')

    if self.__drawing and self.__ox is not None:

      self.xorBox(self.__x, self.__y, self.__ox, self.__oy)
      self.__ox = None
      self.__oy = None

    self.__drawing = 0

    # Set x1,y1 to be smaller coordinates, and x2,y2 to be larger
    x1, x2 = order(int(self.__x / self.gridWidth), int(x / self.gridWidth))
    y1, y2 = order(int(self.__y / self.gridHeight),int(y / self.gridHeight))


    self.panel.SetCursor(wxSTANDARD_CURSOR)
    wxSetCursor(wxSTANDARD_CURSOR)


    areaSelected = (x1 <> x2 or y1 <> y2)

    if self.layouteditor.mode == 'move':
      if areaSelected:
        # We are selecting an area
        self.page.walk(self.selectWidgetInArea, x1, y1, x2, y2, selection)
        try:
          object = selection[-1]
        except IndexError:
          object = self.page
        self.layouteditor.dispatchEvent('ObjectSelected',
                         object = object,
                         originator = "Forms::LayoutEditor",
                         selection = selection)
      else:
        self.layouteditor.dispatchEvent('ObjectSelected',
                         object = self.page,
                         originator = "Forms::LayoutEditor")

    elif self.layouteditor.mode == 'positioning':

       if areaSelected:
         width = x2 - x1 + 1
         height = y2 - y1 + 1
       else:
         width = None
         height = None

       PrepositioningTimer(self.endPrePositioningTemplate,x1,y1,width,height).Start(100,1)



  def selectWidgetInArea(self, object, x1, y1, x2, y2, selection):
    try:
      if x1 <= object.Char__x <= x2 and \
         y1 <= object.Char__y <= y2:
        if object not in selection:
          selection.append(object)
    except AttributeError:
      # Not all child objects have coordinates (x,y)
      pass


  def OnRightUp(self, event):
##    self.toolbar.resetTool(self.layouteditor.mode)
    self.layouteditor.mode = 'move'
    self.layouteditor.dispatchEvent('LayoutEditor:Select')

    x, y = event.GetPositionTuple()

    x = int(x / self.gridWidth)
    y = int(y / self.gridHeight)

    menu = wxMenu('Layout Editor')
    form = self._currentObject._type == 'GFForm' and self._currentObject or \
       self._currentObject.findParentOfType('GFForm')
    page = self._currentObject._type == 'GFPage' and self._currentObject or \
       self._currentObject.findParentOfType('GFPage')
    block = self._currentObject._type == 'GFBlock' and self._currentObject or \
       self._currentObject.findParentOfType('GFBlock')

    menu.AppendMenu(wxNewId(), 'Form', PageMenu(self.instance, form, x, y))
    if page:
      menu.AppendMenu(wxNewId(), page.name, PageMenu(self.instance, page, x, y))
    if block:
      menu.AppendMenu(wxNewId(), block.name, PageMenu(self.instance, block, x, y))

    self.panel.PopupMenu(menu, event.GetPosition())


  def keyTrap(self, event):
    if event.KeyCode() in (WXK_LEFT, WXK_RIGHT, WXK_UP, WXK_DOWN) and \
      len(self._currentSelection):
      if event.AltDown() or event.ControlDown() or event.ShiftDown():
        # caution: event.MetaDown() is always true on some architectures
        resize = None
        if event.KeyCode() == WXK_LEFT:
          resize = (-1,0)
        if event.KeyCode() == WXK_RIGHT:
          resize = (1,0)
        if event.KeyCode() == WXK_UP:
          resize = (0,-1)
        if event.KeyCode() == WXK_DOWN:
          resize = (0,1)
        if resize:
          self.layouteditor.dispatchEvent('BeginUndoGroup')
          for widget in self._currentSelection:
            widget._widgetHandler.relativeResize(*resize)
          self.layouteditor.dispatchEvent('EndUndoGroup')
      else:
        pos = None
        if event.KeyCode() == WXK_LEFT:
          pos = (-1,0)
        if event.KeyCode() == WXK_RIGHT:
          pos = (1,0)
        if event.KeyCode() == WXK_UP:
          pos = (0,-1)
        if event.KeyCode() == WXK_DOWN:
          pos = (0,1)

        if pos:
          self.layouteditor.dispatchEvent('BeginUndoGroup')
          for widget in self._currentSelection:
            if widget._type != 'GFPage':
              widget._widgetHandler.relativeMove(*pos)
          self.layouteditor.dispatchEvent('EndUndoGroup')

    elif event.KeyCode() == WXK_DELETE and \
         len(self._currentSelection):
      self.layouteditor.dispatchEvent('BeginUndoGroup')
      for object in self._currentSelection:
        if object._type != 'GFPage':
          self.layouteditor.dispatchEvent('ObjectDeleted', object=object,
                             originator=self)
      self.layouteditor.dispatchEvent('EndUndoGroup')
    elif event.KeyCode() == WXK_TAB:
      if event.ShiftDown():
        object = self._currentObject
        if hasattr(object,'_widgetHandler') and not object._type == 'GFPage':
          i = object._parent._children.index(object)
          lst = object._parent._children[i:] + object._parent._children[:i]
        else:
          lst = self.page._children[:]
        lst.reverse()
        for newobj in lst:
          if hasattr(newobj,'_widgetHandler'):
            self.instance.dispatchEvent('ObjectSelected',
                                        originator=None,
                                        object=newobj)
            break
      elif not event.ControlDown() and not event.AltDown():
        object = self._currentObject
        if hasattr(object,'_widgetHandler') and not object._type == 'GFPage':
          i = object._parent._children.index(object)
          lst = object._parent._children[i+1:] + object._parent._children[:i+1]
        else:
          lst = self.page._children[:]
        for newobj in lst:
          if hasattr(newobj,'_widgetHandler'):
            self.instance.dispatchEvent('ObjectSelected',
                                        originator=None,
                                        object=newobj)
            break


  def zoomIn(self, event):
    size = UIwxpython.getPointSize()
    if size < 72:
      size = size + 1 # int(size * 1.05 + .5)
      UIwxpython.setPointSize(size)
      self.calcGridSpacing()
      self.refreshPage(self.page)
      self.layouteditor.setFeedback(_('Adjusting base point size to %spt') % size)
    else:
      self.layouteditor.setFeedback(_('Cannot adjust point size to more than 72pt'))


  def zoomOut(self, event):
    size = UIwxpython.getPointSize()
    if size > 6:
      size = size - 1 # int(size * -1.05)
      UIwxpython.setPointSize(size)
      self.calcGridSpacing()
      self.refreshPage(self.page)
      self.layouteditor.setFeedback(_('Adjusting base point size to %spt') % size)
    else:
      self.layouteditor.setFeedback(_('Cannot adjust point size to less than 6pt'))


  def beginPrePositioningTemplate(self, event):
    mode = self.layouteditor.mode
    self.layouteditor.mode = 'positioning'
    if mode == 'move':
      self.layouteditor.dispatchEvent('Cancel:LayoutEditor:Select')
    elif mode == 'refocus':
      self.layouteditor.dispatchEvent('Cancel:LayoutEditor:FocusOrder')

    ##self.panel.SetCursor(wxCROSS_CURSOR)
    wxSetCursor(wxCROSS_CURSOR)
    self.__wizardrunner = event.wizardrunner

  def cancelPrePositioningTemplate(self, event=None):
    if self.layouteditor.mode == 'prepositioning':
      self.__wizardrunner.cancel()
    wxSetCursor(wxSTANDARD_CURSOR)

  def endPrePositioningTemplate(self, x, y, width=None, height=None):
    self.layouteditor.mode = 'move'
    self.layouteditor.dispatchEvent('Cancel:LayoutEditor:Prepositioning')
    self.layouteditor.dispatchEvent('LayoutEditor:Select')
    self.__wizardrunner.positioned_run(x=x, y=y, width=width, height=height)


  def beginFocusOrder(self, event):
    mode = self.layouteditor.mode
    self.layouteditor.mode = 'refocus'
    if mode == 'move':
      event.dispatchAfter('Cancel:LayoutEditor:Select')
    elif mode == 'positioning':
      self.__wizardrunner.cancel()
      # Yes, you are reading this right...
      # I'm triggering the exact event that
      # triggered this method. Blame it on
      # the Tuesday night atmospheric crack
      # levels.
      self.layouteditor.dispatchEvent('LayoutEditor:FocusOrder')
      return

    self.reorderfocus.start(self.page)

  def endFocusOrder(self, event):
    self.reorderfocus.end()
    if self.layouteditor.mode == 'refocus':
      self.layouteditor.mode = 'move'
      self.layouteditor.dispatchEvent('LayoutEditor:Select')

  def beginSelectMode(self, event=None):
    mode = self.layouteditor.mode
    self.layouteditor.mode = 'move'
    if mode == 'refocus':
      self.layouteditor.dispatchEvent('Cancel:LayoutEditor:FocusOrder')
    elif mode == 'positioning':
      self.layouteditor.dispatchEvent('Cancel:LayoutEditor:Prepositioning')

  def cancelSelectMode(self, event=None):
    if self.layouteditor.mode == 'move':
      event.dispatchAfter('LayoutEditor:Select')

  def endWizard(self, event=None):
    mode = self.layouteditor.mode
    self.layouteditor.mode = 'move'
    if mode == 'refocus':
      self.layouteditor.dispatchEvent('Cancel:LayoutEditor:FocusOrder')
    elif mode == 'positioning':
      self.layouteditor.dispatchEvent('Cancel:LayoutEditor:Prepositioning')
    self.layouteditor.dispatchEvent('LayoutEditor:Select')


class PrepositioningTimer(wxTimer):
  def __init__(self, method, *args, **params):
    self.__method = method
    self.__args = args
    self.__params = params
    wxTimer.__init__(self)

  def Notify(self):
    self.__method(*self.__args, **self.__params)

