#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2002-2005 Free Software Foundation
#
# FILE:
# schema/LayoutEditor.py
#
# DESCRIPTION:
# The visual editor for Designer
#
# NOTES:
#

from wxPython.wx import *
from VisualTable import *
from gnue.common.schema.Objects import *
from gnue.designer.base.ToolBase import *

class VisualEditor (ToolBase):

  runtime_section = 'SchemaDiaEditor'

  def init(self):
    # basic initialisation
    self.object = None
    self._app = self.instance._app

    self.dragmode = false
    self.tables=[]
    self.hasNewObj = false
    self.rootObject.walk(self.inventoryObject)

    # Painting related stuff
    self.diagram=wxScrolledWindow(self, -1,
                                  style=wxSUNKEN_BORDER)
    self.panelColor = self.GetBackgroundColour()
    self.panelheight=100
    self.panelwidth=200
    self.diagram.EnableScrolling(true, true)
    self.redrawBackground()
    #self.diagram.SetBackgroundColour(wxWHITE)
    self.calculateTablePositions()
    Sizer = wxBoxSizer(wxHORIZONTAL)
    #Sizer.Add(self.toolPalette, 0)
    Sizer.Add(self.diagram, 1, wxEXPAND)

    self.SetAutoLayout(true)
    self.SetSizer(Sizer)


    # EventAware provided by ToolBase
    self.registerEventListeners({
                       'ObjectSelected'      : self.onSetCurrentObject,
                       'ObjectCreated'       : self.onCreateObject,
                       'ObjectModified'      : self.onModifyObject,
                       'ObjectDeleted'       : self.onDeleteObject,
                      })



    # WX WINDOWS Events
    EVT_PAINT(self.diagram, self.onPaintEvent)
#    EVT_LEFT_DOWN(self.diagram, self.onLeftClickEvent)
    EVT_MOTION(self.diagram, self.onMouseEvent)
    EVT_LEFT_UP(self.diagram, self.onMouseEvent)
    EVT_LEFT_DOWN(self.diagram, self.onMouseEvent)
    EVT_LEFT_DCLICK(self.diagram, self.onDoubleClickEvent)
    EVT_RIGHT_DOWN(self.diagram, self.onRightClickEvent)


  def calculateTablePositions(self):
      dc = wxMemoryDC()
      self.diagram.PrepareDC(dc)
      dc.BeginDrawing()
      x=10
      y=10
      maxy=10
      maxx=10
      for vTbl in self.tables:
        (w,h)=vTbl.calculateDimensions(dc)
        vTbl.moveTo(x,y)
        x=x+10+w
        if y+h>maxy:
          maxy=y+h
        if x>maxx:
          maxx=x
        if x>600:
          x=10
          y=maxy+10
        
      dc.EndDrawing()
      self.resizeCanvas(maxx,maxy)

      # Rules for better positioning:
      #   tables with most connections in the middle
      #   ...


  def inventoryObject(self, object):
    if object._type=="GSTable":
      self.tables.append(VisualTable(object))

  
  def onSetCurrentObject (self, event):
    object = event.object
    handler = event.originator
    if object == None:
      return
    if object != self.object:
      self.setCurrent(object)  

  def setCurrent(self, object):
    # mark actual object (in blue?)
    if self.object!=object:
      self.object=object
      dc = wxClientDC(self.diagram)
      self.diagram.PrepareDC(dc)
      self.drawWholeScreen(dc)

  def findFreePosition(self, x,y,w,h):
    conflict=true
    while conflict:
      conflict=false
      for i in self.tables:
        if i.isInRectangle(x,y,x+w,y+h):
          x=i.xpos+i.width+10
          if x>500:
            x=10
            y=i.ypos+i.height+10
          conflict=true
    self.resizeCanvas(x+w,y+h)
    return (x,y)


  def resizeCanvas(self,xmax,ymax):
    # resize canvas
    if xmax>self.panelwidth:
      self.panelwidth=xmax+10
      self.diagram.SetScrollbars(20, 20, self.panelwidth / 20,
                                 self.panelheight / 20)
    if ymax>self.panelheight:
      self.panelheight=ymax+10
      self.diagram.SetScrollbars(20, 20, self.panelwidth / 20,
                                self.panelheight / 20)     


  def onCreateObject (self, event):
    object = event.object
    handler = event.originator
    if object == None:
      return
    if handler != __name__:
      try:
        GDebug.printMesg(4,"Adding %s (%s)" % (object.name, object._type))
      except:
        if object._type=="GSTable":
          self.tables.append(VisualTable(object))
          self.hasNewObj=true
          self.refresh()      
          return
          vTbl=VisualTable(object)       
          dc = wxClientDC(self.diagram)
          self.diagram.PrepareDC(dc)  
          (w,h)=vTbl.calculateDimensions(dc)

          # position new Table
          (x,y)=self.findFreePosition(10,10,w,h)
          vTbl.moveTo(x,y)
      
          self.tables.append(vTbl)
          
        if object._type=="GSField":
          self.hasNewObj=true
          self.onModifyObject(event)
      

  def onModifyObject (self, event):
    self.refresh()
    return
    tbl=object.findParentOfType('GSTable')
    if tbl!=None:
      for i in self.tables:
        if i.instance==tbl:
          dc = wxClientDC(self.diagram)
          self.diagram.PrepareDC(dc)  
          (w,h)=i.calculateDimensions(dc)
          # check if still freepos
          (x,y)=self.findFreePosition(i.xpos,i.ypos,w,h)
          i.moveTo(x,y)
          i.drawMe(dc,self.object)
          return

  def refresh(self):
    self.diagram.Refresh(true)   

  def onDeleteObject (self, event):
    object = event.object
    handler = event.originator
    if object._type=="GSTable":
      # search for GSTable and remove the VisualTable assigned to it
      for vTbl in self.tables:
        if vTbl.instance==object:
          self.tables.remove(vTbl)
          self.refresh()

    if object._type=="GSField":
      # search for GSTable and update corresponding vtable
      tbl=object.findParentOfType("GSTable")
      for vTbl in self.tables:
        if vTbl.instance==tbl:
          dc = wxMemoryDC()
          vTbl.calculateDimensions(dc)
          self.refresh()
                 
  def onPaintEvent(self, event, clear=false):
      if self.hasNewObj:
        self.calculateTablePositions()
        self.hasNewObj=false
      dc = wxPaintDC(self.diagram)
      self.diagram.PrepareDC(dc)
      if clear:
        dc.SetBackground(self.panelColor)
        dc.Clear()
      self.drawWholeScreen(dc)
      
  def drawWholeScreen(self,dc):
      dc.BeginDrawing()
      #dc.Blit(0,0,self.panelheight,self.panelwidth,self.imagedc,0,0)
      #dc.EndDrawing()
      #return
      #dc.BeginDrawing()
      for vTbl in self.tables:
        vTbl.drawMe(dc,self.object)           
      dc.EndDrawing()

  def redrawBackground(self):
    btm=wxEmptyBitmap(self.panelheight,self.panelwidth,-1)
    dc=wxMemoryDC()
    dc.SelectObject(btm);
    for vTbl in self.tables:
        vTbl.drawMe(dc,self.object)
    self.imagedc=dc

  def calculateMousePos(self,event):
    originX, originY = self.diagram.GetViewStart()
    unitX, unitY = self.diagram.GetScrollPixelsPerUnit()
    return (event.GetX() + (originX * unitX),
            event.GetY() + (originY * unitY))


  def getObjectAt(self,x,y):
    self.tables.reverse()
    for vTbl in self.tables:
      object=vTbl.getObjectAt(x,y)
      if object != None:
        self.tables.reverse()        
        return object
    self.tables.reverse()
    # in case, that no object is selected, choose root object
    return self.rootObject

  def onMouseEvent(self,event):
    if not (event.LeftDown() or event.Dragging() or event.LeftUp()):
      return # Ignore mouse movement without click/drag.
    if event.LeftDown():
      (x,y) = self.calculateMousePos(event)   
      object=self.getObjectAt(x,y)
      if object._type!='VisualTable':
        self.dispatchEvent('ObjectSelected', object=object,\
                           originator=__name__)        
        return
      else:
        self.dragx=x-object.xpos
        self.dragy=y-object.ypos
        self.dragobj=object
        dc = wxClientDC(self.diagram)
        self.diagram.PrepareDC(dc)
        self.spriteOn=false
        self.dragmode=true

    elif event.LeftUp() and self.dragmode:
      self.dragmode=false
      self.refresh()
      
    elif event.Dragging() and self.dragmode:
      (x,y) = self.calculateMousePos(event)   
      dc = wxClientDC(self.diagram)
      self.diagram.PrepareDC(dc)
      dc.BeginDrawing()
      dc.SetLogicalFunction(wxXOR)
      if self.spriteOn:
        self.dragobj.drawMe(dc,self.object,true)
        
      self.dragobj.moveTo(x-self.dragx,y-self.dragy)
      self.spriteOn=true
      self.dragobj.drawMe(dc,self.object,true)

#     TODO: implement a way to move tables as a whole and not
#     only showing the border of the window.
#     one way would be to use a wxMemoryDC to paint into, and
#     which is copied into the real window -> no flickering screen
#     when a object should be moved, first delete it, and repaint the
#     wxMemoryDC, then you can move it on the real DC and you can
#     restore the background by using Blit

      dc.EndDrawing()

  # in case of a right mouse click show a context menu
  def onRightClickEvent(self,event):    
    (x,y) = self.calculateMousePos(event)
    object=self.getObjectAt(x,y)
    menu = wxMenu()

    popup_NEWTABLE    =1
    popup_NEWSTRFIELD =2
    popup_NEWNUMFIELD =3
    popup_NEWINDEX    =4
    popup_NEWUINDEX   =5
    popup_EDIT        =6
    popup_RENAME      =7
    popup_DUPPL       =8
    popup_DEL         =9

    # TODO: Use a generalized popup menu like in the forms designer
    #       instead of this one. the popup menu in the treeview and
    #       the visual editor should look the same

    if object._type=='VisualTable':
      object=object.instance

    self.dispatchEvent('ObjectSelected', object=object, originator=__name__)  

    if object._type=='VisualTable' or object._type=="GSTable":
      menu.Append(popup_NEWSTRFIELD, "Add Field (String)")
      EVT_MENU(self,popup_NEWSTRFIELD,self.onAddStringField)
      menu.Append(popup_NEWNUMFIELD, "Add Field (Number)")
      EVT_MENU(self,popup_NEWNUMFIELD,self.onAddNumberField)
      
    elif object._type=="GSField":
      menu.Append(popup_NEWINDEX, "Add Index")      
      EVT_MENU(self,popup_NEWINDEX,self.onAddIndex)

      menu.Append(popup_NEWUINDEX, "Add Unique Index")      
      EVT_MENU(self,popup_NEWUINDEX,self.onAddUniqueIndex)
      
    else: 
      menu.Append(popup_NEWTABLE, "New Table")
      EVT_MENU(self,popup_NEWTABLE,self.onNewTable)

    if object._type!='GSSchema':
      menu.AppendSeparator()

      menu.Append(popup_EDIT, "Edit")
      EVT_MENU(self,popup_EDIT,self.onEdit)

      menu.Append(popup_RENAME, "Rename")
      EVT_MENU(self,popup_RENAME,self.onRename)

      menu.Append(popup_DUPPL, "Duplicate")
      EVT_MENU(self,popup_DUPPL,self.onDuplicate)

      menu.Append(popup_DEL, "Delete")
      EVT_MENU(self,popup_DEL,self.onDelete)


    clickPt = wxPoint(x + self.diagram.GetPosition().x,
                        y + self.diagram.GetPosition().y)
    self.diagram.PopupMenu(menu, clickPt)
    menu.Destroy()

  def onNewTable(self,event):
    dialog = wxTextEntryDialog(self.diagram,'Name of the new table:')

    if dialog.ShowModal() == wxID_OK:
      tables=self.rootObject.findChildOfType("GSTables")
      if tables==None:
        tables = self.instance.incubator.createObject(
                self.rootObject, 'tables', self.rootObject)
      self.instance.incubator.createObject(self.rootObject, 'table',
                   tables, {"name":dialog.GetValue()})

    dialog.Destroy()

  def onAddNumberField(self,event):
    self.onAddField(event,'number')

  def onAddStringField(self,event):
    self.onAddField(event,'string')
    
  def onAddField(self,event,type='string'):
    dialog = wxTextEntryDialog(self.diagram,'Name of the new field:')

    if dialog.ShowModal() == wxID_OK:
      fields=self.object.findChildOfType("GSFields")
      if fields==None:
        tbl=self.object.findChildOfType("GSTable")
        fields=self.instance.incubator.createObject(
             self.rootObject,'fields', tbl)

      params = {"name":dialog.GetValue()}
      if type=='string':
        params.update({"type":"string","size":20})
      else:
        params.update({"type":type,"size":4})
        
      self.instance.incubator.createObject(self.rootObject, 'field',
                   fields,params)

    dialog.Destroy()
   
  def onAddIndex(self,event):
    tbl=self.object.findParentOfType("GSTable")          
    indexes=tbl.findChildOfType("GSIndexes")          
    if indexes==None:
      indexes = self.instance.incubator.createObject(
          self.rootObject, 'indexes', tbl)
    idx=self.instance.incubator.createObject(
          self.rootObject, 'index', indexes,
                 {"name":"%s_%s" % (tbl.name,self.object.name)})
    self.instance.incubator.createObject(self.rootObject, 'index', idx,
                 {"name":self.object.name})
    self.refresh()

  def onAddUniqueIndex(self,event):
    tbl=self.object.findParentOfType("GSTable")
    indexes=tbl.findChildOfType("GSIndexes")
    if indexes==None:
      indexes = self.instance.incubator.createObject(
           self.rootObject, 'indexes',tbl)
    idx=self.instance.incubator.createObject(
             self.rootObject, 'index', indexes,
                 {"name":"%s_%s" % (tbl.name,self.object.name),
                  "unique":"Y"})
    self.instance.incubator.createObject(self.rootObject, 'index', idx,
                 {"name":self.object.name})        
    self.refresh()


  def onAddConstraint(self,event):
    pass

  def onRename(self,event):
    dialog = wxTextEntryDialog(self.diagram,'New Name')
    if hasattr(self.object,'name'):
      dialog.SetValue(self.object.name)
      
    if dialog.ShowModal() == wxID_OK:
      self.object.name=dialog.GetValue()
      self.instance.onModifyObject(self.object,"schema",
                                    [('name',self.object.name)])
      self.refresh()

    dialog.Destroy()
  
  def onDuplicate(self,event):
    pass

  def onEdit(self,event):
    print "EDIT occured"
    pass
  
  def onDelete(self,event):
    self.instance.incubator.deleteObject(self.rootObject, self.object, firstRun=1)


  def onLeftClickEvent(self,event):    
    (x,y) = self.calculateMousePos(event)   
    for vTbl in self.tables:
      object=vTbl.getObjectAt(x,y)
      if object != None:
        self.dispatchEvent('ObjectSelected', object=object, originator=__name__)


  def onDoubleClickEvent(self,event):    
    (x,y) = self.calculateMousePos(event)   
    object=self.getObjectAt(x,y)
    if object != None and object._type!='VisualTable': 
      # begin to edit this object    
      self.dispatchEvent('ObjectSelected', object=object, originator=__name__)
      # TODO: add edit functions here
