#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2005 Free Software Foundation
#
# FILE:
# TemplateBase.py
#
# DESCRIPTION:
# Provides a base class for templates
#
# NOTES:
#
# Guidelines for writing templates:
#   Never, ever, EVER overwrite the __init__ method.
#   However, you can overwrite the Start method which
#   is called when your template should do its trick.
#   Anything normally occurring in an __init__ can
#   occur at the beginning of Start.
#
#   You can be guaranteed to have an empty <form|report>
#   object passed to the Start method.  If you successfully
#   created your object, you should return a TRUE value in
#   GetFinal.  Otherwise, return false.  If a TRUE value is
#   returned, Designer assumes that the passed <Form|Report>
#   object now contains a valid object.
#
# NOTE: It is VERY important that in any reference to a
#   "<insert type here> wizard", the word "wizard" must be
#   in lower case, as many "Xyz Wizards" (proper name) have
#   been trademarked. (No, I'm not kidding :)

from gnue.common.formatting import GTypecast
from gnue.common.definitions.GParserHelpers import GContent

#
# These Identify the types of templates. A template
# should identify itself as one of the following in
# the TemplateInformation hash's "Behavior" attribute:
#

TEMPLATE=0      # This is a template that does not prompt
                # for user input.  Designer only calls
                # the Start() and GetFinal() methods.

WIZARD=1        # This is a template that prompts the
                # user for input in a wizard fashion.
                # Designer calls Start(), then calls
                # GetStep() to get a step. After a user
                # clicks next, Designer calls ValidateStep()
                # and if this returns true, calls GetStep()
                # if there are more steps to complete,
                # or, if no more steps, calls GetFinal().


#
# This is a base template class.  No templates should
# directly inherit this class, but one of the decendent
# classes (i.e., FormTemplate, ReportTemplate, etc)
#
class TemplateBase:

  ##########################################
  #
  #  The following few methods should be
  #  subclassed by the template developer.

  # The first step in our wizard
  FIRST_STEP = '0'

  # Stores all the variables
  # entered by the wizard user.
  variables = {}

  def Start(self, object, current={}):
    return 0


  # Verify contents of current step
  # Return None if no problems, otherwise
  # return a tuple of error message strings
  def ValidateStep(self, stepCode):
    return None


  # Get a step
  def GetStep(self, stepCode):
    return None


  # We have all the data, so generate our object
  def Finalize(self):
    return 1


  ##########################################
  #
  #  The following methods are convenience
  #  methods.

  def AddElement(self, tag, parent, attributes={}, content=""):
    instance = self.parser.instance
    if content:
      attr = {'_content_': content}
    else:
      attr = {}
    attr.update(attributes)
    return instance.incubator.createObject(instance.rootObject,
                        tag, parent, attributes=attr, select=0)


  def SetContent(self, element, content):
    i = 0
    while i < len(element._children):
      if element._children[i]._type == '_content_':
        element._children.pop(i)
      else:
        i += 1
    content = GContent(element,content)


  def ModifyElement(self, element, **modifications):
    oldMods = {}
    for attr in modifications.keys():
      try:
        oldMods[attr] = element.__dict__[attr]
      except KeyError:
        oldMods[attr] = None

    element.__dict__.update(modifications)

    self.parser.instance.dispatchEvent('ObjectModified',
        object=element, originator=__name__,
        new=modifications, old=oldMods)


  def GetAvailableConnections(self):
    rv = []
    p = self._connections.getAllConnectionParameters()
    for conn in p.keys():
      if p[conn].has_key('comment'):
        rv.append((conn, '%s [%s]' % (p[conn]['comment'], conn)))
      else:
        rv.append((conn, '%s [%s]' % (conn, p[conn]['provider'])))
    return rv


  def GetTemporaryConnection(self, connection_name):
    
    return self._connections.getConnection(connection_name,1)


  def GetAvailableSources (self, connection_name):

    if not self.__cachedSources.has_key (connection_name):
      self.__cachedSources [connection_name] = tables = \
        self.GetTemporaryConnection (connection_name).readSchema ()

    result = []

    if tables is not None:
      for table in tables.findChildrenOfType ('GSTable', False, True):
        result.append ((table.name, hasattr (table, 'description') and
                                    table.description or table.name))
    return result


  def GetSourceSchema (self, connection_name, source_name):

    if not connection_name in self.__cachedSources:
      self.__cachedSources [connection_name] = \
          self.GetTemporaryConnection (connection_name).readSchema ()

    schema = self.__cachedSources [connection_name]
    if schema is not None:
      for table in schema.findChildrenOfType ('GSTable', False, True):
        if table.name.lower () == source_name.lower ():
          return table
    else:
      return None


  def GetAvailableFields (self, connection_name, source_name):
    result = []
    source = self.GetSourceSchema (connection_name, source_name)
    if source is not None:
      for field in source.findChildrenOfType ('GSField', False, True):
        result.append ((field.name, hasattr (field, 'description') \
                      and field.description or field.name))

    return result


  def GetUniqueName(self, name, limitingObject=None):
    return self.instance.getUniqueName(name, limitingObject)


  ##########################################
  #
  #  The following methods are of no concern
  #  to the wizard/template developer and
  #  may/will be changed later on.

  def __init__(self, parser):
    self.parser = parser
    self.instance = parser.instance
    self._connections = parser.instance._app.connections
    self.__cachedConnections= {}
    self.__cachedSources= {}



###########################################################
#
# Wizard elements
#
###########################################################

# Available field types
boolean = GTypecast.boolean
text = GTypecast.text
number = GTypecast.number
whole = GTypecast.whole
integer = GTypecast.integer
name = GTypecast.name


class WizardItem:
  pass

class WizardText (WizardItem):
  def __init__(self, text):
    self.text = text

class WizardInput (WizardItem):
  def __init__(self, variable, label, typecast=text,
               set=None, size=10, lowerbound=None, upperbound=None,
               forceupper=0, forcelower=0, required=0, lines=1,
               maxSelections=1, orderable=0):

    self.variable = variable
    self.label = label
    self.typecast = typecast
    self.set = set
    self.size = size
    self.lowerbound = lowerbound
    self.upperbound = upperbound
    self.forceupper = forceupper
    self.required = required
    self.forcelower = forcelower
    self.lines = lines

    # The following only have an effect is set != None
    self.maxSelections = maxSelections
    self.orderable = orderable


# A wizard can raise this if not enough information
# is provided or if the root object is not in a
# sufficient state.
class InsufficientInformation(StandardError):
  pass
