# GNU Enterprise Forms - Forms Instance Management
#
# Copyright 2001-2005 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: GFInstance.py 7811 2005-08-09 18:30:12Z reinhard $

"""
GFInstance manages forms instances in a 1:N relationship. It sits between the
UI and the form and passes events between the two in a semi-intelligent
manner.
"""

# NOTES:
#  Once all the events are moved back in here
#    make the next/prec methods more generic in the GFForm
#    change self._form in here to a list

import os
import sys
import dircache

from gnue.forms.GFForm import *
from gnue.forms.GFParser import loadFile
from gnue.forms import VERSION
from gnue.forms import GFKeyMapper
from gnue.common.apps import i18n, errors
from gnue.common.datasources import Exceptions, GConnections
from gnue.common.datasources.GDataSource import getAppserverResource
from gnue.common import events
from gnue.common.utils.FileUtils import dyn_import
from gnue.common.utils.FileUtils import openResource, openBuffer
from gnue.common.logic.language import AbortRequest
from gnue.common.apps.GClientApp import *


# =============================================================================
# Exceptions
# =============================================================================

class FileOpenError (StartupError):
  def __init__ (self, message):
    msg = u_("Unable to open file: %s") % message
    StartupError.__init__ (self, msg)



# =============================================================================
# Forms instance manager class
# =============================================================================

class GFInstance (events.EventAware):

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, manager, connections, ui, disableSplash = False,
      parameters = {}, parentContainer = None, moduleName = None):
    # moduleName is here only for Designer to be able to pass it in
    #when debugging a form.
    
    gEnter (4)

    # Configure event handling
    gDebug (4, "Initializing event handling")

    self.eventController = events.EventController ()
    events.EventAware.__init__ (self, self.eventController)
    self.registerEventListeners ({
                           # First, all events are passed to the focus widget
                           '__before__'          : self.__beforeEvent,

                           # Focus-related events
                           'requestENTER'        : self.nextEntry,
                           'requestNEXTENTRY'    : self.nextEntry,
                           'requestPREVENTRY'    : self.previousEntry,
                           'requestNEXTPAGE'     : self.nextPage,
                           'requestPREVPAGE'     : self.previousPage,
                           'requestNEXTBLOCK'    : self.nextBlock,
                           'requestPREVBLOCK'    : self.previousBlock,
                           'requestFIRSTRECORD'  : self.firstRecord,
                           'requestLASTRECORD'   : self.lastRecord,
                           'requestPREVRECORD'   : self.prevRecord,
                           'requestNEXTRECORD'   : self.nextRecord,
                           'requestRECORDNUMBER' : self.jumpToRecord,
                           'requestJUMPPROMPT'   : self.requestJumpTo,
                           'requestJUMPRECORD'   : self.jumpRecords,
                           'requestJUMPROWSUP'   : self.jumpRowsUp,
                           'requestJUMPROWSDOWN' : self.jumpRowsDown,
                           'requestPAGE'         : self.gotoPage,
                           'requestFOCUS'        : self.changeFocus,

                           # Data set manipulation
                           'requestNEWRECORD'    : self.newRecord,
                           'requestMARKFORDELETE': self.deleteRecord,
                           'requestCOMMIT'       : self.executeCommit,
                           'requestPRINTOUT'     : self.executePrintout,
                           'requestROLLBACK'     : self.executeRollback,
                           'requestENTERQUERY'   : self.requestQuery,
                           'requestEXECQUERY'    : self.executeQuery,
                           'requestCANCELQUERY'  : self.cancelQuery,
                           'requestCOPYQUERY'    : self.copyQuery,

                           # Miscellaneous stuff
                           'requestEXIT'         : self.executeExit,
                           'requestABOUT'        : self.executeAbout,
                           'requestMODETOGGLE'   : self.toggleInsertMode,

                           'fireTRIGGER'         : self.fireTrigger,
                           'buttonActivated'     : self.fireButton,

##                            # Dialog Support
##                            'requestDIALOG'        : self.activateDialog,
                     })

    self.connections = connections       # Link to the GBaseApp's GConnections
    self.manager = manager               # Link to the GBaseApp Instance that
                                         #   created this GFInstance
    self._uimodule = ui                  # The UI created in the GBaseApp
    self._disableSplash = disableSplash  # Disable splashscreen
    self._parameters = parameters        # The parameters passed to the
                                         # GBaseApp instance
    self._formsDictionary = {}           # A dictionary containing all the
                                         # forms loaded from a file

    self._parentContainer = parentContainer

    # Load user customized key mappings
    options = gConfigDict()
    mapping = {}

    for key in options.keys ():
      if string.lower (key) [:4] == 'key_':
        mapping [key [4:]] = options [key]

    GFKeyMapper.KeyMapper.loadUserKeyMap (mapping)

    # Construct an instance of the UI driver if an UI module is available
    if self._uimodule is not None:
      gDebug (4, "Initializing user interface driver")
      self._uiinstance = self._uimodule.GFUserInterface (self.eventController,
                                       disableSplash   = self._disableSplash,
                                       parentContainer = self._parentContainer,
                                       moduleName = moduleName)

    gLeave (4)


  # ---------------------------------------------------------------------------
  # Add the basic dialogs
  # ---------------------------------------------------------------------------

#   def addDialogs (self):
#     """
#     Loads the base dialogs into memory.
# 
#     Base dialogs include items such as the jump to record
#     dialog and error dialogs.
#     """
# 
#     # Import and register dialogs
#     basedir = os.path.dirname (sys.modules [self.__module__].__file__)
#     basedir = os.path.join (basedir, 'dialogs')
# 
#     for dialogName in dircache.listdir (basedir):
# 
#       if dialogName [0] == '_' or dialogName == '.svn':
#         continue
# 
#       filename    = os.path.join (basedir, dialogName)
#       (name, ext) = os.path.splitext (dialogName)
# 
#       if os.path.isfile (filename) and ext == '.py':
#         dialog = dyn_import ('gnue.forms.dialogs.%s' % name)
# 
#       elif os.path.isdir (filename):
#         dialog = dyn_import ('gnue.forms.dialogs.%s' % dialogName)
# 
#       else:
#         continue
# 
#       try:
#         self.addFormFromBuffer (dialog.buildForm ())
# 
#       except StandardError, mesg:
#         print "WARNING: Cannot build %s form \n%s" % (dialogName, mesg)


  # ---------------------------------------------------------------------------
  # Load a form from a buffer
  # ---------------------------------------------------------------------------

  def addFormFromBuffer (self, buffer):
    """
    Loads a GObj based form tree when passed a string containing gfd markup.

    @param buffer: A string containing the forms definition (gfd)
    """

    try:
      fileHandle = openBuffer (buffer)
      self.addFormFromFilehandle (fileHandle)
      fileHandle.close ()

    except IOError:
      raise FileOpenError, errors.getException () [2]


  # ---------------------------------------------------------------------------
  # Load a form from a file specified by it's filename
  # ---------------------------------------------------------------------------

  def addFormFromFile (self, fileName):
    """
    Loads a GObj based form tree when passed a file name.

    @param fileName: A string containing a URI
    """

    try:
      if fileName [:12] == 'appserver://':
        # (w, h) = self._uimodule.getMaxFormSize ()
        param = {'language': i18n.language, 'formwidth': 80, 'formheight': 20}
        fileHandle = getAppserverResource (fileName, param, self.connections)
      else:
        fileHandle = openResource (fileName)

      self.addFormFromFilehandle (fileHandle, fileName)
      fileHandle.close ()

    except IOError:
      raise FileOpenError, errors.getException () [2]


  # ---------------------------------------------------------------------------
  # Add a form from a file-like object specified by it's filehandle
  # ---------------------------------------------------------------------------

  def addFormFromFilehandle (self, fileHandle, url = None):
    """
    Loads a GObj based form tree when passed a valid python file handle.

    A copy of the instance is passed into the parser so that it can work with
    things like the GConnections stored in the base app.

    @param fileHandle: A python file handle.
    """

    # Load the file bypassing the initialization We bypass the initialization
    # because <dialog>s are really <form>s and they don't like being children
    # of another form
    form = loadFile (fileHandle, self, initialize = 0, url = url)

    # Extract the child <dialog>s from the main form tree
    self.reapSubforms (form)

    # Add the main form into the dictionary
    self._formsDictionary [form.name] = form


  # ---------------------------------------------------------------------------
  # Remove sub-forms from the main tree
  # ---------------------------------------------------------------------------

  def reapSubforms (self, formTree):

    for child in formTree._children [:]:
      if isinstance (child, GFForm):
        child.setParent (None)
        self._formsDictionary [child.name] = child
        formTree._children.remove (child)


  # ---------------------------------------------------------------------------
  # Initialize the UI and activate the main form
  # ---------------------------------------------------------------------------

  def activate (self):

    gEnter (4)

    # Initialize all the forms loaded into memory
    gDebug (4, "Initializing form objects")

    for formObject in self._formsDictionary.values ():
      formObject.phaseInit ()

    gDebug (4, "Initializing user interface driver object")
    self._uiinstance.initialize ()

    # Build the UIs for all the forms
    gDebug (4, "Building forms")
    for dialog in self._formsDictionary.keys ():
      self.buildForm (dialog)

    # Bring up the main form
    form = self._formsDictionary ['__main__']

    if form.style == 'dialog':
      # we assume a dialog to be modal by definition and that program execution
      # stops here until the dialog get's closed. So do *not* enter another
      # main loop
      gDebug (4, "Activating main form as dialog")
      self.activateForm ('__main__', modal = True)

    else:
      gDebug (4, "Activating main form")
      self.activateForm ('__main__')

      gDebug (4, "Startup complete")
      gDebug (4, "Entering main loop")
      self._uiinstance.mainLoop ()
      gDebug (4, "Returning from main loop")

    gLeave (4)


  # ---------------------------------------------------------------------------
  # Activate a given form
  # ---------------------------------------------------------------------------

  def activateForm (self, formName = '__main__', parameters = {}, modal = 0):

    gEnter (4)

    form = self._formsDictionary [formName]

    if len (parameters.keys ()):
      form._parameters = parameters

    if not form._currentEntry:
      raise errors.ApplicationError, \
         u_("There are no navigable widgets in this form. Unable to display.")

    form.processTrigger ('On-Activation')

    # First set the current entry to be focused and up to date before
    # displaying the real UI form. 
    form.refreshDisplay (form)
    self.updateStatus (form)

    self.dispatchEvent ('gotoENTRY', object = form._currentEntry, _form = form)
    self._uiinstance.activateForm (formName, modal)

    gLeave (4)


  # ---------------------------------------------------------------------------
  # Build a user interface for a form tree
  # ---------------------------------------------------------------------------

  def buildForm (self, formName = '__main__'):

    form = self._formsDictionary [formName]
    self._uiinstance.buildForm (form, formName)


  # ===========================================================================
  # UI FUNCTIONS
  # ===========================================================================

  # ---------------------------------------------------------------------------
  # Update Statusbar: Insert Mode flag
  # ---------------------------------------------------------------------------

  def updateInsertMode (self, form):

    self.updateStatusBar (insertMode = form._insertMode, form = form)


  # ---------------------------------------------------------------------------
  # Update Statusbar: Record counter
  # ---------------------------------------------------------------------------

  def updateRecordCounter (self, form):

    self.updateStatusBar (currentRecord = form._currentBlock._currentRecord +1,
                      maxRecord = form._currentBlock._recordCount, form = form)


  # ---------------------------------------------------------------------------
  # Update Statusbar: Page counter
  # ---------------------------------------------------------------------------

  def updatePageCounter (self, form):

    maxPages = len (form._layout._pageList)
    count = form._layout._pageList.index (form._currentPage)
    self.updateStatusBar (currentPage = count + 1, maxPage = maxPages,
                          form = form)


  # ---------------------------------------------------------------------------
  # Update Statusbar: Tip
  # ---------------------------------------------------------------------------

  def updateTip (self, form):

    tip = ''
    if form._currentEntry:
      if form._currentEntry.getOption ('tip'):
        tip = form._currentEntry.getOption ('tip')
    self.updateStatusBar (tip = tip, form = form)


  # ---------------------------------------------------------------------------
  # Update Statusbar: Record status
  # ---------------------------------------------------------------------------

  def updateRecordStatus(self, form):

    if form._currentBlock._resultSet.current.isDeleted ():
      status = 'deleted'
    elif form._currentBlock._resultSet.current.isPending ():
      status = 'modified'
    elif form._currentBlock.mode == 'query':
      status = 'query'
    else:
      status = 'saved'

    self.updateStatusBar (recordStatus = status, form = form)


  # ---------------------------------------------------------------------------
  # Update Statusbar: all fields of the status bar
  # ---------------------------------------------------------------------------

  def updateStatus (self, form):
    self.updateTip (form = form)
    self.updateInsertMode (form = form)
    self.updateRecordCounter (form = form)
    self.updateRecordStatus (form = form)
    self.updatePageCounter (form = form)


  #
  # updateStatusBar
  #
  # generates the event to the UI that
  # tells it to update it's status bar

  # ---------------------------------------------------------------------------
  # Update the status bar
  # ---------------------------------------------------------------------------

  def updateStatusBar (self, tip = None, recordStatus = None,
           insertMode = None, currentRecord = None, maxRecord = None,
           currentPage = None, maxPage = None, form = None):

    if tip is None and hasattr (self, '__lasttip'):
      tip = self.__lasttip

    self.__lasttip = tip
    self.dispatchEvent ('uiUPDATESTATUS',
                        tip           = tip,
                        recordStatus  = recordStatus,
                        insertMode    = insertMode,
                        currentRecord = currentRecord,
                        maxRecord     = maxRecord,
                        currentPage   = currentPage,
                        maxPage       = maxPage,
                        _form         = form)


  # ===========================================================================
  # EVENT FUNCTIONS
  # ===========================================================================
  
  # From here down should be nothing but eventListeners listed in the __init__
  # above.


  # ---------------------------------------------------------------------------
  # Update status bar due to changes in an entry
  # ---------------------------------------------------------------------------

  def _entryUpdated (self, form):
    """
    Common code snipped called when something has changed with an entry and it
    has to update the status bar. Used to cut down on repeated code.
    """
    self.dispatchEvent ('gotoENTRY', object = form._currentEntry, _form = form)
    self.updateRecordCounter (form)
    self.updateRecordStatus (form)


  # ---------------------------------------------------------------------------
  # Handle an event before it's sent to usual event processing
  # ---------------------------------------------------------------------------

  def __beforeEvent (self, event):
    """
    This is called before any normal event processing is done. This method
    passes the event to the current focus widget and sees if that widget wants
    to handle that event.

    @param event: The event currently being processed.
    """

    if hasattr (event, '_form') and event._form._currentEntry:

      # If the display will need to be refreshed, then the proxied event should
      # set this to 1
      event.refreshDisplay = 0

      # Pass off the event to the current entry's event handler
      event._form._currentEntry.subEventHandler.dispatchEvent (event)

      # Refresh entry display if appropriate
      if event.refreshDisplay:
        # FIXME: This is not clean. Maybe the event should set event.__after__?
        if hasattr (event._form._currentEntry, '_displayHandler'):
          event._form._currentEntry._displayHandler.generateRefreshEvent ()

        event._form.refreshUIEvents ()

      # If the entry needs an error message displayed, then the proxied event
      # should set this to the message text
      if event.__errortext__:
        self.displayMessageBox (event.__errortext__, 'Error')


  # ---------------------------------------------------------------------------
  # Move focus to the next entry
  # ---------------------------------------------------------------------------

  def nextEntry (self, event):
    """
    Called whenever an event source has requested that the focus change to the
    next data entry object

    @param event: The event currently being processed.
    """

    if not event._form.endEditing ():
      return

    origEntry = event._form._currentEntry
    event._form._currentEntry.processTrigger ('ON-NEXT-ENTRY',
                                              ignoreAbort = False)

    # If the trigger changed focus, no need in us doing it too...
    if origEntry == event._form._currentEntry:
      message = event._form.nextEntry ()
      if message:
        self.displayMessageBox (message)
        return

      self.dispatchEvent ('gotoENTRY', object = event._form._currentEntry,
                          _form = event._form)

    self.updateRecordStatus (event._form)
    self.updateTip (event._form)


  # ---------------------------------------------------------------------------
  # Move focus to the previous entry
  # ---------------------------------------------------------------------------

  def previousEntry (self, event):
    """
    Called whenever an event source has requested that the focus change to the
    next data entry object

    param event: The event that requested the previous entry.
    """

    if not event._form.endEditing ():
      return

    origEntry = event._form._currentEntry
    event._form._currentEntry.processTrigger ('ON-PREVIOUS-ENTRY',
                                              ignoreAbort = False)

    # If the trigger changed focus, no need in us doing it too...
    if origEntry == event._form._currentEntry:
      event._form.previousEntry ()
      self.dispatchEvent ('gotoENTRY', object = event._form._currentEntry,
                          _form = event._form)

    self.updateRecordStatus (event._form)
    self.updateTip (event._form)


  # ---------------------------------------------------------------------------
  # Move focus to the next block
  # ---------------------------------------------------------------------------

  def nextBlock (self,event):
    """
    Called whenever an event source has requested that the focus change to the
    next data entry block
    """

    if not event._form.endEditing ():
      return

    event._form.nextBlock ()
    self.updateStatus (event._form)


  # ---------------------------------------------------------------------------
  # Move focus to the previous block
  # ---------------------------------------------------------------------------

  def previousBlock (self, event):
    """
    Called whenever an event source has requested that the focus change to the
    next data entry block
    """

    if not event._form.endEditing ():
      return

    event._form.previousBlock ()
    self.updateStatus (event._form)


  # ---------------------------------------------------------------------------
  # Jump to the next page
  # ---------------------------------------------------------------------------

  def nextPage (self, event):
    """
    Called to make the form jump to the next page
    """

    if not event._form.endEditing ():
      return

    currentIndex = event._form._layout._pageList.index(event._form._currentPage)
    if currentIndex == len (event._form._layout._pageList) - 1:
      nextIndex = 0
    else:
      nextIndex = currentIndex + 1

    event._form.findAndChangeFocus (event._form._layout._pageList [nextIndex])
    self.updateStatus (event._form)


  # ---------------------------------------------------------------------------
  # Jump to the previous page
  # ---------------------------------------------------------------------------

  def previousPage (self, event):
    """
    Called to make the form jump to the previous page
    """

    if not event._form.endEditing ():
      return

    pageList = event._form._layout._pageList
    currentIndex = pageList.index (event._form._currentPage)
    event._form.findAndChangeFocus (pageList [currentIndex - 1])
    self.updateStatus (event._form)

  # ---------------------------------------------------------------------------
  # Jump to a specific page
  # ---------------------------------------------------------------------------

  def gotoPage (self, event):
    """
    Jump to a specific page of the form
    """

    if not event._form.endEditing ():
      return

    newpage = event._form._layout._pageList [event.data]
    event._form.findAndChangeFocus (newpage)
    self.updateStatus (event._form)


  # ---------------------------------------------------------------------------
  # Jump to the previous record
  # ---------------------------------------------------------------------------

  def prevRecord (self, event):
    """
    Called whenever an event source has requested that the form moves to the
    previous record in memory
    """

    if not event._form.endEditing ():
      return

    message = event._form.prevRecord ()
    if message:
      self.displayMessageBox (message)
      return

    self._entryUpdated (event._form)


  # ---------------------------------------------------------------------------
  # Jump to the next record
  # ---------------------------------------------------------------------------

  def nextRecord (self, event):
    """
    Called whenever an event source has requested that the form advance to the
    next record in memory
    """

    if not event._form.endEditing ():
      return

    message = event._form.nextRecord ()
    if message:
      self.displayMessageBox (message)
      return

    self._entryUpdated (event._form)


  # ---------------------------------------------------------------------------
  # Jump to the first record
  # ---------------------------------------------------------------------------

  def firstRecord (self, event):
    """
    Called whenever an event source has requested that the form advance to the
    first record in memory
    """

    if not event._form.endEditing ():
      return

    message = event._form.firstRecord ()
    if message:
      self.displayMessageBox (message)
      return

    self._entryUpdated (event._form)


  # ---------------------------------------------------------------------------
  # Jump to the last record
  # ---------------------------------------------------------------------------

  def lastRecord (self, event):
    """
    Called enever an event source has requested that the form advance to the
    last record in memory
    """

    if not event._form.endEditing ():
      return

    message = event._form.lastRecord ()
    if message:
      self.displayMessageBox (message)
      return

    self._entryUpdated (event._form)


  #
  # ---------------------------------------------------------------------------
  # Jump to a specific record
  # ---------------------------------------------------------------------------

  def jumpToRecord (self, event):
    """
    Called whenever an event source has requested that the form move to a
    specific record
    """

    if event._form._currentBlock.mode == 'query':
      event._form.triggerSetStatusText (_('You cannot do that in query mode.'))
      return

    if not event._form.endEditing ():
      return

    try:
      count = abs (int (event.data)) - 1

    except ValueError:
      message = _("Invalid numeric value entered.")

    else:
      message = event._form.jumpRecord (count)

    if message:
      self.displayMessageBox (message, 'Error')
      return

    self._entryUpdated (event._form)


  # ---------------------------------------------------------------------------
  # Jump records forward or backward
  # ---------------------------------------------------------------------------

  def jumpRecords (self, event):
    """
    Jump a specified number of records forward or backward.
    """

    if not event._form.endEditing ():
      return

    # Manipulate event and redirect to jumpToRecord
    rows     = event.data
    curRec   = event._form._currentBlock._currentRecord + 1
    recCount = event._form._currentBlock._recordCount

    newValue   = max (1, curRec + rows)
    event.data = min (recCount, newValue)

    self.jumpToRecord (event)
    self._entryUpdated (event._form)


  # ---------------------------------------------------------------------------
  # Jump a number of records backward
  # ---------------------------------------------------------------------------

  def jumpRowsUp (self, event):

    if not event._form.endEditing ():
      return

    # Manipulate event and redirect to jumpToRecord
    if event._form._currentEntry._rows > 1:
      event.data = -event._form._currentEntry._rows + 1

    elif event._form._currentBlock._rows >1:
      event.data = -event._form._currentBlock._rows + 1

    else:
      event.data = 0

    if event.data:
      self.jumpRecords (event)


  # ---------------------------------------------------------------------------
  # Jump a number of records forward
  # ---------------------------------------------------------------------------

  def jumpRowsDown (self, event):

    if not event._form.endEditing ():
      return

    # Manipulate event and redirect to jumpToRecord
    if event._form._currentEntry._rows > 1:
      event.data = event._form._currentEntry._rows - 1

    elif event._form._currentBlock._rows >1:
      event.data = event._form._currentBlock._rows - 1

    else:
      event.data = 0

    if event.data:
      self.jumpRecords (event)


  # ---------------------------------------------------------------------------
  # Verify state of data and exit form
  # ---------------------------------------------------------------------------

  def executeExit(self, event):
    """
    When exit is requested verify that the data has been saved
    """

    try:
      event._form.processTrigger ('Pre-Exit', ignoreAbort = False)

      # TODO: WTF? This isn't saving a currently edited field
      if event._form._currentBlock.autoCommit and not event._form.isSaved ():
        event._form.endEditing ()
        event._form.commit ()

      if not event._form.isSaved ():
        self.displayMessageBox (_("Data not saved. Save changes or clear "
               "the form to proceed."), 'Error')
        event._form.refreshUIEvents ()
      else:
        event._form.processTrigger ('On-Exit')
        self.dispatchEvent ('exitApplication', _('Current data is saved'),
                           _formName = event._form.name, _form = event._form)

    except AbortRequest, t:
      self.displayMessageBox (t, 'Warning')
      event._form.refreshUIEvents ()


  # ---------------------------------------------------------------------------
  # Mark a record for deletion
  # ---------------------------------------------------------------------------

  def deleteRecord(self, event):
    """
    Tells the form to mark a record for delete
    """

    if event._form.readonly:
      self.dispatchEvent ('formALERT', _('Form is readonly'),
                          _form = event._form)
    elif not event._form._currentBlock.deletable:
      self.dispatchEvent ('formALERT', _('Block does not allow delete'),
                          _form = event._form)
    else:
      event._form.deleteRecord ()
      self.updateRecordStatus (event._form)


  # ---------------------------------------------------------------------------
  # Tells the form to create a new record
  # ---------------------------------------------------------------------------

  def newRecord(self, event):
    """
    Tells the form to create a new record
    """

    if not event._form.endEditing ():
      return

    if event._form.readonly:
      self.dispatchEvent ('formALERT', _('Form is readonly'),
                          _form = event._form)
    elif event._form._currentBlock.editable not in ('Y', 'new'):
      self.dispatchEvent ('formALERT', _('Block does not allow insert'),
                          _form = event._form)
    else:
      event._form.newRecord ()
      self._entryUpdated (event._form)


  # ---------------------------------------------------------------------------
  # Display the about dialog
  # ---------------------------------------------------------------------------

  def executeAbout (self, event):
    parameters = {
      'name'        : event._form.title or "Unknown",
      'formversion' : event._form.getOption ('version') or "Unknown",
      'author'      : event._form.getOption ('author') or "Unknown",
      'description' : event._form.getOption ('description') or "Unknown",
    }

    self._uiinstance.showAbout (**parameters)


  # ---------------------------------------------------------------------------
  # Display a generic message box
  # ---------------------------------------------------------------------------

  def displayMessageBox (self, message = '', kind = 'Info', cancel = False,
      caption = 'GNUe Message', title = None):
    return self._uiinstance.showMessage (message, kind, title, cancel)


  #
  # requestJumpTo
  #
  #
  # ---------------------------------------------------------------------------
  # Display a dialog box prompting for a record number to jump to
  # ---------------------------------------------------------------------------

  def requestJumpTo(self,event):
    """
    Displays a dialog box prompting for record to jump to then jumps to the
    requested record
    """

    fields = [(u_("Recordnumber"), 'record', 'string', None, None, [])]
    result = self._uiinstance.getInput (u_("Jump to record"), fields)

    if result is not None:
      if not event._form.endEditing ():
        return

      try:
        count = abs (int (float (result ['record']))) - 1

      except ValueError:
        message = _("Invalid numeric value entered.")

      else:
        message = event._form.jumpRecord (count)

      if message:
        self.displayMessageBox (message, 'Error')
        return

      self._entryUpdated (event._form)


  # ---------------------------------------------------------------------------
  # Toggle insert mode
  # ---------------------------------------------------------------------------

  def toggleInsertMode (self, event):
    """
    Tells the form to toggle mode
    """

    event._form.toggleInsertMode ()
    self.updateInsertMode (event._form)


  # ---------------------------------------------------------------------------
  # Fire a trigger named 'process-printout' (if it exists)
  # ---------------------------------------------------------------------------

  def executePrintout (self, event):
    """
    If the form has a trigger named process-printout then fire it
    """

    if event._form._triggerDictionary.has_key ('process-printout'):
      event._form.fireTrigger ('process-printout')
    else:
      # TODO: should probably do something else as a default if trigger not
      #       available like a screen print
      self.dispatchEvent ('uiPRINTOUT', _form = event._form)


  # ---------------------------------------------------------------------------
  # Perform a rollback
  # ---------------------------------------------------------------------------

  def executeRollback (self, event):
    """
    Tells the form to rollback everything it contains
    """
    event._form.rollback ()
    self._entryUpdated (event._form)


  # ---------------------------------------------------------------------------
  # Change the focus to an entry object
  # ---------------------------------------------------------------------------

  def changeFocus(self, event):
    """
    Change the focus to the entry object requested by the event source
    """

    if not event._form.endEditing ():
      return False

    if event.data._type in ['GFEntry', 'GFImage']:
      if not event.data.isNavigable (event._form.getCurrentMode ()):
        return False

      newEntry = event.data
      message = event._form.changeFocus (newEntry)
      if message:
        self.displayMessageBox (message)
        return False

      self.dispatchEvent ('gotoENTRY', object = event._form._currentEntry,
                         _form = event._form)

      self.updateRecordStatus (event._form)
      self.updateTip (event._form)

    return True


  # ---------------------------------------------------------------------------
  # Initialize a query
  # ---------------------------------------------------------------------------

  def requestQuery (self, event):

    if not event._form.endEditing ():
      return

    message = event._form.initQuery ()
    if message:
      self.displayMessageBox (message, 'Error')
      return

    event._form.refreshDisplay (event._form)
    self.dispatchEvent ('gotoENTRY', object = event._form._currentEntry,
                       _form = event._form)
    self.updateStatus (event._form)


  # ---------------------------------------------------------------------------
  # Cancel current query
  # ---------------------------------------------------------------------------

  def cancelQuery (self, event):

    if not event._form.endEditing ():
      return

    message = event._form.cancelQuery ()
    if message:
      self.displayMessageBox (message, 'Error')
      return

    event._form.refreshDisplay (event._form)
    self.dispatchEvent ('gotoENTRY', object = event._form._currentEntry,
                       _form = event._form)
    self.updateStatus (event._form)


  # ---------------------------------------------------------------------------
  # Copy the current query
  # ---------------------------------------------------------------------------

  def copyQuery (self, event):
    if not event._form.endEditing ():
      return

    message = event._form.copyQuery ()
    if message:
      self.displayMessageBox (message, 'Error')
      return

    event._form.refreshDisplay (event._form)
    self.dispatchEvent ('gotoENTRY', object = event._form._currentEntry,
                       _form = event._form)
    self.updateStatus (event._form)


  # ---------------------------------------------------------------------------
  # Execute the current query
  # ---------------------------------------------------------------------------

  def executeQuery (self, event):

    if not event._form.endEditing ():
      return

    message = event._form.executeQuery ()
    if message:
      self.displayMessageBox (message, 'Error')

    self.dispatchEvent ('gotoENTRY', object = event._form._currentEntry,
                       _form = event._form)
    event._form.refreshDisplay (event._form._currentBlock)
    self._entryUpdated (event._form)


  # ---------------------------------------------------------------------------
  # Perform a commit
  # ---------------------------------------------------------------------------

  def executeCommit (self, event):

    if not event._form.endEditing ():
      return

    message = event._form.commit ()
    if message:
      self.displayMessageBox (message, 'Error')
      gDebug(4,message)
      return

    self._entryUpdated (event._form)
    event._form.refreshDisplay (event._form._currentBlock)


  # ---------------------------------------------------------------------------
  # Fire a trigger of the form
  # ---------------------------------------------------------------------------

  def fireTrigger (self, event):

    event._form.fireTrigger (event.data)


  # ---------------------------------------------------------------------------
  # Fire the action-trigger bound to a button
  # ---------------------------------------------------------------------------

  def fireButton (self, event):

    event.data.processTrigger ('On-Action', False)
