# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU GPublic
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2005 Free Software Foundation
#
# $Id: UIdriver.py 7659 2005-06-26 16:05:08Z btami $
#
# DESCRIPTION:
# A PyWin32 based user interface driver for GNUe forms.
#
# NOTES:
#

import sys
import string
import types
import math

from gnue.common.apps import i18n
from gnue.forms.uidrivers._base import Exceptions

try:
  import win32gui, win32con, afxres, commctrl, win32clipboard, win32ui
except ImportError:
  raise Exceptions.DriverNotSupported, _("The GNUe-Forms Win32 driver requires PyWin32.")
from pywin.mfc import window

from gnue.common import events
from gnue.common.apps import GDebug
from gnue.common.apps import GConfig
from gnue.common.utils.TextUtils import lineWrap

from gnue.forms import VERSION
from gnue.forms.GFForm import *
from gnue.forms.uidrivers._commonGuiToolkit import UIdriver as commonToolkit

from gnue.forms.uidrivers.win32.GFwin32App import *
#from gnue.forms.uidrivers.wx.UIWXSplashScreen import *
from gnue.forms.uidrivers.win32.widgets._base  import *
from gnue.forms.uidrivers.win32.common import textEncode
from gnue.forms.uidrivers.win32.dialog import BaseDialog
from gnue.forms.uidrivers.win32 import dialogs


EDIT = 0x0081
STATIC = 0x0082
BUTTON = 0x0080


def OnWMVScroll(hwnd, msg, wParam, lParam, widget):
  return widget.OnWMVScroll(hwnd, msg, wParam, lParam)

def OnWMMenuselect(hwnd, msg, wParam, lParam, widget):
  return widget.OnWMMenuselect(hwnd, msg, wParam, lParam)

def OnWMNotify(hwnd, msg, wParam, lParam, widget):
  return widget.OnWMNotify(hwnd, msg, wParam, lParam)

def OnWMDestroy(hwnd, msg, wParam, lParam, widget):
  win32gui.DestroyWindow(hwnd)

def OnWMClose(hwnd, msg, wParam, lParam, widget):
  return widget.OnWMClose(hwnd, msg, wParam, lParam)

def OnWMSize(hwnd, msg, wParam, lParam, widget):
  return widget.OnWMSize(hwnd, msg, wParam, lParam)

def OnWMCommand(hwnd, msg, wParam, lParam, widget):
  return widget.OnWMCommand(hwnd, msg, wParam, lParam)

def OnWMDefault(hwnd, msg, wParam, lParam, widget):
  return win32gui.DefWindowProc(hwnd, msg, wParam, lParam)

#
# GFUserInterface
#
# The public interface to the User Interface
# All UIs must provide this class
#
class GFUserInterface(commonToolkit.GFUserInterface):

  _MBOX_KIND = {'Info'    : {'type'   : _('Info'),
                             'icon'   : win32con.MB_ICONINFORMATION,
                             'buttons': win32con.MB_OK},
                'Warning' : {'type'   : _('Warning'),
                             'icon'   : win32con.MB_ICONWARNING,
                             'buttons': win32con.MB_OK},
                'Question': {'type'   : _('Question'),
                             'icon'   : win32con.MB_ICONQUESTION,
                             'buttons': win32con.MB_YESNO},
                'Error'   : {'type'   : _('Error'),
                             'icon'   : win32con.MB_ICONERROR,
                             'buttons': win32con.MB_OK}}

  _RESPONSE = {win32con.IDOK    : True,
               win32con.IDYES   : True,
               win32con.IDNO    : False,
               win32con.IDCANCEL: None }

  _message_map = { win32con.WM_VSCROLL    : OnWMVScroll,
                   win32con.WM_MENUSELECT : OnWMMenuselect,
                   win32con.WM_NOTIFY     : OnWMNotify,
                   win32con.WM_DESTROY    : OnWMDestroy,
                   win32con.WM_CLOSE      : OnWMClose,
                   win32con.WM_SIZE       : OnWMSize,
                   win32con.WM_COMMAND    : OnWMCommand }

  _wndclass = None

  def _wndproc(self, hwnd, msg, wParam, lParam):
    try:
      widget = self._win32app._HwndToTkObj[hwnd]
    except:
      return win32gui.DefWindowProc(hwnd, msg, wParam, lParam)
    try:
      OnWM = self._message_map.get(msg, OnWMDefault)
      x = OnWM(hwnd, msg, wParam, lParam, widget)
    except:
      x = -1
    return x


  def _initialize(self):

    self._disabledColour = afxres.AFX_IDC_COLOR_LIGHTGRAY
    self._win32app = getWin32App()
    win32gui.InitCommonControls()
    
    try:
      icon = win32gui.LoadImage(0, sys.prefix+'\py.ico', win32con.IMAGE_ICON,
                                          0, 0, win32con.LR_LOADFROMFILE | win32con.LR_DEFAULTSIZE)
    except:
      icon = win32gui.LoadIcon(0, win32con.IDI_APPLICATION)

    if not self._wndclass:
      # Register the "GNUeWindow" class.
      wc = win32gui.WNDCLASS()
      wc.lpszClassName = "GNUeWindow"
      wc.hCursor = win32gui.LoadCursor( 0, win32con.IDC_ARROW )
      wc.hIcon = icon
      wc.hbrBackground = win32con.COLOR_BTNSHADOW #win32con.COLOR_WINDOW
      wc.lpfnWndProc = self._wndproc
      self.__class__._wndclass = win32gui.RegisterClass(wc)

    #
    # SplashScreen
    #
#    if not self._disableSplash:
#      self.splash = UIWXSplashScreen()

    font_name = gConfigForms('faceName')
    if not font_name:
      # explicite boolean check forces generated report parameter dialog
      # to use ANSI_VAR_FONT (GConfig typecast initialization problem...)
      if gConfigForms ('fixedWidthFont') == True:
        fnt = win32con.ANSI_FIXED_FONT
      else:
        fnt = win32con.ANSI_VAR_FONT

      lf = win32gui.GetObject(win32gui.GetStockObject(fnt))
      font_name = lf.lfFaceName

    font_spec = {'name':font_name, 'height':int(gConfigForms('pointSize'))}
    self._font = win32ui.CreateFont(font_spec)


    # Create a dummy window used to compute sizes
    dummyWindow = window.Wnd(win32ui.CreateWnd())
    dc = dummyWindow.GetWindowDC()

    dc.SelectObject(self._font)
    metrics = dc.GetTextMetrics()
    maxWidth = metrics["tmAveCharWidth"]
    maxHeight = metrics["tmHeight"]
    maxDescent = metrics["tmDescent"]
    maxLeading = metrics["tmExternalLeading"]

    self.textWidth    = int(maxWidth+maxLeading)  # The pixel width of text inside a widget
    self.textWidth    = int(math.ceil(self.textWidth*1.1))
    self.textHeight   = int(maxHeight+maxDescent)  # The pixel height of text inside a widget
    self.widgetWidth  = self.textWidth             # The pixel width of a 1 char widget (for things like buttons)
    self.widgetHeight = self.textHeight + 5       # The pixel height of a 1 char widget (for things like buttons)

    dummyWindow.ReleaseDC(dc)

  #############################################################################
  #
  # Private UIBase support functions
  #
  # Called in UIbase functions to perform UI interaction with this specific
  # widget set.
  #

  #
  # _exit
  #
  # Tells the application to close it's main window
  #
  def _exit(self,formName):
    exitApp = 1
    for child in self._children:
      if child._form.name == formName:
        child.mainWindow.Hide()

        if child._form.style == 'dialog':
          win32gui.PostQuitMessage(0) # Terminate the dialog

      exitApp = exitApp and not win32gui.IsWindowVisible(child.mainWindow.GetHwnd())

    if exitApp:
      for child in self._children:
        self._win32app._MainWindowList.remove(child.mainWindow)
        child.mainWindow.Destroy()

    if len(self._win32app._MainWindowList) == 0:
      win32gui.PostQuitMessage(0) # Terminate the app.

    for window in self._win32app._MainWindowList:
      if isinstance(window, Win32Window):
        window.Enable(1)
        win32gui.SetActiveWindow(window.GetHwnd())

  def _beep(self):
    win32gui.MessageBeep(0)

  #############################################################################
  #
  # Incoming Event Processors
  #
  # Processes the incoming events from other objects
  # From here down should be nothing but eventListeners listed

  #
  # mainLoop
  #
  # The primary loop of the user interface.  Called once the UI is
  # fully activated
  #
  def mainLoop(self):
    self._win32app.MainLoop() # simply call the wxApp's MainLoop method

  #
  # formAlert
  #
  # Rings a bell and alters the statusbar to display
  # a line of text
  #
  def formAlert(self, event):
    self._beep()
    ui = self._gfObjToUIWidget[event._form]
    if hasattr (ui, 'statusBar'):
      win32gui.SendMessage(ui.statusBar.GetHwnd(), commctrl.SB_SETTEXT, 0, textEncode(event.data))

  #
  # Called whenever forms goes into a "wait" state in which user cannot
  # interact with interface (e.g., while waiting for a query or a commit)
  #
  def beginWait (self, event):
    win32ui.DoWaitCursor(1)

  #
  # Called whenever forms leaves a "wait" state
  #
  def endWait (self, event):
    win32ui.DoWaitCursor(0)


  #
  # Clipboard routines
  #
  # If a particular UI has a system-wide clipboard,
  # these methods should be overridden to use that
  # clipboard.
  #
  def getClipboardContents(self, event):
    if None == win32clipboard.OpenClipboard():
      success = win32clipboard.GetClipboardData(win32con.CF_TEXT)
      win32clipboard.CloseClipboard()
    else:
      success = 0
      GDebug.printMesg(1,'Unable to open clipboard for read')

    if success:
      value = success
    else:
      GDebug.printMesg(1,'Unable to obtain clipboard contents. Defaulting to Empty.')
      value = None

    GDebug.printMesg(6, "Getting clipboard value '%s'" % value)
    event.__result__ = value


  def setClipboardContents(self, event):
    GDebug.printMesg(6,"Setting clipboard '%s'" % event.text)

    if None == win32clipboard.OpenClipboard():
      win32clipboard.EmptyClipboard()
      handle = win32clipboard.SetClipboardText(event.text)
      GDebug.printMesg(6,"Set Clipboard Data Handle: %s" % handle)
      win32clipboard.CloseClipboard()
    else:
      GDebug.printMesg(6,'Unable to open clipboard for write')


  def setTitle(self, event):
    ui = self._gfObjToUIWidget[event._form]
    try:
      win32gui.SetWindowText(ui.mainWindow.GetHwnd(), event.title)
    except AttributeError:
      pass


  # ---------------------------------------------------------------------------
  # create a modal message box
  # ---------------------------------------------------------------------------

  def _showMessage (self, message, kind = 'Info', title = None, cancel = False):
    """
    This function creates a message box of a given kind and returns True, False
    or None depending on the button pressed.
    @param message: the text of the messagebox
    @param kind: type of the message box. Valid types are 'Info', 'Warning',
        'Question', 'Error'
    @param title: title of the message box
    @param cancel: If True a cancel button will be added to the dialog
    @return: True if the Ok-, Close-, or Yes-button was pressed, False if the
        No-button was pressed or None if the Cancel-button was pressed.
    """
    mbRec  = self._MBOX_KIND.get (kind)
    flags = win32con.MB_TASKMODAL | mbRec['icon'] | mbRec['buttons']

    if title is not None and len (title):
      if isinstance (title, types.StringType):
        title = unicode (title, i18n.encoding)
    else:
      title = mbRec['type']

    cButtons = [win32con.MB_OKCANCEL, win32con.MB_YESNOCANCEL]

    if cancel and not mbRec ['buttons'] in cButtons:
      if mbRec ['buttons'] == win32con.MB_OK:
        flags = flags | win32con.MB_OKCANCEL
      elif mbRec ['buttons'] == win32con.MB_YESNO:
        flags = flags | win32con.MB_YESNOCANCEL

    res = win32gui.MessageBox(0, message, title, flags)

    return self._RESPONSE [res]


  # ---------------------------------------------------------------------------
  # Start an input dialog and return the data record or None if cancelled
  # ---------------------------------------------------------------------------

  def _getInput (self, title, fields, cancel = True):

    dialog = dialogs.InputDialog (title, fields, cancel)
    dialog.DoModal ()
    return dialog.inputData


  # ---------------------------------------------------------------------------
  # Show an exception
  # ---------------------------------------------------------------------------

  def _showException (self, group, name, message, detail):
      
    dialog = ExceptionDisplay (group, name, message, detail)
    dialog.DoModal ()


  # ---------------------------------------------------------------------------
  # Display an about box
  # ---------------------------------------------------------------------------

  def _showAbout (self, name, appversion, formversion, author, description):
    d = dialogs.AboutBox (name, appversion, formversion, author, description)
    d.DoModal ()


# =============================================================================
# This class implements a dialog for displaying exceptions
# =============================================================================

class ExceptionDisplay (BaseDialog):

  _TITLE = {'system'     : _("GNUe Internal System Error"),
            'admin'      : _("GNUe Unexpected Error"),
            'application': _("GNUe Application Error")}

  _FORMAT = {
     'system': u_("An unexpected internal error has occured:\r\n%s.\r\n"
                  "This means you have found a bug in GNU Enterprise. "
                  "Please report it to gnue-dev@gnu.org"),
     'admin': u_("An unexpected error has occured:\r\n%s.\r\n"
                 "Please contact your system administrator."),
     'application': u_("An unexpected error has occured:\r\n%s.\r\n"
                       "Please contact your system administrator.")}

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, group, name, message, detail):

    BaseDialog.__init__(self, self._TITLE.get (group, _('Error')), cancel = False, ok = False)
    self.message_map [win32con.WM_CTLCOLORSTATIC] = self.OnCtlColorStatic

    self.detail = detail.replace('\n','\r\n')
    
    cs = win32con.WS_CHILD | win32con.WS_VISIBLE
    position = (0, 0, 0, 0)

    self.iconID = ID = getNextId()
    s = cs | win32con.SS_ICON #| win32con.WS_BORDER
    self.template.append([STATIC, 'icon', ID, (5,8,32,32), s])

    self.messageID = ID = getNextId()
    s = cs | win32con.SS_LEFT
    self.template.append([STATIC, textEncode(self._FORMAT.get (group, "%s") % message), \
                          ID, position, s])

    s = cs | win32con.WS_TABSTOP | win32con.BS_DEFPUSHBUTTON
    self.template.append([BUTTON, textEncode(u_('Close')), win32con.IDCLOSE, (0, 0, 56, 15), s])

    self.detailsID = ID = getNextId()
    s = cs | win32con.WS_TABSTOP | win32con.BS_PUSHBUTTON
    self.template.append([BUTTON, textEncode(u_('>> Details')), ID, (0, 0, 56, 15), s])

    self.detailID = ID = getNextId()
    s = cs | win32con.ES_MULTILINE | win32con.ES_READONLY | win32con.WS_TABSTOP \
           | win32con.ES_AUTOVSCROLL #| win32con.WS_VSCROLL
    es = win32con.WS_EX_STATICEDGE
    self.template.append([EDIT, textEncode(self.detail), ID, position, s, es])

    self._showsDetail  = False


  def OnInitDialog(self, hwnd, msg, wparam, lparam):
    BaseDialog.OnInitDialog(self, hwnd, msg, wparam, lparam)

    item = win32gui.GetDlgItem(self.hwnd, self.detailID)
    win32gui.EnableWindow(item, 0)

    ico = win32gui.LoadIcon(0, win32con.IDI_ERROR)
    bmCtrl = win32gui.GetDlgItem(self.hwnd, self.iconID)
    win32gui.SendMessage(bmCtrl, win32con.STM_SETICON, ico, 0)

    self.Recalculate()


  def OnCommand(self, hwnd, msg, wparam, lparam):
    id = win32api.LOWORD(wparam)
    if id == win32con.IDCLOSE:
      win32gui.EndDialog(hwnd, 1)
    elif id == self.detailsID:
      self._showsDetail = not self._showsDetail
      if self._showsDetail:
        item = win32gui.GetDlgItem(self.hwnd, self.detailID)
        win32gui.EnableWindow(item, 1)
        win32gui.SetDlgItemText(hwnd, id, u_("<< Details"))
        self.Recalculate()
      else:
        item = win32gui.GetDlgItem(self.hwnd, self.detailID)
        win32gui.EnableWindow(item, 0)
        win32gui.SetDlgItemText(hwnd, id, u_(">> Details"))
        self.Recalculate()

  def OnCtlColorStatic(self, hwnd, msg, wparam, lparam):
    if win32gui.GetDlgCtrlID(lparam) == self.detailID:
      win32gui.SetBkMode(wparam, win32con.OPAQUE) #TRANSPARENT) #
      win32gui.SetTextColor(wparam, win32api.RGB(0, 0, 0))
      return 1

  def Recalculate(self):
    border = 10
    
    dlgWidth = self.Width(self.detailsID) + self.Width(win32con.IDCLOSE) + 20
    dlgWidth = max(dlgWidth, self.Width(self.messageID))
    if self._showsDetail:
      dlgWidth = max(dlgWidth, self.Width(self.detailID))

    ypos = border

    self.SetPosition(self.messageID, border+40, ypos)
    ypos += self.Height(self.messageID) + 10
    
    if self._showsDetail:
      self.SetPosition(self.detailID, border, ypos)
      ypos += self.Height(self.detailID) + 10
    else:
      dlgItem = win32gui.GetDlgItem(self.hwnd, self.detailID)
      win32gui.SetWindowPos(dlgItem, 0, 0, 0, 0, 0, 0)

    self.SetPosition(self.detailsID, dlgWidth + 2*border \
                     - self.Width(self.detailsID) - 8, ypos + 10)

    self.SetPosition(win32con.IDCLOSE, dlgWidth + 2*border \
                     - self.Width(self.detailsID) \
                     - self.Width(win32con.IDCLOSE) -15, ypos + 10)

    ypos += self.Height(win32con.IDCLOSE) + border
    
    # resize the dialog
    win32gui.SetWindowPos(self.hwnd, 0,
                              0, 0,
                              dlgWidth + 2*border + 10, ypos + 45,
                              win32con.SWP_NOACTIVATE | win32con.SWP_NOZORDER \
                              | win32con.SWP_NOMOVE)

    # center the dialog
    centerWindow(self.hwnd)


  def Width(self, id):
    item = win32gui.GetDlgItem(self.hwnd, id)
    if id == self.messageID or id == self.detailID:
      text = win32gui.GetWindowText(item)
      # GetWindowText has only 512 byte buffer, sigh...
      if id == self.detailID:
        text = self.detail
      if '\r\n' in text:
        w = max ([win32gui.GetTextExtentPoint32(self.dc, t) [0] for t in text.split ('\r\n')])
      else:
        w, h = win32gui.GetTextExtentPoint32(self.dc, text)
      return w + 40
    else:
      l,t,r,b = win32gui.GetWindowRect(item)
      return r-l


  def Height(self, id):
    item = win32gui.GetDlgItem(self.hwnd, id)
    if id == self.messageID or id == self.detailID:
      text = win32gui.GetWindowText(item)
      if id == self.detailID:
        text = self.detail
      if '\r\n' in text:
        h = sum ([win32gui.GetTextExtentPoint32(self.dc, t) [1] for t in text.split ('\r\n')])
      else:
        w, h = win32gui.GetTextExtentPoint32(self.dc, text)
      if id == self.detailID:
        h += 2 # extra for WS_EX_STATICEDGE
      return h
    else:
      l,t,r,b = win32gui.GetWindowRect(item)
      return b-t


  def SetPosition(self, id, x, y):
    item = win32gui.GetDlgItem(self.hwnd, id)
    win32gui.SetWindowPos(item, 0,
                          x, y,
                          self.Width(id), self.Height(id),
                          win32con.SWP_NOACTIVATE | win32con.SWP_NOZORDER)
