#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2005 Free Software Foundation
#
# DESCRIPTION:
# Class that contains the internal python object representation of a GNUe Form
# built from GFObjects
#
# NOTES:
#
# $Id: GFForm.py 7983 2005-09-24 09:52:49Z reinhard $

import sys
import string
import traceback

from gnue.common.apps import i18n
from gnue.common import events
from gnue.common.datasources.Exceptions import ConnectionError as DBError
from gnue.common.definitions.GRootObj import GRootObj
from gnue.common.logic.language import AbortRequest
from gnue.common.definitions.GObjects import GObj
from gnue.common.datasources import ConnectionTriggerObj
from gnue.forms.GFObjects import *
from gnue.forms import GFParser


#-----------------------------------------------------------------------------
# Defines which objects are "Tab Stops"
#-----------------------------------------------------------------------------
TabStops = ('GFEntry','GFButton')


#-----------------------------------------------------------------------------
# GFForm
#-----------------------------------------------------------------------------
class GFForm(GRootObj, GFObj, events.EventAware):
  """
  Contains the internal python object representation of a GNUe Form
  built from GFObjects
  """

  #---------------------------------------------------------------------------
  # instance is passed into the initializer so that
  # designer can pass that in
  #---------------------------------------------------------------------------
  def __init__(self, parent=None, instance=None):
  # TODO: with a little tweaking we can now stop passing  GFParser.getXMLelements
    GRootObj.__init__(self, 'form', GFParser.getXMLelements, GFParser)
    GFObj.__init__(self, parent)

    self._type = "GFForm"
    self.name="__main__"

    # Datasources referced by this form
    self._datasourceDictionary = {}

    # Dictionary to locate named triggers
    self._triggerDictionary = {}

    # Insert/Overwrite mode
    self._insertMode = True

    # Focus information
    self._currentPage = None
    self._currentBlock = None
    self._currentEntry = None

    # set
    self._instance = instance
    self._app = self._instance #TODO: Imports are broken do to
                               #TODO: switch from _app to _instance

    # Hackery until proper layout support is added
    self._standardnamespaces = {'Char': 'GNUe:Forms:Char'}

    # The "None" init gives datasources time to setup master/detail
    self._inits = [self.primaryInit, None, self.secondaryInit]

    #
    # Trigger support
    #
    self._triggerns = {}
    self._triggerGlobal = True # Put this object into the global namespace by it's name

    self._validTriggers = { 'ON-STARTUP':     'On-Startup',
                            'ON-ACTIVATION':  'On-Activation',
                            'PRE-EXIT':       'Pre-Exit',
                            'ON-EXIT':        'On-Exit',
                            'PRE-COMMIT':     'Pre-Commit',
                            'POST-COMMIT':    'Post-Commit' }

    self._triggerFunctions = {'setFocus':{'function':self.triggerSetFocus,
                                          'global': True,
                                          },
                              'getAuthenticatedUser':{'function':self.triggerGetAuthenticatedUser,
                                                      'global': True,
                                                      },
                              'setTitle':{'function':self.triggerSetTitle,
                                                      'global': True,
                                                      },
#                               'getCurrentEntry':{'function':self.triggerGetCurrentEntry,
#                                                  'global': True,
#                                                  },
                              'getCurrentBlockName':{'function':self.triggerGetCurrentBlockName,
                                                 'global': True,
                                                 },
#                               'getCurrentPage':{'function':self.triggerGetCurrentPage,
#                                                 'global': True,
#                                                },
                              'setStatusText':{'function':self.triggerSetStatusText,
                                               'global': True,
                                               },
                              'getParameter':{'function':self.getParameter,
                                              'global': True,
                                              },
                              'setParameter':{'function':self.triggerSetParameter,
                                              'global': True,
                                              },
                              'showMessage':{'function':self.triggerShowMessageBox,
                                             'global': True,
                                             },
                              'messageBox': {'function':self.triggerMessageBox,
                                             'global': True
                                             },
                              'commit':{'function':self.triggerCommit,
                                        'global': True,
                                        },
                              'rollback':{'function':self.rollback,
                                        'global': True,
                                        },
                              'close':{'function':self.triggerClose,
                                       'global': True,
                                       },
                              'beep':{'function':self.triggerBeep,
                                       'global': True,
                                       },
                              'getFeature':{'function':self.triggerGetFeature,
                                            'global': True,
                                            },
                              'setFeature':{'function':self.triggerSetFeature,
                                            'global': True,
                                            },
                              'runForm':{'function':self.triggerRunForm,
                                         'global': True,
                                         } ,
                              'runReport':{'function':self.triggerRunReport,
                                         'global': True,
                                         } ,
                              'activateDialog':{'function':self.triggerActivateDialog,
                                                'global': True,
                                                } ,
                              'activateTrigger':{'function':self.fireTrigger,
                                                'global': True,
                                                } ,
                              'initQuery':{'function':self.initQuery,
                                           },
                              }

    self._features = {
      'GUI:MENUBAR:SUPPRESS': False,
      'GUI:TOOLBAR:SUPPRESS': False,
      'GUI:STATUSBAR:SUPPRESS': False,
    }

    self._in_trigger_lock = False

  #---------------------------------------------------------------------------
  # Called during phaseInit startup by GParser.
  #---------------------------------------------------------------------------
  def primaryInit(self):
    """
    Called automatically during phaseInit() startup of GParser.
    """
    # Initialize our events system
    events.EventAware.__init__(self, self._instance.eventController)

    # Find the logic and layout controllers
    for child in self._children:
      if child._type == 'GFLogic':
        self._logic = child
      elif child._type == 'GFLayout':
        self._layout = child

    self.initTriggerSystem()
    ConnectionTriggerObj.addAllConnections(self._instance.connections, self._triggerNamespaceTree)
    self._triggerns.update(self._triggerNamespaceTree._globalNamespace)

    ## TODO: This ain't right!  Fix after 0.5.0
    #self._triggerns['runForm'] = self.triggerRunForm
    ## TODO


  #---------------------------------------------------------------------------
  # Secondary init-stuff that must happen after all our children have init'ed
  #---------------------------------------------------------------------------
  def secondaryInit(self):
    """
    Called automatically during phaseInit().
    It must happen after all children have init'ed.
    """
    # create the first records
    for key in self._datasourceDictionary.keys():
      if not self._datasourceDictionary[key].hasMaster():
        self._datasourceDictionary[key].createEmptyResultSet()

    # Set initial focus
    self.findAndChangeFocus(self)
    self.processTrigger('On-Startup')


  #---------------------------------------------------------------------------
  # Get a user parameter. If parameter not specified, pull default value
  #---------------------------------------------------------------------------
  def getParameter(self, parameter):
    """
    Gets a user parameter. If parameter is not specified, pulls default value.
    @param parameter: case-insensitive name of parameter
    @return: parameter value or None
    """
    param = string.lower(parameter)
    try:
      return self._parameters[param]
    except KeyError:
      rv = None
      for child in self._children:
        if isinstance(child, GFParameter) and child._name == param:
          rv = child.default
          self._parameters[param] = rv
          break
    return rv


  #===========================================================================
  # Focus functions
  #===========================================================================

  #---------------------------------------------------------------------------
  # Runs findFocus and then runs changeFocus with that value.
  #---------------------------------------------------------------------------
  def findAndChangeFocus(self, object):
    """
    Change the focus to the first focusable element of the given object.

    The object passed can be a page or a block (in which case the first entry
    on that page/block will be focused) or an entry (in which case exactly this
    entry will be focused).

    The focus will be set on GF and on UI layer.

    @param: Object from the GF layer
    """
    entry = self.findFocus(object)
    if entry:
      # Set focus on GF layer
      self.changeFocus(entry)
      # Set focus on UI layer
      self.dispatchEvent('gotoENTRY', object=entry, _form=self)


  #---------------------------------------------------------------------------
  # Find the next focusable item given an item of type Page, Block, or Entry
  # Only called by findAndChangeFocus
  #---------------------------------------------------------------------------
  def findFocus(self, object):
    """
    Finds the next focusable item of a page, a block or an entry.
    @param: object (page, block or entry) to find next focus for.
    @return: None or entry with next focus
    """
    
    if object._type == 'GFField':
      assert gDebug (5, "GFField was passed to 'findFocus'.")
      try:
        object = object._entryList[0]
      except IndexError:
        return None

    entry = None

    mode = self.getCurrentMode()

    if isinstance(object, GFObj):
      if object.isNavigable(mode):
        return object
      else:
        if hasattr(object, 'getFocusOrder'):
          for child in object.getFocusOrder():
            entry = self.findFocus(child)
            if entry:
              return entry
        #
        # We're in trouble.  Scan all children for a
        # focusable widget.  Will dump GFField objects
        # stored in blocks into itself.
        #
        for child in object._children:
          entry = self.findFocus(child)
          if entry:
            break

    return entry


  #---------------------------------------------------------------------------
  # Returns the editing mode of the current record
  #---------------------------------------------------------------------------
  def getCurrentMode(self):
    """
    Returns the editing mode of the current record
    @return: 'query', 'new' or 'edit'
    """
    if self._currentBlock:
      if self._currentBlock.mode == 'query':
        return 'query'
      elif self._currentBlock.isEmpty():
        return 'new'
      else:
        return 'edit'
    else:
      return 'new'

  #---------------------------------------------------------------------------
  # Changes to the requested entry object requested by an event source
  #---------------------------------------------------------------------------
  def changeFocus(self, widget, fireFocusTriggers=True):
    """
    Changes focus to the requested entry object.
    @param widget: entry to put focus on
    @param fireFocusTriggers: whether to fire focus-change trigger?
    """

    message = None

    # cannot change focus to same entry
    if widget == self._currentEntry:
      assert gDebug (5, "Current entry did not change from %s" % widget)
      return

    cen = self._currentEntry is not None and self._currentEntry.name or '-'
    win = widget is not None and widget.name or '-'
    assert gDebug (5, "Changing focus from %s (%s) to %s (%s)" % \
               (self._currentEntry, cen, widget, win))

    try:
      if self._currentEntry:

        if self._currentEntry._type != 'GFButton':
          val = self._currentEntry.getValue()
          if self._currentEntry._field.minLength and val is not None and len(str(val)):
            if len(str(val)) < self._currentEntry._field.minLength:
              message = u_("Minimum required length %d") \
                        % self._currentEntry._field.minLength
              # Piggybacking off the triggers message box code
              self.triggerShowMessageBox (message)
              return

        event = events.Event('endEDITMODE',_form=self)
        self.dispatchEvent(event)

        #self._instance.dispatchEvent(event)
        if event.__error__:
          return True

      fieldChange = widget != self._currentEntry
      try:
        blockChange = widget._block != self._currentBlock
      except AttributeError:
        # Buttons don't have a block, but also
        # don't trigger a block change
        blockChange = False
      pageChange = widget._page != self._currentPage

      if fireFocusTriggers:
        try:
          if fieldChange:
            self._currentEntry.processTrigger('Pre-FocusOut', ignoreAbort=False)
            if hasattr(self._currentEntry,'_field'):
              self._currentEntry._field.processTrigger('Pre-FocusOut', ignoreAbort=False)
          if blockChange:
            self._currentBlock.processTrigger('Pre-FocusOut', ignoreAbort=False)
          if pageChange:
            self._currentPage.processTrigger('Pre-FocusOut', ignoreAbort=False)
            self._currentPage.processTrigger('Post-FocusOut', ignoreAbort=False)
          if blockChange:
            self._currentBlock.processTrigger('Post-FocusOut', ignoreAbort=False)
          if fieldChange:
            self._currentEntry.processTrigger('Post-FocusOut', ignoreAbort=False)
            if hasattr(self._currentEntry,'_field'):
              self._currentEntry._field.processTrigger('Post-FocusOut', ignoreAbort=False)
        except AttributeError:
          pass

      oldEntry = self._currentEntry

      self._currentEntry = widget
      try:
        self._currentBlock = self._currentEntry._block
      except AttributeError:
        pass # Buttons, et al
      self._currentPage = self._currentEntry._page

      if pageChange:
        self.dispatchEvent('gotoPAGE',self._currentPage, _form=self);

      if blockChange:
        self.refreshDisplay(self._currentBlock)

      assert gDebug (5, "Updating entries old: %s, new: %s" % \
                 (oldEntry, self._currentEntry))

      self.dispatchEvent('updateENTRY', oldEntry, _form=self)
      self.dispatchEvent('updateENTRY', self._currentEntry, _form=self)

      if fireFocusTriggers:
        if pageChange:
          self._currentPage.processTrigger('Pre-FocusIn', ignoreAbort=False)
        if blockChange:
          self._currentBlock.processTrigger('Pre-FocusIn', ignoreAbort=False)
        if fieldChange:
          self._currentEntry.processTrigger('Pre-FocusIn', ignoreAbort=False)
          if hasattr(self._currentEntry,'_field'):
            self._currentEntry._field.processTrigger('Pre-FocusIn', ignoreAbort=False)
          if hasattr(self._currentEntry,'_field'):
            self._currentEntry._field.processTrigger('Post-FocusIn', ignoreAbort=False)
          self._currentEntry.processTrigger('Post-FocusIn', ignoreAbort=False)
        if blockChange:
          self._currentBlock.processTrigger('Post-FocusIn', ignoreAbort=False)
        if pageChange:
          self._currentPage.processTrigger('Post-FocusIn', ignoreAbort=False)
        if blockChange or pageChange:
          self._instance.updateStatus (self)

      assert gDebug (5, "calling refreshUIEvents ()")
      self.refreshUIEvents()
      assert gDebug (5, "CurrentEntry is %s" % self._currentEntry)

    except AbortRequest, t:
      assert gDebug (5, "Trigger Error!")
      message = _("Trigger Exception :\n") + t.message

    return message


  #===========================================================================
  # Events
  #
  # Incoming Event handlers
  #===========================================================================

  #---------------------------------------------------------------------------
  # Create a new, empty record.
  #---------------------------------------------------------------------------
  def newRecord(self):
    """
    Creates a new, empty record.
    """
    if not self.readonly:
      self._currentBlock.newRecord()


  # --------------------------------------------------------------------------
  # Delete the current record.
  # --------------------------------------------------------------------------

  def deleteRecord (self):
    """
    Deletes the actual record.
    """
    if not self.readonly:
      self._currentBlock.deleteRecord()
      self.refreshUIEvents()


  # --------------------------------------------------------------------------
  # Undelete the current record.
  # --------------------------------------------------------------------------

  def undeleteRecord (self):
    """
    Undeletes the actual record.
    """
    if not self.readonly:
      self._currentBlock.undeleteRecord()
      self.refreshUIEvents()


  #---------------------------------------------------------------------------
  # Check all blocks in the form whether they are saved (committed) or not.
  #---------------------------------------------------------------------------
  def isSaved(self):
    """
    Checks all block in the form whether they are saved (committed) or not.
    @return: boolean, True if all the blocks are committed.
    """

    # Is the current entry changed?
    # FIXME: should only cause the form to appear unsaved if the entry is bound
    # to a field.
    if (self._currentEntry._type != 'GFButton' and \
        self._currentEntry._displayHandler.isPending()):
      return False

    # Are there any not yet posted changes in any of the blocks?
    for block in self._logic._blockList:
      if block.isPending():
        return False

    # Does a connection have any pending (already posted but not yet committed)
    # changes?
    for connection in (self.__getConnections ()).values ():
      if connection.isPending ():
        return False

    return True


  #---------------------------------------------------------------------------
  # Enter the form into Query mode
  #---------------------------------------------------------------------------
  def initQuery(self):
    """
    Enters the form into Query mode.
    @return: None if all went well, error message in case of error.
    """
    message = None
    if self._currentBlock.mode != 'query':

      # Auto-commit?
      if self._currentBlock.autoCommit:
        self.commit()

      for block in self._logic._blockList:
        if block.isPending():
          message = _("Data not saved. Save changes or clear the form to "
                      "proceed.")
          return message

    for block in self._logic._blockList:
      block.processRollback()
      for entry in block._entryList:
        if hasattr(entry, 'queryDefault'):
          assert gDebug (5, "%s will be set to %s" % (entry.name, entry.queryDefault))
          entry.setValue(entry.queryDefault)

    # If Enter-Query is hit once, enter query mode
    # If Enter-Query is hit twice, bring back conditions from last query.
    # If Enter-Query is hit thrice, cancel the query and go into normal mode.

    self.dispatchEvent('beginENTERQUERY', _form=self)
    #self._instance.dispatchEvent('beginENTERQUERY', _form=self)

    for block in self._logic._blockList:
      block.initQuery()

    self.triggerSetStatusText(_('Enter your query criteria.'))
    return message


  #---------------------------------------------------------------------------
  # Cancel Query mode
  #---------------------------------------------------------------------------
  def cancelQuery(self):
    """
    Cancels Query mode.
    @return: None
    """
    self.dispatchEvent('endENTERQUERY', _form=self)
    #self._instance.dispatchEvent('endENTERQUERY', _form=self)
    message = None  #TODO: not used here and in return line /kilo/
    for block in self._logic._blockList:
      block.cancelQuery()

    self.triggerSetStatusText(_('Query canceled.'))
    return message


  #---------------------------------------------------------------------------
  # Copy Query, ie bring back conditions from the last query
  #---------------------------------------------------------------------------
  def copyQuery(self):
    """
    Copies the Query, ie brings back conditions from the last query.
    @return: None
    """
    if self._currentBlock.mode != 'query':
      self.initQuery()

    self.dispatchEvent('endENTERQUERY', _form=self);
    #self._instance.dispatchEvent('endENTERQUERY', _form=self);
    message = None  #TODO: not used here and in return line /kilo/
    for block in self._logic._blockList:
      block.copyQuery()

    return message


  #---------------------------------------------------------------------------
  # Run the query
  #---------------------------------------------------------------------------
  def executeQuery (self):
    """
    Runs the query.
    @return: None if all went well, error message if there was an error
    """
    if self._currentBlock.mode != 'query':
      return _("Form not in query mode")

    message = None
    self.dispatchEvent ('endENTERQUERY', _form = self);
    self.dispatchEvent ('beginWAIT', _form = self);

    try:
      try:
        self._currentBlock.processQuery ()

      except DBError:
        self.rollback (1)
        message = u_("Database query error:\n%(exType)s\n%(exMessage)s") \
            % {'exType'   : sys.exc_info ()[0],
               'exMessage': sys.exc_info ()[1]}

    finally:
      self.dispatchEvent ('endWAIT', _form = self)

    self.refreshDisplay (self._currentBlock)
    if self._currentBlock.isEmpty ():
      self.triggerSetStatusText (_('Query returned no results.'))
    else:
      self.triggerSetStatusText (_('Query successful.'))

    return message


  #---------------------------------------------------------------------------
  # Commit all pending changes
  #---------------------------------------------------------------------------
  def commit(self):
    """
    Commits all pending changes.
    @return: None if all went well, error message otherwise
    """

    assert gEnter (4)

    self.endEditing ()
    if self.readonly:
      return _('Form is readonly')
    
    message = None
    newBlock = None                     # block to jump to after commit
    self.dispatchEvent ('beginWAIT', _form = self)

    try:
      try:
        # Save all current records, since they get lost in the Pre-Commit code
        for block in self._logic._blockList:
          block._precommitRecord = block._currentRecord

        # Form level pre-commit triggers
        try:
          self.processTrigger ('Pre-Commit', ignoreAbort = False)

        except AbortRequest:
          assert gDebug (5, "Trigger form Pre-Commit threw a AbortRequest!")
          message = u_("Form trigger returned error")


        # Process the commit on all blocks
        for block in self._logic._blockList:
          assert gDebug (5, "Saving %s" % block.name)
          try:
            block.processCommit ()

          except AbortRequest:
            assert gDebug (5, "Trigger block Pre-Commit threw a AbortRequest!")
            message = u_("Block trigger returned error")
            # jump to offending block
            newBlock = block
          except:
            # jump to offending block
            newBlock = block
            raise

        try:
          # Now do the real commit () on the backend connections (only once per
          # connection, if multiple blocks are sharing the same connection)
          for conn in self.__getConnections ().values ():
            conn.commit ()

        except:
          # Make sure the block is in consistent state again; this has to be
          # done in any case if the processCommit was successful, even if the
          # connection commit failed!
          for block in self._logic._blockList:            
            block.finalizeCommit (False)
          raise

        for block in self._logic._blockList:            
          block.finalizeCommit (True)

        # Execute Post-Commit-Trigger for each block
        for block in self._logic._blockList:
          block.processTrigger ('Post-Commit')

        for block in self._logic._blockList:
          if block.autoClear:
            block.processClear ()

        self.dispatchEvent ('cannotCOMMIT')
        self.dispatchEvent ('cannotROLLBACK')

        # Execute Post-Commit-Trigger for the form
        self.processTrigger ('Post-Commit')

      except DBError:
        message = u_("Database commit error:\n%(exType)s\n%(exMessage)s") \
            % {'exType'   : sys.exc_info ()[0],
               'exMessage': sys.exc_info ()[1]}               

    finally:
      # Make sure the current block still has the focus, even if an exception
      # occured during commit or in a trigger
      if newBlock is not None and newBlock != self._currentBlock:
        self.findAndChangeFocus (newBlock)
      self.dispatchEvent ('endWAIT', _form = self)

    assert gLeave (4)

    return message


  #---------------------------------------------------------------------------
  # Rolls back any uncommitted transatcion
  #---------------------------------------------------------------------------
  def rollback (self, recover = False):
    """
    Rolls back any uncommitted transaction.
    @return: None
    """
    self.endEditing()

    # Call rollback only once per connection (if multiple blocks are sharing
    # the same connection)
    for (cName, connection) in self.__getConnections ().items ():
      connection.rollback ()

    for block in self._logic._blockList:
      block.processRollback (recover, backendRollback = False)

    self.refreshDisplay (self)
    self._currentBlock.jumpRecord(self._currentBlock._currentRecord)
    self.dispatchEvent ('cannotCOMMIT')
    self.dispatchEvent ('cannotROLLBACK')


  # ---------------------------------------------------------------------------
  # Get all connections used by the form
  # ---------------------------------------------------------------------------

  def __getConnections (self):
    """
    This function creates a dictionary of all connections referenced by the form,
    where the connection-name is the key and the connection instance is
    the value.

    @return: dictionary with all connections used by the form
    """

    result = {}
    for dLink in self._datasourceDictionary.values():
      try:
        if dLink._connection is not None:
          result [dLink.connection] = dLink._connection
      except AttributeError:
        pass
        
    
    return result


  #---------------------------------------------------------------------------
  # Launch a trigger
  #---------------------------------------------------------------------------
  def fireTrigger(self, triggerName):
    """
    Launches a trigger.
    @param triggerName: name of  the trigger to be launched.
    @return: None
    """
    self._triggerDictionary[triggerName](self)


  #---------------------------------------------------------------------------
  # Updates every UI entry of a field.
  #---------------------------------------------------------------------------
  def updateUIEntry(self, field):
    """
    Updates every UI entry of a field.
    @param field: the field object whose ui is to be refreshed
    @return: None
    """
    for entry in field._entryList:
      self.dispatchEvent('updateENTRY', entry, _form=self)
      #self._instance.dispatchEvent('updateENTRY',entry, _form=self)


  #---------------------------------------------------------------------------
  # Called whenever an event source has requested that the
  # focus change to the next data entry object
  #---------------------------------------------------------------------------
  def nextEntry(self, reverse=False, first=False, onlyInBlock=False):
    """
    Called whenever an event source has requested that the focus change
    to the next data entry object.
    @param reverse: boolean, step focus in reverse direction?
    @param first: boolean, change focus to the first entry in the block?
    @param onlyInBlock: boolean, can jump out of block to next block?
    @return: None
    """
    currentBlock = self._currentBlock
    mode = self.getCurrentMode()

    if currentBlock.transparent and not ( \
          onlyInBlock or \
          currentBlock.autoNextRecord and not ( \
              currentBlock.isEmpty() or \
              (not reverse and currentBlock.isLastRecord() and \
                   not (currentBlock.autoCreate and \
                        currentBlock.editable in ('Y', 'new')) or \
              (reverse and currentBlock.isFirstRecord()) \
          ))):
      source = self._currentEntry._page.getFocusOrder()
      stayInBlock = False
    else:
      source = currentBlock.getFocusOrder()
      stayInBlock = True

    # If we want the previous entry, then reverse the focusorder we're using
    if reverse:
      source.reverse()

    nextEntry = None
    firstEntry = None
    keepNext = False

    for object in source:

      if object.isNavigable(mode):
        if stayInBlock and \
           (currentBlock.name != object.block):
          continue
        # If we only wanted the first navigable field in the block, then return
        if first:
          nextEntry = object
          break

        # Put the first field as the next to rollover
        if nextEntry == None:
          nextEntry = object
          firstEntry = object

        # If we're at the current focused entry,
        # then the next entry will be what we want
        if object == self._currentEntry:
          keepNext = True

        # If we've already passed the current entry
        # Then this is the entry to return
        elif keepNext:
          nextEntry = object
          break

    # If we've cycled back around to the first entry, then do special checks
    if nextEntry == firstEntry:

      # If we should navigate to the next record, do it...
      if reverse and not currentBlock.isFirstRecord():
        currentBlock.prevRecord()
        self.changeFocus(nextEntry)
      elif not reverse and \
         currentBlock.autoNextRecord and \
         not currentBlock.isEmpty() and \
         not (not currentBlock.autoCreate and \
              currentBlock.isLastRecord()):
           currentBlock.nextRecord()

           # If new record is empty, then the
           # nextEntry logic has been taken care of...
           if currentBlock.isEmpty():
             return
           else:
             self.changeFocus(nextEntry)

      # Otherwise, are we transparent? If so, go to next block/page
      elif currentBlock.transparent and \
         self._currentPage.transparent:

        # Jump to the next/(previous) page if block is page as transparent
        pages =  self._layout._pageList
        i = pages.index(self._currentPage)

        if reverse:
          try:
            dest = self._layout._pageList[i - 1]
          except IndexError:
            dest = self._layout._pageList[-1]
          # TODO: this fails if last entry is not navigable
          self.findAndChangeFocus(dest._entryList[-1])
        else:
          try:
            dest = self._layout._pageList[i + 1]
          except IndexError:
            dest = self._layout._pageList[0]
          self.findAndChangeFocus(dest)
      else:
        self.changeFocus(nextEntry)

    else:
      self.changeFocus(nextEntry)


  #---------------------------------------------------------------------------
  # Called whenever an event source has requested that the
  # focus change to the previous data entry object
  #---------------------------------------------------------------------------
  def previousEntry(self):
    """
    Called whenever an event source has requested that the focus change
    to the previous data entry object.
    @return: None
    """
    return self.nextEntry(reverse=True)


  #---------------------------------------------------------------------------
  # Refreshes all the UI fields to match the virtual form's fields
  #---------------------------------------------------------------------------
  def refreshDisplay(self, block):
    """
    Refreshes all the UI fields to match the virtual form's fields.
    @param block: block object containing fields to refresh
    @return: None
    """
    block.walk(self.__refreshDisplay)
    self.refreshUIEvents()


  #---------------------------------------------------------------------------
  # Refresh a given data entry object's UI
  #---------------------------------------------------------------------------
  def __refreshDisplay(self, object):
    if object._type in ('GFEntry', "GFImage"):
      self.dispatchEvent('updateENTRY',object, _form=self)


  #---------------------------------------------------------------------------
  # Signal the UI Drivers of navigation button relevance
  #---------------------------------------------------------------------------
  def refreshUIEvents(self):
    """
    Signals the UI drivers of navigation button relevance.
    @return: None
    """
    block = self._currentBlock
    if not block:
      assert gDebug(5,'No current block - cannot refresh UI')
      return

    dispatchEvent = self.dispatchEvent
    if block.mode == 'query':
      dispatchEvent('canCANCELQUERY')
      dispatchEvent('canEXECQUERY')
      dispatchEvent('cannotENTERQUERY')
      dispatchEvent('cannotPREVRECORD')
      dispatchEvent('cannotFIRSTRECORD')
      dispatchEvent('cannotNEXTRECORD')
      dispatchEvent('cannotLASTRECORD')
      dispatchEvent('cannotCOMMIT')
      dispatchEvent('cannotROLLBACK')
      dispatchEvent('cannotJUMPPROMPT')
      dispatchEvent('cannotNEWRECORD')
      dispatchEvent('cannotMARKFORDELETE')
    else:
      dispatchEvent('canENTERQUERY')
      dispatchEvent('cannotCANCELQUERY')
      dispatchEvent('cannotEXECQUERY')

      if not self.isSaved():
        dispatchEvent('canCOMMIT')
      else:
        dispatchEvent('cannotCOMMIT')

      dispatchEvent('canROLLBACK')
      dispatchEvent('canJUMPPROMPT')
      dispatchEvent('canMARKFORDELETE')

      if block._resultSet.isFirstRecord():
        dispatchEvent('cannotPREVRECORD')
        dispatchEvent('cannotFIRSTRECORD')
      else:
        dispatchEvent('canPREVRECORD')
        dispatchEvent('canFIRSTRECORD')

      if block._resultSet.isLastRecord():
        if block._resultSet.current.isEmpty():
          dispatchEvent('cannotNEXTRECORD')
        else:
          dispatchEvent('canNEXTRECORD')
        dispatchEvent('cannotLASTRECORD')
      else:
        dispatchEvent('canNEXTRECORD')
        dispatchEvent('canLASTRECORD')
        dispatchEvent('canNEWRECORD')

      if block._resultSet.current.isEmpty():
        dispatchEvent('cannotNEWRECORD')
      else:
        dispatchEvent('canNEWRECORD')

      if block._resultSet.current.isDeleted () \
          or block._resultSet.current.isVoid ():
        dispatchEvent('cannotMARKFORDELETE')
        dispatchEvent('canUNDELETE')
        dispatchEvent('beginMARKFORDELETE')
      else:
        dispatchEvent('endMARKFORDELETE')
        dispatchEvent('canMARKFORDELETE')
        dispatchEvent('cannotUNDELETE')


  #---------------------------------------------------------------------------
  # Called whenever an event source has requested that the
  # focus change to the next data entry block
  #---------------------------------------------------------------------------
  def nextBlock(self):
    """
    Change focus to the next data entry block.
    @return: None
    """
    try:
      nextBlock = self._logic._blockList[self._logic._blockList.index(self._currentBlock) + 1]
    except IndexError:
      nextBlock = self._logic._blockList[0]
    self.findAndChangeFocus(nextBlock)


  #---------------------------------------------------------------------------
  # Convenience routine to find the previous block
  #---------------------------------------------------------------------------
  def findPreviousBlock(self):
    """
    Finds the previous block.
    @return: the block found
    """
    try:
      return self._logic._blockList[self._logic._blockList.index(self._currentBlock) - 1]
    except IndexError:
      return self._logic._blockList[-1]


  #---------------------------------------------------------------------------
  # Called whenever an event source has requested that the
  # focus change to the previous data entry block
  #---------------------------------------------------------------------------
  def previousBlock(self):
    """
    Change focus to the previous data entry block.
    @return: None
    """
    self.findAndChangeFocus(self.findPreviousBlock())


  #---------------------------------------------------------------------------
  # Signal to the current Entry to stop editing
  # mode and save it's value to the virtual form
  #---------------------------------------------------------------------------
  def endEditing(self):
    """
    Signals the current entry to stop editing mode and
    save it's value to the virtual form.
    @return: Boolean, True if succeeded, False if failed.
    """
    event = events.Event('endEDITMODE', None, _form=self)
    self.dispatchEvent(event)
    return not event.__error__


  #---------------------------------------------------------------------------
  # Step to previous record in block.
  #---------------------------------------------------------------------------
  def prevRecord(self):
    """
    Steps to the previous record in the current block.
    @return: None
    """
    if self._currentBlock.mode == 'query':
      self.triggerSetStatusText(_('You cannot do that in query mode.'))
      return
    self._currentBlock.prevRecord()


  #---------------------------------------------------------------------------
  # Jump to first record in block.
  #---------------------------------------------------------------------------
  def firstRecord(self):
    """
    Jumps to the first record in the current block.
    @return: None
    """
    if self._currentBlock.mode == 'query':
      self.triggerSetStatusText(_('You cannot do that in query mode.'))
      return
    self._currentBlock.firstRecord()


  #---------------------------------------------------------------------------
  # Jump to last record in block.
  #---------------------------------------------------------------------------
  def lastRecord(self):
    """
    Jumps to the last record in the current block.
    @return: None
    """
    if self._currentBlock.mode == 'query':
      self.triggerSetStatusText(_('You cannot do that in query mode.'))
      return
    self._currentBlock.lastRecord()


  #---------------------------------------------------------------------------
  # Step to next record in block.
  #---------------------------------------------------------------------------
  def nextRecord(self):
    """
    Steps to the next record in the current block.
    @return: None
    """
    if self._currentBlock.mode == 'query':
      self.triggerSetStatusText(_('You cannot do that in query mode.'))
      return
    self._currentBlock.nextRecord()


  #---------------------------------------------------------------------------
  # Jumpt to a given record in block.
  #---------------------------------------------------------------------------
  def jumpRecord(self, count):
    """
    Jumps to a given record in the current block.
    @param count: ordinal of the record
    @return: None
    """
    if self._currentBlock.mode == 'query':
      self.triggerSetStatusText(_('You cannot do that in query mode.'))
      return
    self._currentBlock.jumpRecord(count)


  #---------------------------------------------------------------------------
  # Toggles insert mode
  #---------------------------------------------------------------------------
  def toggleInsertMode(self):
    """
    Toggles insert mode.
    @return: None
    """
    self._insertMode = not self._insertMode


  #===========================================================================
  # Trigger functions
  #
  # The following functions are only used to implement
  # functions exposed in the trigger namespace.
  #===========================================================================

  #---------------------------------------------------------------------------
  # trigger commits
  #---------------------------------------------------------------------------
  def triggerCommit(self):
    """
    Causes the form to commit
    """  
    event = events.Event('requestCOMMIT',_form=self)
    self._instance.executeCommit(event)
  
  #---------------------------------------------------------------------------
  # trigger beep
  #---------------------------------------------------------------------------
  def triggerBeep(self):
    """
    Tells the UI to beep (if possible)
    
    @return: None
    """
    self.dispatchEvent('formBEEP', _form=self)
  
  #---------------------------------------------------------------------------
  # display a standard message box
  #---------------------------------------------------------------------------
  def triggerShowMessageBox (self, msgtxt, caption='GNUe Message',
      title='Information', kind = 'Info', cancel = False):
    """
    Displays a standard message box.
    @param msgtxt: text to be displayed
    @param caption: caption to be displayed, default='GNUe Message'
    @param title: title of messagebox, default='Information'
    @param kind: 'Question', 'Info', 'Warning', or 'Error', default='Info'
    @param cancel: Flag whether to include a Cancel button, default='False'
    @return: True for <Yes> or <Ok> button, False for <No> button, None for
        <Cancel> button.
    """
    return self._instance.displayMessageBox (msgtxt, kind, cancel, caption,
                                             title)


  # --------------------------------------------------------------------------
  # Show a message box using the given message
  # --------------------------------------------------------------------------
  def triggerMessageBox (self, message, kind = None, title = None,
      cancel = False):
    """
    This function brings up a message box of a given kind.
    @param message: text to be displayed
    @param kind: 'Question', 'Info', 'Warning', or 'Error'
    @param cancel: Boolean flag indicating wether a cancel button will be
        included or not.
    @return: True for <Yes> or <Ok> button, False for <No> button, None for
        <Cancel> button.
    """
    return self._instance.displayMessageBox (message, kind, cancel, '', title)


  #---------------------------------------------------------------------------
  # Switch input focus to a specfic entry widget
  #---------------------------------------------------------------------------
  def triggerSetFocus(self, object):
    """
    Switches input focus to a specific widget.
    @param object: the widget that should get the focus
    """
    # add global focus locking
    if self._in_trigger_lock:
      print "Already called by a trigger"
      return

    self._in_trigger_lock = True

    focus = object._object
    if focus._type == 'GFField':
      try:
        focus = focus._entryList[0]
      except KeyError:
        raise "setFocus failed: GFField is not bound to an GFEntry object."

    if focus._type != 'GFEntry':
      raise "setFocus failed: Can just switch to GFEntry objects. " +\
            "You passed a '%s' object." % focus._type
    self.changeFocus(focus, 1)
    self.dispatchEvent('gotoENTRY', object=self._currentEntry, _form=self)
    self._in_trigger_lock = False


  #
  # allow the trigger to get a handle to
  # an actual GFBlock object
  #
  #
  def triggerGetCurrentBlockName(self):
    return self._currentBlock.name
#   #
#   # allow the trigger to get a handle to
#   # an actual GFEntry object
#   #
#   # TODO: WHY?  This seems like a good way to write
#   # TODO: triggers that break due to internal forms changes
#   #
#   def triggerGetCurrentEntry(self):
#     return self._currentEntry
#
#
#   #
#   # allow the trigger to get a handle to
#   # an actual GFBlock object
#   #
#   # TODO: WHY?  This seems like a good way to write
#   # TODO: triggers that break due to internal forms changes
#   #
#   def triggerGetCurrentBlock(self):
#     return self._currentBlock
#
#   #
#   # allow the trigger to get a handle to
#   # an actual GFPage object
#   #
#   # TODO: WHY?  This seems like a good way to write
#   # TODO: triggers that break due to internal forms changes
#   #
#   def triggerGetCurrentPage(self):
#     return self._currentPage


  #---------------------------------------------------------------------------
  # Display a custom message on the form's status bar
  #---------------------------------------------------------------------------
  def triggerSetStatusText(self, tip=''):
    """
    Displays a custom message on the form's status bar.
    @param tip: message to be displayed
    @return: None
    """
    self._instance.updateStatusBar(tip=tip, form=self)


  #---------------------------------------------------------------------------
  # Launch a new instance of gnue-forms running a different form.
  #---------------------------------------------------------------------------
  def triggerRunForm(self, fileName, parameters={}):
    """
    Launches a new instance of GNUe-Forms, running a different form.
    @param fileName: the name of the .gfd file to be displayed
    @param parameters: dictionary of parameters to be passed
                        to the newly run form
    @return: None
    """
    from GFInstance import GFInstance
    instance = GFInstance(self._instance.manager,
                          self._instance.connections,
                          self._instance._uimodule,
                          1,
                          parameters=parameters)
    instance.addFormFromFile(fileName)
    #instance.addDialogs ()
    instance.activate()


  #---------------------------------------------------------------------------
  # Launch a new instance of gnue-reports running a new report.
  #---------------------------------------------------------------------------
  def triggerRunReport(self, reportFile, parameters={}, **parms):
    """
    Launches a new instance of GNUe-Reports, running a new report.
    @param reportFile: the name of the .grd file to be processed
    @param parameters: dictionary of parameters to be passed
                        to the newly run report
    @param **params:
      These roughly correspond to the ./gnue-reports options
              destination
              destinationType
              destinationOptions (dict)
              filter
              filterOptions (dict)
              sortoption
              includeStructuralComments
              omitGNUeXML
    @return: None
    """
    from gnue.reports.base.GREngine import GREngine
    from gnue.reports.base import GRFilters, GRExceptions
    rep_engine = GREngine(self._instance.connections)
    rep_engine.processReport(reportFile, parameters=parameters, **parms)


  #---------------------------------------------------------------------------
  # Launch a standard or custom dialog
  # Values can be passed back and forth thru the parameters dictionary
  #---------------------------------------------------------------------------
  def triggerActivateDialog(self, dialogName, parameters={}, modal=0):
    """
    Launches a standard or a custom dialog.
    @param dialogName: name of the dialog to be displayed
    @param parameters: dictionary of parameters used to pass values
                        back and forth
    @param modal: whether the dialog should be modal or not
    @return: None
    """

    if dialogName == '_about':
      self._instance._uiinstance.showAbout (**parameters)
    else:      
      self._instance.activateForm(dialogName, parameters, modal)    
                                

  #---------------------------------------------------------------------------
  # Close this copy of gnue-forms
  #---------------------------------------------------------------------------
  def triggerClose(self):
    """
    Closes this copy of GNUe-Forms.
    @return: None, or 1 if error
    """
    event = events.Event('requestEXIT', _form=self)
    #self._instance.dispatchEvent(event)
    self.dispatchEvent(event)
    if event.__error__:
      return 1


  #---------------------------------------------------------------------------
  # Set feature values. Features are like toolbars, menubars, and statusbar
  #---------------------------------------------------------------------------
  def triggerSetFeature(self, feature, value):
    """
    Sets feature values.
    Features are things like toolbars, menubars and statusbar.
    @param feature: 'GUI:MENUBAR:SUPPRESS' or 'GUI:TOOLBAR:SUPPRESS' or
                        'GUI:STATUSBAR:SUPPRESS'
    @param value: True or False
    @return: None
    """
    if not self._features.has_key(feature):
      raise KeyError, "Trigger attempted to set unknown feature %s" % feature
    else:
      self._features[feature] = value


  #---------------------------------------------------------------------------
  # Get feature values. Features are like toolbars, menubars, and statusbar
  #---------------------------------------------------------------------------
  def triggerGetFeature(self, feature):
    """
    Gets feature values.
    Features are things like toolbars, menubars and statusbar.
    @param feature: 'GUI:MENUBAR:SUPPRESS' or 'GUI:TOOLBAR:SUPPRESS' or
                        'GUI:STATUSBAR:SUPPRESS'
    @return:    Boolean
    """
    try:
      return self._features[feature]
    except KeyError:
      raise KeyError, "Trigger attempted to get unknown feature %s" % feature


  #---------------------------------------------------------------------------
  # TODO: Description
  #---------------------------------------------------------------------------
  def triggerGetAuthenticatedUser(self, connection=None):
    """
    Gets authenticated user.
    """
    return self._instance.connections.getAuthenticatedUser(connection)


  #---------------------------------------------------------------------------
  # Set the title of the form.
  #---------------------------------------------------------------------------
  def triggerSetTitle(self, title):
    """
    Sets and displays the title of the form.
    @param title: new title
    """
    self.title = title
    # code to fire event to update the display goes here
    self.dispatchEvent('setTitle', title=title, _form=self);


  #---------------------------------------------------------------------------
  # Set a user parameter.
  #---------------------------------------------------------------------------
  def triggerSetParameter(self, parameter, value):
    """
    Sets a user parameter.
    @param parameter: the (case-insensitive) name of the parameter to be set
    @value: new value for parameter
    """
    param = string.lower(parameter)
    self._parameters[param] = value


  #---------------------------------------------------------------------------
  # Runs another trigger.
  #---------------------------------------------------------------------------
  def triggerActivateTrigger(self, name):
    """
    Runs another trigger.
    @param name: name of the trigger to be run
    """
    self.processTrigger (name, False)

