#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2005 Free Software Foundation
#
# FILE:
# GFField.py
#
# DESCRIPTION:
"""
The primary data entry widget in forms
"""
# NOTES:
#

from gnue.common.apps import errors
from GFValue import GFValue
import string

############################################################
# GFField
#
# Matches the Entry form tag in the gfd
#
# It send events to its parent GFBlock
#
class GFField(GFValue):
  def __init__(self, parent=None, value=None):
    GFValue.__init__(self, parent, value, 'GFField')

    # Default attributes (these may be replaced by parser)
    self.typecast="text"
    self.case="mixed"
    self.style="default"
    self.defaultToLast = False
    self.rtrim = False
    self.ltrim = False

    # Runtime variables
    self._uppercase = False
    self._lowercase = False
    self._numeric = False
    self._queryable = True

    self._inits = [self.initialize]
    self._allowedValues = None
    self._allowedValuesDescr = [""]

    self._rows = 1
    self._gap = 0

    # This will be populated by GFEntry's initialize
    self._entryList = []

    #
    # Trigger exposure
    #
    self._validTriggers ={ 'PRE-FOCUSOUT':   'Pre-FocusOut',
                           'POST-FOCUSOUT':  'Post-FocusOut',
                           'PRE-FOCUSIN':    'Pre-FocusIn',
                           'POST-FOCUSIN':   'Post-FocusIn',
                           'PRE-COMMIT':     'Pre-Commit',
                           'POST-COMMIT':    'Post-Commit',
                           'PRE-QUERY':      'Pre-Query',
                           'POST-QUERY':     'Post-Query',
##                           'PRE-MODIFY':     'Pre-Modify', #This must be a typo
                           'PRE-INSERT':     'Pre-Insert',
                           'PRE-DELETE':     'Pre-Delete',
                           'PRE-UPDATE':     'Pre-Update',
                           'PRE-CHANGE':     'Pre-Change',
                           'POST-CHANGE':    'Post-Change',
                         }

    self._triggerFunctions={'allowedValues':{'function':self.allowedValues,},
                            'autofillBySequence':{'function':self.triggerAutofillBySequence},
                            'isEmpty':{'function':self.isEmpty},
                            'resetForeignKey':{'function':self.resetForeignKey},
                            'set':{'function':self.triggerSetValue},
                            'get':{'function':self.getValue},
                            'clear': {'function': self.resetToDefault},
                            'getFKDescription': {
                               'function': self.getFKDescription},
                            'setFKDescription': {
                               'function': self.setFKDescription},
                           }
    self._triggerProperties={
          'parent':       { 'get':self.getParent},
          'value':        { 'set':self.triggerSetValue,
                            'get':self.getValue },
          'rtrim':        { 'set':self.triggerSetRtrim,
                            'get':self.triggerGetRtrim },
          'readonly':     { 'set':self.triggerSetReadonly,
                            'get':self.triggerGetReadonly },
          'maxLength':    { 'set':self.triggerSetMaxLength,
                            'get':self.triggerGetMaxLength },
          'queryDefault': { 'set':self.triggerSetQueryDefault,
                            'get':self.triggerGetQueryDefault },
          'ltrim':        { 'set':self.triggerSetLtrim,
                            'get':self.triggerGetLtrim },
          'minLength':    { 'set':self.triggerSetMinLength,
                            'get':self.triggerGetMinLength },
          'editable':     { 'set':self.triggerSetEditable,
                            'get':self.triggerGetEditable },
          'ignoreCaseOnQuery':{
                            'set':self.triggerSetIgnoreCaseOnQuery,
                            'get':self.triggerGetIgnoreCaseOnQuery },
          'case':         { 'set':self.triggerSetCase,
                            'get':self.triggerGetCase },
          'default':      { 'set':self.triggerSetDefault,
                            'get':self.triggerGetDefault },
          'defaultToLast':{ 'set':self.triggerSetDefaultToLast,
                            'get':self.triggerGetDefaultToLast },
          'required':     { 'set':self.triggerSetRequired,
                            'get':self.triggerGetRequired },
          'queryable':    { 'set':self.triggerSetQueryable,
                            'get':self.triggerGetQueryable },
      }
    self._triggerSet = self.triggerSetValue
    self._triggerGet = self.getValue


  def isEmpty(self):
    return self.getValue() in ("",None)

  def _buildObject(self):
    # Convert deprecated attributes

    # Foreign keys
    # DEPRECATED with 0.4.x .. Delete before 1.0
    if hasattr(self,'foreign_key') and self.foreign_key:
      (self.fk_source, self.fk_key) = string.split(self.foreign_key,'.')
      del self.foreign_key
    if hasattr(self,'foreign_key_description') and self.foreign_key_description:
      self.fk_description = self.foreign_key_description
      del self.foreign_key_description

    # Deprecated with 0.5.4 .. Delete before 1.0
    if hasattr(self,'min_length'):
      self.minLength = self.min_length
      del self.min_length
    if hasattr(self,'max_length'):
      self.maxLength = self.max_length
      del self.max_length

    return GFValue._buildObject(self)

  def initialize(self):

    self._block = block = self.findParentOfType('GFBlock')
    block._fieldMap[self.name] = self
    block._fieldList.append(self)

    if self.typecast == 'number':
      self._numeric = True
      self._allowFormulas = gConfigForms("AllowNumericFormulas")
    if self.case == 'upper':
      self._uppercase = True
    elif self.case == 'lower':
      self._lowercase = True


##    self._formatmask = ""
##    self._inputmask = hasattr(self,'inputmask') and self.inputmask or ""
##    self._displaymask = hasattr(self,'displaymask') and self.displaymask or ""


    if hasattr(self, 'default') and self.default != None and len(self.default):
      default = self.default
    else:
      default = None

    if not hasattr(self,'field') or not len(self.field):
      self.field = "__GNUe__%s" % self.name
      self._block._dataSourceLink.referenceUnboundField(self.field, default)
      self._bound = False
    else:
      self._block._dataSourceLink.referenceField(self.field, default)
      self._bound = True

    # Initialize the foreign key lookup (if necessary)
    if hasattr(self,'fk_source'):
      try:
        self.fk_key
      except NameError:
        raise "fk_source specified, but no fk_key"

      try:
        self._fk_descr = string.split(self.fk_description,',')
      except NameError:
        self.fk_description = self.fk_key
        self._fk_descr = string.split(self.fk_key,',')

      self._fk_datasource = \
         block._form._datasourceDictionary. get (string.lower (self.fk_source))

      if self._fk_datasource is None:
        raise errors.ApplicationError, \
            u_("Datasource '%s' not found") % self.fk_source

      # Reference the foreign keys to their datasources
      # (so they are selected)
      for field in [self.fk_key] + self._fk_descr:
        self._fk_datasource.referenceField(field, None)

      # Register event handling functions
      self._fk_datasource.registerEventListeners ({
          'dsResultSetActivated': self.__dsResultSetActivated,
          'dsResultSetChanged'  : self.__dsResultSetActivated, # sic!
          'dsCursorMoved'       : self.__dsCursorMoved})


    if hasattr(self, 'queryDefault') and \
         self.queryDefault != None and \
         self._bound and \
         len(self.queryDefault):
      block._queryDefaults[self] = self.queryDefault


  # ---------------------------------------------------------------------------
  # Event handling functions for datasource events
  # ---------------------------------------------------------------------------

  def __dsResultSetActivated (self, event):
    self.__loadAllowedValues (event.resultSet)

  # ---------------------------------------------------------------------------

  def __dsCursorMoved (self, event):
    # The entry causing the fk record change has already posted a setValue for
    # this field; our current record already contians the correct data. All we
    # have to do is tell our UI to update.
    self._block._form.updateUIEntry (self)


  # ---------------------------------------------------------------------------
  #
  # ---------------------------------------------------------------------------

  def isQueryable(self):
    return self._queryable


  #
  # Pulls the proper value from the resultSet
  #
  def getValue(self):
    # TODO: No longer true...
    # We have to check to see if the _block has been
    # setup yet.  The Trigger system calls getValue
    # before it's ready so we fake it
    try:
      mode = self._block.mode
    except AttributeError:
      # Not initialized yet.
      return

    if mode == 'query':
          try:
            value = self._block._queryValues[self]
          except KeyError:
            value = None

    elif mode == 'init':
      value = self._block._initializingRecord [self.field]

    else:
      if self._block._resultSet and self._block._resultSet.current:
        value = self._block._resultSet.current.getField(self.field)
      else:
        value = None
    # TODO: part of the removed block check from above
    #else:
    #  value = None

    if value == None:       value = ''

    if self.rtrim:
      try:
        value = string.rstrip(value)
      except:
        pass
    if self.ltrim:
      try:
        value = string.lstrip(value)
      except:
        pass

    # TODO: check to see if this is still needed
    #CheckBox does not work if comment following "if" block (ra3vat)
    if self.style == 'checkbox' and value == '':      value = 0

    return value

  # Clear out the current value (setting to null, or default value)
  def resetToDefault(self):
    try:
      default = self._block._lastValues[self.field]
    except KeyError:
      if hasattr(self, 'default') and self.default != None and len(self.default):
        default = self.default
      else:
        default = None
    self.setValue(default)


  def setValue(self, value):
    # TODO: This is now being done in the display handler
    #if self.style=='checkbox' and value =='':
    #  value = 0
    #
    # Number fields
    #if self.typecast == "number" and value != None:
    #  value = float(value)

    try:
      mode = self._block.mode
    except AttributeError:
      # Not initialized yet.
      return

    if self.rtrim:
      try:
        value = string.rstrip(value)
      except:
        pass
    if self.ltrim:
      try:
        value = string.lstrip(value)
      except:
        pass

    if mode == 'query':
      self._block._queryValues[self] = value

    elif mode == 'init':
      self._block._initializingRecord [self.field] = value

    else:
      self._block.processTrigger('Pre-Change')
      self.processTrigger('Pre-Change')
      self._block._resultSet.current[self.field] = value

      if self.defaultToLast:
        self._block._lastValues[self.field] = value

      self._block.processTrigger('Post-Change')
      self.processTrigger('Post-Change')

###      if self._block._resultSet.current.isPending():
###        self.dispatchEvent('canCOMMIT')
###        self.dispatchEvent('canROLLBACK')

    # If the field value of the current record has changed, update the UI
    if mode != 'init':                  # init doesn't affect current record!
      self._block._form.updateUIEntry(self)

      # If the field is a foreign key, move the result set to the selected value
      if hasattr(self,'_GFField__fk_resultSet'):
        self.__fk_resultSet.findRecord({self.fk_key: value})


  # ---------------------------------------------------------------------------
  # If a field has a foreign key, return it's description
  # ---------------------------------------------------------------------------

  def getFKDescription (self):
    """
    This function returns the description of the foreign key or None if there's
    no fk_source available.

    @return: value of fk_description
    """

    if self._allowedValues is None:
      return None

    value = self.getValue ()
    return self._allowedValues.get (value)


  # ---------------------------------------------------------------------------
  # Set the field value by it's foreign key description
  # ---------------------------------------------------------------------------

  def setFKDescription (self, fkValue):
    """
    This function set's the field's value by it's foreign key description.

    @param fkValue: value of fk_description
    """

    if self._allowedValues is None or self.getFKDescription () == fkValue:
      return None

    # TODO: Shall we raise an exception if an invalid value is given ? Same
    # applies to setValue () which does *not* raise one atm too
    self.setValue (self._allowedValuesReverse.get (fkValue))




  #
  # allowedValues
  #
  def __loadAllowedValues (self, resultSet):
    self._allowedValues = {"":""}
    self._allowedValuesDescr = [""]
    self._allowedValuesReverse = {"":""}
    dropdownSeparator = gConfigForms('DropdownSeparator')
    if dropdownSeparator[:1] == '"' and \
           dropdownSeparator[-1:] == '"' and \
           len(dropdownSeparator) > 2:
      dropdownSeparator = dropdownSeparator[1:-1]

    array = resultSet.getArray ([self.fk_key] + self._fk_descr)

    for line in array:
      key = "%s" % line [0]
      descr = string.join (["%s" % i for i in line [1:]], dropdownSeparator)

      self._allowedValues[key] = descr
      self._allowedValuesDescr.append(descr)
      self._allowedValuesReverse[descr] = key

    # And now, position the resultSet to the correct record according to the
    # current field content
    resultSet.findRecord({self.fk_key: self.getValue()})

    # Remember the resultSet for later
    self.__fk_resultSet = resultSet
 
    assert gDebug (5,'Created for DropDown: %s' % self._allowedValues)

  def allowedValues(self):
    return (self._allowedValues, self._allowedValuesDescr)

  def resetForeignKey(self, resultSet = None):
    # Added so forms triggers could set after init/execute queries
    # which allows filtering of dropdown's in trigger code
    if resultSet == None:
      resultSet = self._fk_datasource.createResultSet()

    if hasattr (self._fk_datasource, 'master') and self._fk_datasource.master:
      # we should have been notified in this case, did we ?
      pass
    else:
      self.__loadAllowedValues(resultSet)
    pass


  # This gets called by the block whenever the current record of our own
  # resultset changes. We want the fk resultset to follow.
  def gotNewCurrentRecord (self):
    # If the field is a foreign key, move the result set to the selected value
    if hasattr(self,'_GFField__fk_resultSet'):
      self.__fk_resultSet.findRecord({self.fk_key: self.getValue()})


  def isEditable(self, mode):
    if mode == 'query':
      return self.queryable and self._block.queryable
    elif self._block._form.readonly:
      return False
    else:
      return ( mode == 'new' and self.editable in ('Y','new','null') and \
               self._block.editable in ('Y','new')) or \
             ( mode == 'edit' and \
               self._block.editable in ('Y','update') and ( \
                       self.editable in ('Y','update') or ( \
                       self.editable == 'null' and self.isEmpty())))

  ####################################################################
  #
  # Trigger functions
  #
  def triggerAutofillBySequence(self,sequenceName):
    if (not self.getValue()) or self.getValue()=="":
      sequenceNumber = self._block._dataSourceLink._connection.getSequence(sequenceName)
      self.setValue(sequenceNumber)

  #
  # triggerSetValue
  #
  # does proper typecasting when value is set via a trigger
  #
  def triggerSetValue(self, value):
    if self.style=='checkbox' and value =='':
      value = 0
    if self.typecast == "number" and value is not None:
      if value == '':
        value = 0
      else:
        value = float (value)

    self.setValue (value)

  def triggerGetReadonly(self):
    return self.isNavigable(self._block._form.getCurrentMode())

  def triggerSetReadonly(self,value):
    self.editable = value and 'Y' or 'N'

  def triggerGetEditable(self):
    return self.editable

  def triggerSetEditable(self, value):
    if not editable in ('Y','N','null','update','new'):
      editable = editable and 'Y' or 'N'  # In case they pass a boolean
    self.editable = editable

  def triggerSetRtrim(self, value):
    self.rtrim = not not value # Force boolean

  def triggerGetRtrim(self):
    return self.rtrim

  def triggerSetQueryDefault(self, value):
    self.queryDefault = value

  def triggerGetQueryDefault(self):
    return self.queryDefault

  def triggerSetLtrim(self, value):
    self.ltrim = not not value # Force boolean

  def triggerGetLtrim(self):
    return self.ltrim

  def triggerSetMinLength(self, value):
    self.minLength = value

  def triggerGetMinLength(self):
    return self.minLength

  def triggerSetMaxLength(self, value):
    self.maxLength = value

  def triggerGetMaxLength(self):
    return self.maxLength

  def triggerSetIgnoreCaseOnQuery(self, value):
    self.ignoreCaseOnQuery = not not value # Force boolean

  def triggerGetIgnoreCaseOnQuery(self):
    return self.ignoreCaseOnQuery

  def triggerSetCase(self, value):
    assert (value in ('mixed','upper','lower'))
    self.case = value

  def triggerGetCase(self):
    return self.case

  def triggerSetDefault(self, value):
    self.default = value

  def triggerGetDefault(self):
    return self.default

  def triggerSetDefaultToLast(self, value):
    self.defaultToLast = not not value # Force boolean

  def triggerGetDefaultToLast(self):
    return self.defaultToLast

  def triggerSetRequired(self, value):
    self.required = not not value # Force boolean

  def triggerGetRequired(self):
    return self.required

  def triggerSetQueryable(self, value):
    self.queryable = not not value # Force boolean

  def triggerGetQueryable(self):
    return self.queryable

