# GNU Enterprise Forms - Curses UI Driver - Entry Widget
#
# Copyright 2000-2005 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: entry.py 7898 2005-09-12 14:31:40Z jamest $

import curses

from _base import UIHelper

# =============================================================================
# Entry class
# =============================================================================

class UIEntry (UIHelper):

  # ---------------------------------------------------------------------------
  # Initialization
  # ---------------------------------------------------------------------------

  def __init__ (self, event):

    UIHelper.__init__ (self, event)

    self.__style = event.object.style
    if self.__style in ['default', 'password', 'label', 'dropdown', 'listbox']:
      self.__length = event.object ['Char:width']
    else:
      self.__length = None

    self.__height    = event.object ['Char:height']
    self.__value     = {}
    self.__offset    = {}
    self.__selection = {}
    self.__enabled   = {}
    self.__voffset   = {}

    self.__isMultiline = (self.__style in ['default', 'password'] and \
                          self.__height > 1)

    # additional indices for listboxes
    self.__index  = {}
    self.__pindex = {}
    self.__oldCursor = 1

    if self.__style == 'dropdown':
      self.__allowedValues = event.object._field._allowedValues

    elif self.__style == 'listbox':
      self.__data = event.object._field._allowedValues.values ()
      self.__data.sort ()

    if self.__style == 'checkbox':
      self._setCursor (1, 0)

  # ---------------------------------------------------------------------------
  # Initialization per row
  # ---------------------------------------------------------------------------

  def _init (self, index):

    self.__value [index] = None
    self.__selection [index] = None
    self.__enabled [index] = True
    self.__offset [index]  = 0
    self.__voffset [index] = 0
    self.__index [index]   = 0
    self.__pindex [index]  = 1

  # ---------------------------------------------------------------------------
  # Focus has changed to this entry
  # ---------------------------------------------------------------------------

  def _getFocus (self, index):

    self.__offset [index]  = 0
    self.__voffset [index] = 0
    self.__index [index]   = 0
    self.__pindex [index]  = 1

    if self.__style == 'listbox':
      self.__oldCursor = curses.curs_set (0)

    self.__repaint (index)

  # ---------------------------------------------------------------------------
  # Focus has changed away from this entry
  # ---------------------------------------------------------------------------

  def _loseFocus (self, index):

    if self.__style == 'listbox':
      curses.curs_set (self.__oldCursor)

    self.__repaint (index)

  # ---------------------------------------------------------------------------
  # Set value for entry
  # ---------------------------------------------------------------------------

  def setValue (self, value, index = 0, enabled = True):

    if self.__style == 'listbox' and value in self.__data:
      if self.__value [index] <> value:
        self.__index [index]  = self.__data.index (value)
        self.__pindex [index] = 1
        self.__offset [index] = self.__index [index]

    self.__value [index]   = value
    self.__enabled [index] = enabled
    self.__repaint (index)

  # ---------------------------------------------------------------------------
  # Set cursor position
  # ---------------------------------------------------------------------------

  def setCursorPosition (self, position, index = 0):

    assert gDebug (2, "----- new position: %s" % position)
    assert gDebug (2, "      Value       : %r" % self.__value [index])
    assert gDebug (2, "      Curr.Offset : %s/%s" % (self.__offset [index],
      self.__voffset [index]))

    if self.__style in ['checkbox', 'listbox']:
      return

    if self.__isMultiline:
      if position == 0 or self.__value [index] is None:
        self.__offset [index]  = 0
        self.__voffset [index] = 0
        self.__repaint (index)
        self._setCursor (0, 0)
        return

      # Grab the text portion, which is everything up to position. If the last
      # character is a newline, we can remove this, since it would result in a
      # wrong vertical offset
      value = self.__value [index][:position+1]
      vcorr = [0, 1][value [-1] == '\n']

      assert gDebug (2, "      Cut-Part    : %r" % value)
      assert gDebug (2, "      Vert.Correct: %s" % vcorr)

      # Get the vertical cursor-position and offset
      cpVertical  = value.count ('\n') - self.__voffset [index] - vcorr
      needRepaint = False
      
      if cpVertical < 0:
        # the position is not visible right now, so we need to scroll up (which
        # means changing the voffset). We set the cursor-position to line 0.
        self.__voffset [index] += cpVertical
        cpVertical  = 1
        needRepaint = True

      elif cpVertical >= self.__height:
        # the position is not visible right now, so we need to scroll down
        # (which means changing the voffset).
        self.__voffset [index] = value.count ('\n') - self.__height + 1 - vcorr
        cpVertical  = self.__height                      # Pos. are Zero-based
        needRepaint = True

      # Now, after having a valid row, we need to determine the horizontal
      # offset based on the last line holding our requested position.
      currentLine = value.splitlines () [-1]
      cpHorizontal = len (currentLine) - self.__offset [index] - 1

      if cpHorizontal < 0:
        # Beyond left margin, so scroll to the left
        self.__offset [index] += cpHorizontal
        cpHorizontal = 0
        needRepaint  = True

      elif cpHorizontal > self.__length:
        self.__offset [index] = len (currentLine) - self.__length
        cpHorizontal = self.__length
        needRepaint  = True

      if needRepaint:
        self.__repaint (index)

      cpHorizontal = min (cpHorizontal, self.__length)
      cpHorizontal = max (cpHorizontal, 0)
      cpVertical   = min (cpVertical, self.__height - 1)
      cpVertical   = max (cpVertical, 0)

      assert gDebug (2, "H/V: %s/%s - Offsets %s/%s" % (cpHorizontal, cpVertical,
        self.__offset [index], self.__voffset [index]))

      self._setCursor (cpHorizontal, cpVertical)


    else:
      if self.__length:
        npos = position - self.__offset [index]
        if npos > self.__length:
          self.__offset [index] = position - self.__length
          npos = self.__length
          self.__repaint (index)

        elif npos < 0:
          self.__offset [index] += npos
          npos = 0
          self.__repaint (index)

        position = npos

      self._setCursor (position, 0)


  # ---------------------------------------------------------------------------
  # Set start and end of selection area
  # ---------------------------------------------------------------------------

  def setSelectedArea (self, selection1, selection2, index = 0):

    if selection1 == selection2:
      self.__selection [index] = None
    else:
      self.__selection [index] = (selection1, selection2)

    self.__repaint (index)

  # ---------------------------------------------------------------------------
  # Update entry representation on screen
  # ---------------------------------------------------------------------------

  def __repaint (self, index):

    if self.__style == 'listbox':
      # First draw the visible items of the listbox
      offset = self.__offset [index]
      lines  = self.__data [offset:offset + self.__height]

      for (line, value) in enumerate (lines):
        text = value.ljust (self.__length) [:self.__length]
        attr = self.__getAttr (index)

        # Note: this is not safe if there's a gap !
        self._setText (index + line, text, attr, self.__selection [index])

      self._parent.move (self._x, self._y + self.__pindex [index] - 1)

    elif self.__isMultiline:
      # Create all visible, empty lines
      data    = [''.ljust (self.__length)] * self.__height
      hOffset = self.__offset [index]
      vOffset = self.__voffset [index]

      # Overwrite these empty lines with the data as stated by v/h-Offset
      if self.__value [index]:
        add = self.__value [index].splitlines () [vOffset:vOffset+self.__height]
        for (ix, text) in enumerate (add):
          text = text [hOffset:hOffset + self.__length]
          data [ix] = text.ljust (self.__length) [:self.__length]

      attr = self.__getAttr (index)

      # And write everything to screen
      for (ix, text) in enumerate (data):
        self._setText (index + ix, text, attr, self.__selection [index])

    else:
      value  = self.__value [index]
      offset = self.__offset [index]

      if self.__style in ['default', 'label', 'dropdown']:
        text = value or ''
        text = text [offset:offset + self.__length]
        text += ' ' * (self.__length - len (text))

      elif self.__style == 'password':
        text = '*' * len (value or '')
        text = text [offset:offset + self.__length]
        text += ' ' * (self.__length - len (text))

      elif self.__style == 'checkbox':
        if self.__value [index]:
          text = '[X]'
        else:
          text = '[ ]'

      attr = self.__getAttr (index)

      self._setText (index, text, attr, self.__selection [index])


  # ---------------------------------------------------------------------------
  # Get the current screen attributes to be used
  # ---------------------------------------------------------------------------

  def __getAttr (self, index):

    if self.__style == 'label':
      attr = self._uiDriver.attr ['background']
    elif not self.__enabled [index]:
      attr = self._uiDriver.attr ['disabled']
    elif index == self._focusIndex:
      attr = self._uiDriver.attr ['focusentry']
    else:
      attr = self._uiDriver.attr ['entry']

    return attr


  # ---------------------------------------------------------------------------
  # handle keypress
  # ---------------------------------------------------------------------------

  def _keypress (self, key):

    if self.__style == 'dropdown' and key == chr (self._uiDriver.lookupKey):
      res = self._uiDriver.getOption (u_("Select option"), self.__allowedValues)
      if res is not None:
        self._request ('REPLACEVALUE', text = res)
    else:
      UIHelper._keypress (self, key)


  # ---------------------------------------------------------------------------
  # handle function keypress
  # ---------------------------------------------------------------------------

  def _fkeypress (self, key, shift, ctrl, meta):

    if self.__style == 'listbox' and key in [curses.KEY_DOWN, curses.KEY_UP]:
      self.__move ([1, -1][key == curses.KEY_UP])
    else:
      UIHelper._fkeypress (self, key, shift, ctrl, meta)


  # ---------------------------------------------------------------------------

  def __move (self, direction):

    index = self._focusIndex
    self.__pindex [index] += direction
    self.__index [index] += direction

    if self.__pindex [index] > self.__height:
      if self.__index [index] < len (self.__data):
        self.__offset [index] += direction

    elif self.__pindex [index] < 1:
      self.__offset [index] = max (0, self.__offset [index] - 1)

    self.__index [index] = max (0, self.__index [index])
    self.__index [index] = min (len (self.__data) - 1, self.__index [index])

    self.__pindex [index] = max (1, self.__pindex [index])
    self.__pindex [index] = min (self.__pindex [index], self.__height)

    self.__value [index] = self.__data [self.__index [index]]
    self._request ('REPLACEVALUE', text = self.__data [self.__index [index]])
    

# =============================================================================
# Configuration data
# =============================================================================

configuration = {
  'baseClass'  : UIEntry,
  'provides'   : 'GFEntry',
  'container'  : 0,
  }
