#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2004 Free Software Foundation
#
# FILE:
# GFForm.py
#
# DESCRIPTION:
"""
Class that contains the internal python object representation of a GNUe Form
built from GFObjects
"""
# NOTES:
#

import sys
import string
import traceback

from gnue.common.apps import GDebug
from gnue.common import events
from gnue.common.datasources.GDataObjects import ConnectionError as DBError
from gnue.common.definitions.GRootObj import GRootObj
from gnue.common.logic.language import AbortRequest
from gnue.common.definitions.GObjects import GObj
from gnue.common.datasources import ConnectionTriggerObj
from gnue.forms.GFObjects import *
from gnue.forms import GFParser


# Defines which objects are "Tab Stops"
TabStops = ('GFEntry','GFButton')

#
# GFForm
#
# instance is passed into the initializer so that
# designer can pass that in
#
class GFForm(GRootObj, GFObj, events.EventAware):
  def __init__(self, parent=None, instance=None):
  # TODO: with a little tweaking we can now stop passing  GFParser.getXMLelements
    GRootObj.__init__(self, 'form', GFParser.getXMLelements, GFParser)
    GFObj.__init__(self, parent)

    self._type = "GFForm"
    self.name="__main__"

    # Datasources referced by this form
    self._datasourceDictionary = {}

    # Dictionary to locate named triggers
    self._triggerDictionary = {}

    # Insert/Overwrite mode
    self._insertMode = True

    # Focus information
    self._currentPage = None
    self._currentBlock = None
    self._currentEntry = None

    # set
    self._instance = instance
    self._app = self._instance #TODO: Imports are broken do to
                               #TODO: switch from _app to _instance

    # Hackery until proper layout support is added
    self._standardnamespaces = {'Char': 'GNUe:Forms:Char'}


    # The "None" init gives datasources time to setup master/detail
    self._inits = [self.primaryInit, None, self.secondaryInit]

    #
    # Trigger support
    #
    self._triggerns={}
    self._triggerGlobal = True # Put this object into the global namespace by it's name

    self._validTriggers = { 'ON-STARTUP':     'On-Startup',
                            'ON-ACTIVATION':  'On-Activation',
                            'PRE-EXIT':       'Pre-Exit',
                            'ON-EXIT':        'On-Exit',
                            'PRE-COMMIT':     'Pre-Commit',
                            'POST-COMMIT':    'Post-Commit' }

    self._triggerFunctions = {'setFocus':{'function':self.triggerSetFocus,
                                          'global': True,
                                          },
                              'getAuthenticatedUser':{'function':self.triggerGetAuthenticatedUser,
                                                      'global': True,
                                                      },
#                               'getCurrentEntry':{'function':self.triggerGetCurrentEntry,
#                                                  'global': True,
#                                                  },
#                               'getCurrentBlock':{'function':self.triggerGetCurrentBlock,
#                                                  'global': True,
#                                                  },
#                               'getCurrentPage':{'function':self.triggerGetCurrentPage,
#                                                 'global': True,
#                                                },
                              'setStatusText':{'function':self.triggerSetStatusText,
                                               'global': True,
                                               },
                              'getParameter':{'function':self.getParameter,
                                              'global': True,
                                              },
                              'setParameter':{'function':self.triggerSetParameter,
                                              'global': True,
                                              },
                              'showMessage':{'function':self.triggerShowMessageBox,
                                             'global': True,
                                             },
                              'commit':{'function':self.commit,
                                        'global': True,
                                        },
                              'rollback':{'function':self.rollback,
                                        'global': True,
                                        },
                              'close':{'function':self.triggerClose,
                                       'global': True,
                                       },
                              'getFeature':{'function':self.triggerGetFeature,
                                            'global': True,
                                            },
                              'setFeature':{'function':self.triggerSetFeature,
                                            'global': True,
                                            },
                              'runForm':{'function':self.triggerRunForm,
                                         'global': True,
                                         } ,
                              'activateDialog':{'function':self.triggerActivateDialog,
                                                'global': True,
                                                } ,
                              'activateTrigger':{'function':self.fireTrigger,
                                                'global': True,
                                                } ,
                              'initQuery':{'function':self.initQuery,
                                           },
                              }

    self._features = {
      'GUI:MENUBAR:SUPPRESS': False,
      'GUI:TOOLBAR:SUPPRESS': False,
      'GUI:STATUSBAR:SUPPRESS': False,
    }

    self._in_trigger_lock = False

  #
  # primaryInit
  #
  # Called during phaseInit startup by GParser.
  #
  def primaryInit(self):
    # Initialize our events system
    events.EventAware.__init__(self, self._instance.eventController)

    # Find the logic and layout controllers
    for child in self._children:
      if child._type == 'GFLogic':
        self._logic = child
      elif child._type == 'GFLayout':
        self._layout = child

    self.initTriggerSystem()
    ConnectionTriggerObj.addAllConnections(self._instance.connections, self._triggerNamespaceTree)
    self._triggerns.update(self._triggerNamespaceTree._globalNamespace)

    ## TODO: This ain't right!  Fix after 0.5.0
    #self._triggerns['runForm'] = self.triggerRunForm
    ## TODO


  #
  # Secondary init (stuff that must happen
  # after all our children have init'ed)
  #
  def secondaryInit(self):
    # create the first records
    for key in self._datasourceDictionary.keys():
      if not self._datasourceDictionary[key].hasMaster():
        self._datasourceDictionary[key].createEmptyResultSet()

    # Set initial focus
    self.findAndChangeFocus(self)
    self.processTrigger('On-Startup')

  #
  # Get a user parameter. If parameter not specified, pull default value
  #
  def getParameter(self, parameter):
    param = string.lower(parameter)
    try:
      return self._parameters[param]
    except KeyError:
      rv = None
      for child in self._children:
        if isinstance(child, GFParameter) and child._name == param:
          rv = child.default
          self._parameters[param] = rv
          break
    return rv

  #
  # Set a user parameter.
  #
  def triggerSetParameter(self, parameter, value):
    param = string.lower(parameter)
    self._parameters[param]=value

  def triggerActivateTrigger(self,name):
    self.processTrigger(name)

  #
  #
  # Focus functions
  #
  #

  #
  # findAndChangeFocus
  #
  # Runs findFocus and then runs
  # changeFocus with that value.
  #
  def findAndChangeFocus(self, object):
    entry = self.findFocus(object)
    if entry:
      self.changeFocus(entry)

  #
  # Find focus
  #
  # Find the next focusable item given
  # an item of type Page, Block, or Entry
  #
  # Only called by findAndChangeFocus
  #
  def findFocus(self, object):

    if object._type == 'GFField':
      GDebug.printMesg(0,"GFField was passed to 'findFocus'.")
      try:
        object = object._entryList[0]
      except IndexError:
        return None

    entry = None
    if isinstance(object, GFObj):
      if (object._type in TabStops and
          (object._navigable) and ((not object.readonly) or
         (self._currentBlock and \
          self._currentBlock.mode=='query' and \
          object._queryable))):
        return object
      else:
        if hasattr(object, '_entryList'):
          for child in object._entryList:
            entry = self.findFocus(child)
            if entry:
              return entry
        #
        # We're in trouble.  Scan all children for a
        # focusable widget.  Will dump GFField objects
        # stored in blocks into itself.
        #
        for child in object._children:
          entry = self.findFocus(child)
          if entry:
            break

    return entry

  #
  # changeFocus
  #
  # changes to the requested entry object requested by
  # an event source
  #
  def changeFocus(self, widget, fireFocusTriggers=1):

    message = None

    # cannot change focus to same entry
    if widget == self._currentEntry:
      return

    try:
      if self._currentEntry:

        if self._currentEntry._type != 'GFButton':
          val = self._currentEntry.getValue()
          if self._currentEntry._field.min_length and val is not None and len(str(val)):
            if len(str(val)) < self._currentEntry._field.min_length:
              message = _("Minimum required length %d" % self._currentEntry._field.min_length )
              # Piggybacking off the triggers message box code
              self.triggerShowMessageBox(message)
              return

        event = events.Event('endEDITMODE',_form=self)
        self.dispatchEvent(event)

        #self._instance.dispatchEvent(event)
        if event.__error__:
          return True

      fieldChange = widget != self._currentEntry
      try:
        blockChange = widget._block != self._currentBlock
      except AttributeError:
        # Buttons don't have a block, but also
        # don't trigger a block change
        blockChange = False
      pageChange = widget._page != self._currentPage

      if fireFocusTriggers:
        try:
          if fieldChange:
            self._currentEntry.processTrigger('Pre-FocusOut', ignoreAbort=False)
            if hasattr(self._currentEntry,'_field'):
              self._currentEntry._field.processTrigger('Pre-FocusOut', ignoreAbort=False)
          if blockChange:
            self._currentBlock.processTrigger('Pre-FocusOut', ignoreAbort=False)
          if pageChange:
            self._currentPage.processTrigger('Pre-FocusOut', ignoreAbort=False)
            self._currentPage.processTrigger('Post-FocusOut', ignoreAbort=False)
          if blockChange:
            self._currentBlock.processTrigger('Post-FocusOut', ignoreAbort=False)
          if fieldChange:
            self._currentEntry.processTrigger('Post-FocusOut', ignoreAbort=False)
            if hasattr(self._currentEntry,'_field'):
              self._currentEntry._field.processTrigger('Post-FocusOut', ignoreAbort=False)
        except AttributeError:
          pass

      oldEntry = self._currentEntry

      self._currentEntry = widget
      try:
        self._currentBlock = self._currentEntry._block
      except AttributeError:
        pass # Buttons, et al
      self._currentPage = self._currentEntry._page

      if pageChange:
        self.dispatchEvent('gotoPAGE',self._currentPage, _form=self);

      if blockChange:
        self.refreshDisplay(self._currentBlock)

      self.dispatchEvent('updateENTRY', oldEntry, _form=self)
      self.dispatchEvent('updateENTRY', self._currentEntry, _form=self)

      if fireFocusTriggers:
        if pageChange:
          self._currentPage.processTrigger('Pre-FocusIn', ignoreAbort=False)
        if blockChange:
          self._currentBlock.processTrigger('Pre-FocusIn', ignoreAbort=False)
        if fieldChange:
          self._currentEntry.processTrigger('Pre-FocusIn', ignoreAbort=False)
          if hasattr(self._currentEntry,'_field'):
            self._currentEntry._field.processTrigger('Pre-FocusIn', ignoreAbort=False)
          if hasattr(self._currentEntry,'_field'):
            self._currentEntry._field.processTrigger('Post-FocusIn', ignoreAbort=False)
          self._currentEntry.processTrigger('Post-FocusIn', ignoreAbort=False)
        if blockChange:
          self._currentBlock.processTrigger('Post-FocusIn', ignoreAbort=False)
        if pageChange:
          self._currentPage.processTrigger('Post-FocusIn', ignoreAbort=False)

      self.refreshUIEvents()

    except AbortRequest, t:
      GDebug.printMesg(1, "Trigger Error!")
      message = _("Trigger Exception :\n") + t.msg
    return message


  #
  #
  # Events
  #
  #

  #
  # Incoming Event handlers
  #

  def newRecord(self):
    if not self.readonly:
      self._currentBlock.newRecord()

  def deleteRecord(self):
    if not self.readonly:
      self._currentBlock.deleteRecord()
      self.refreshUIEvents()

  def isSaved(self):
    if (self._currentEntry._type != 'GFButton' and \
        self._currentEntry._displayHandler.isPending()):
      return 0

    for block in self._logic._blockList:
      if not block.isSaved():
        return 0

    return 1

  def initQuery(self):
    message = None
    if self._currentBlock.mode != 'query':
      for block in self._logic._blockList:
        if not block.isSaved():
          message = _('Data not saved. Save changes or clear the form to proceed.')
          return message

    for block in self._logic._blockList:
      block.processRollback()
      for entry in block._entryList:
        if hasattr(entry,'queryDefault'):
          GDebug.printMesg(1, "%s will be set to %s" % (entry.name, entry.queryDefault))
          entry.setValue(entry.queryDefault)

    # If Enter-Query is hit once, enter query mode
    # If Enter-Query is hit twice, bring back conditions from last query.
    # If Enter-Query is hit thrice, cancel the query and go into normal mode.

    self.dispatchEvent('beginENTERQUERY', _form=self)
    #self._instance.dispatchEvent('beginENTERQUERY', _form=self)

    for block in self._logic._blockList:
      block.initQuery()


    return message


  def cancelQuery(self):
    self.dispatchEvent('endENTERQUERY', _form=self)
    #self._instance.dispatchEvent('endENTERQUERY', _form=self)
    message = None
    for block in self._logic._blockList:
      block.cancelQuery()

    return message

  def copyQuery(self):
    if self._currentBlock.mode != 'query':
      self.initQuery()

    self.dispatchEvent('endENTERQUERY', _form=self);
    #self._instance.dispatchEvent('endENTERQUERY', _form=self);
    message = None
    for block in self._logic._blockList:
      block.copyQuery()

    return message


  def executeQuery(self):
    if self._currentBlock.mode != 'query':
      return _("Form not in query mode")

    self.dispatchEvent('endENTERQUERY', _form=self);
    self.dispatchEvent('beginWAIT',None, _form=self);
    #self._instance.dispatchEvent('endENTERQUERY', _form=self);
    #self._instance.dispatchEvent('beginWAIT',None, _form=self);

    message = None
    try:
      self._currentBlock.processQuery()

    except DBError:
      self.rollback(1)
      message = _("Database query error:\n%s\n%s ") % (sys.exc_info()[0], sys.exc_info()[1])


    self.dispatchEvent('endWAIT', _form=self)
    #self._instance.dispatchEvent('endWAIT', _form=self)
    # TODO: may need to make self._currentBlock just self
    self.refreshDisplay(self._currentBlock)



  def commit(self):
    self.dispatchEvent('beginWAIT', _form=self)
    #self._instance.dispatchEvent('beginWAIT', _form=self)

    message = None

    try:
      if not self.readonly:
        # Form level pre-commit
        try:
          self.processTrigger('Pre-Commit', ignoreAbort=False)
        except AbortRequest:
          GDebug.printMesg(1, "Trigger form Pre-Commit threw a AbortRequest!")
          self.dispatchEvent('endWAIT',None, _form=self)
          #self._instance.dispatchEvent('endWAIT',None, _form=self)
          return _("Form trigger returned error")

        for block in self._logic._blockList:
          GDebug.printMesg(1, "Saving %s"%block.name)
          try:

            # This gets lost in the Pre-Commit code
            block._precommitRecord = block._currentRecord

            block.processCommit()
          except AbortRequest:
            GDebug.printMesg(1, "Trigger block Pre-Commit threw a AbortRequest!")
            self.dispatchEvent('endWAIT',None, _form=self)
            #self._instance.dispatchEvent('endWAIT',None, _form=self)
            return _("Block trigger returned error")
          block.processTrigger('Post-Commit')

        for block in self._logic._blockList:
            block.switchRecord(0)

        self.dispatchEvent('cannotCOMMIT')
        self.dispatchEvent('cannotROLLBACK')

      else:
        message = _('Form is readonly')
        #self._instance.dispatchEvent('endWAIT',None, _form=self)
        self.dispatchEvent('endWAIT',None, _form=self)

        return message

      self.processTrigger('Post-Commit')

    except DBError:
      self.rollback(1)
      message = _("Database commit error:\n%s\n%s ") % (sys.exc_info()[0], sys.exc_info()[1])
    except:
      print _("\n\nGFForm: Unexpected Exception:")
      print '-'*60
      traceback.print_exc(file=sys.stdout)
      print '-'*60

    #self._instance.dispatchEvent('endWAIT',None, _form=self)
    self.dispatchEvent('endWAIT',None, _form=self)

    return message

  def rollback(self, recover=0):
    self.endEditing()
    for block in self._logic._blockList:
      block.processRollback(recover)
    self.refreshDisplay(self)
    self._currentBlock.jumpRecord(self._currentBlock._currentRecord)
    self.dispatchEvent('cannotCOMMIT')
    self.dispatchEvent('cannotROLLBACK')

  #
  # fireTrigger
  #
  def fireTrigger(self, triggerName):
    self._triggerDictionary[triggerName](self)


  #
  #
  #
  def updateUIEntry(self,field):
    for entry in field._entryList:
      self.dispatchEvent('updateENTRY',entry, _form=self)
      #self._instance.dispatchEvent('updateENTRY',entry, _form=self)

  #
  #
  #

  #
  # nextEntry
  #
  # Called whenever an event source has requested that the
  # focus change to the next data entry object
  #
  def nextEntry(self, reverse=False, first=False, onlyInBlock=False):

    if self._currentBlock.transparent and not onlyInBlock:
      source = self._currentEntry._page._focusOrder[:]
    else:
      source = self._currentBlock._focusOrder[:]

    # If we want the previous entry, then reverse the focusorder we're using
    if reverse:
      source.reverse()

    nextEntry = None
    firstEntry = None
    keepNext = False
    for object in source:

      if ((object._navigable) and ((not object.readonly) or
         (self._currentBlock.mode=='query' and object._queryable))):

        # If we only wanted the first navigable field in the block, then return
        if first:
          nextEntry = object
          break

        # Put the first field as the next to rollover
        if nextEntry == None:
          nextEntry = object
          firstEntry = object

        # If we're at the current focused entry,
        # then the next entry will be what we want
        if object == self._currentEntry:
          keepNext = True

        # If we've already passed the current entry
        # Then this is the entry to return
        elif keepNext:
          nextEntry = object
          break

    if nextEntry == firstEntry and self._currentBlock.transparent and self._currentPage.transparent:
      # Jump to the next/(previous) page if block is page as transparent
      pages =  self._layout._pageList
      i = pages.index(self._currentPage)

      if reverse:
        try:
          dest = self._layout._pageList[i-1]
        except IndexError:
          dest = self._layout._pageList[-1]
        # TODO: this fails if last entry is not navigable
        self.findAndChangeFocus(dest._entryList[-1])
      else:
        try:
          dest = self._layout._pageList[i+1]
        except IndexError:
          dest = self._layout._pageList[0]
        self.findAndChangeFocus(dest)

    else:
      self.changeFocus(nextEntry)

  #
  # previousEntry
  #
  # Called whenever an event source has requested that the
  # focus change to the next data entry object
  #
  def previousEntry(self):
    return self.nextEntry(reverse=True)

  #
  # refreshDisplay
  #
  # Refreshes all the UI fields to match
  # the virtual form's fields
  #
  def refreshDisplay(self, block):
    block.walk(self.__refreshDisplay)
    self.refreshUIEvents()

  def __refreshDisplay(self, object):
    if object._type == 'GFEntry' or object._type == "GFImage":
      self.dispatchEvent('updateENTRY',object, _form=self)

  #
  # Signal the UI Drivers of navigation button relevance
  #
  def refreshUIEvents(self):
    block = self._currentBlock
    if not block: return

    dispatchEvent = self.dispatchEvent
    if block.mode == 'query':
      dispatchEvent('canCANCELQUERY')
      dispatchEvent('canEXECQUERY')
      dispatchEvent('cannotENTERQUERY')
      dispatchEvent('cannotPREVRECORD')
      dispatchEvent('cannotFIRSTRECORD')
      dispatchEvent('cannotNEXTRECORD')
      dispatchEvent('cannotLASTRECORD')
      dispatchEvent('cannotCOMMIT')
      dispatchEvent('cannotROLLBACK')
      dispatchEvent('cannotJUMPPROMPT')
      dispatchEvent('cannotNEWRECORD')
      dispatchEvent('cannotMARKFORDELETE')
    else:
      dispatchEvent('canENTERQUERY')
      dispatchEvent('cannotCANCELQUERY')
      dispatchEvent('cannotEXECQUERY')
##      dispatchEvent('canNEXTRECORD')

      # TODO: There should probably be some tests here....
      if not self.isSaved():
        dispatchEvent('canCOMMIT')
      else:
        dispatchEvent('cannotCOMMIT')

      dispatchEvent('canROLLBACK')
      dispatchEvent('canJUMPPROMPT')
      dispatchEvent('canNEWRECORD')
      dispatchEvent('canMARKFORDELETE')

      if block._resultSet.isFirstRecord():
        dispatchEvent('cannotPREVRECORD')
        dispatchEvent('cannotFIRSTRECORD')
      else:
        dispatchEvent('canPREVRECORD')
        dispatchEvent('canFIRSTRECORD')

      if block._resultSet.isLastRecord():
        dispatchEvent('cannotLASTRECORD')
        if block._resultSet.current.isEmpty():
          dispatchEvent('cannotNEXTRECORD')
          dispatchEvent('cannotNEWRECORD')
        else:
          dispatchEvent('canNEWRECORD')
      else:
        dispatchEvent('canNEXTRECORD')
        dispatchEvent('canLASTRECORD')
        dispatchEvent('canNEWRECORD')

  #
  # nextBlock
  #
  # Called whenever an event source has requested that the
  # focus change to the next data entry block
  #
  def nextBlock(self):

    try:
      nextBlock = self._logic._blockList[self._logic._blockList.index(self._currentBlock)+1]
    except IndexError:
      nextBlock = self._logic._blockList[0]
    self.findAndChangeFocus(nextBlock)



  #
  # findPreviousBlock
  #
  # Convenience routine to find the previous block
  #
  def findPreviousBlock(self):
    try:
      return self._logic._blockList[self._logic._blockList.index(self._currentBlock)-1]
    except IndexError:
      return self._logic._blockList[-1]


  #
  # previousBlock
  #
  # Called whenever an event source has requested that the
  # focus change to the next data entry block
  #
  def previousBlock(self):
    self.findAndChangeFocus(self.findPreviousBlock())


  #
  # endEditing
  #
  # Signal to the current Entry to stop editing
  # mode and save it's value to the virtual form
  #
  def endEditing(self):
    event = events.Event('endEDITMODE',None, _form=self)
    self.dispatchEvent(event)
    return not event.__error__


  def prevRecord(self):
#    if self._currentBlock.mode == 'query':
#      GDebug.printMesg(5,'Cannot go to previous record: in query mode!')
#      return
    self._currentBlock.prevRecord()

  def firstRecord(self):
#    if self._currentBlock.mode == 'query':
#      GDebug.printMesg(5,'Cannot go to first record: in query mode!')
#      return
    self._currentBlock.firstRecord()

  def lastRecord(self):
#    if self._currentBlock.mode == 'query':
#      GDebug.printMesg(5,'Cannot go to last record: in query mode!')
#      return
    self._currentBlock.lastRecord()

  def nextRecord(self):
#    if self._currentBlock.mode == 'query':
#      GDebug.printMesg(5,'Cannot go to next record: in query mode!')
#      return
    self._currentBlock.nextRecord()

  def jumpRecord(self,count):
#    if self._currentBlock.mode == 'query':
#      GDebug.printMesg(5,'Cannot jump to record: in query mode!')
#      return
    self._currentBlock.jumpRecord(count)

  def toggleInsertMode(self):
    self._insertMode = not self._insertMode

  #
  #
  # Trigger functions
  #
  # The following functions are only used to implement
  # functions exposed in the trigger namespace.
  #
  #

  #
  # triggerShowMessageBox
  #
  # display a standard message box
  #
  def triggerShowMessageBox(self,msgtxt):
    self._instance.displayMessageBox(msgtxt)

  #
  # triggerSetFocus
  #
  # allow the trigger to switch input focus
  # to a specfic entry widget
  #
  def triggerSetFocus(self,object):
    # add global focus locking
    if self._in_trigger_lock:
      print "Already called by a trigger"
      return

    self._in_trigger_lock = True

    focus = object._object
    if focus._type=='GFField':
      try:
        focus = focus._entryList[0]
      except KeyError:
        raise "setFocus failed: GFField is not bound to an GFEntry object."

    if focus._type != 'GFEntry':
      raise "setFocus failed: Can just switch to GFEntry objects. " +\
            "You passed a '%s' object." % focus._type
    self.changeFocus(focus,1)
    self.dispatchEvent('gotoENTRY',object=self._currentEntry,_form=self)

    self._in_trigger_lock = False

#   #
#   # allow the trigger to get a handle to
#   # an actual GFEntry object
#   #
#   # TODO: WHY?  This seems like a good way to write
#   # TODO: triggers that break due to internal forms changes
#   #
#   def triggerGetCurrentEntry(self):
#     return self._currentEntry
# 
# 
#   #
#   # allow the trigger to get a handle to
#   # an actual GFBlock object
#   #
#   # TODO: WHY?  This seems like a good way to write
#   # TODO: triggers that break due to internal forms changes
#   #
#   def triggerGetCurrentBlock(self):
#     return self._currentBlock
# 
#   #
#   # allow the trigger to get a handle to
#   # an actual GFPage object
#   #
#   # TODO: WHY?  This seems like a good way to write
#   # TODO: triggers that break due to internal forms changes
#   #
#   def triggerGetCurrentPage(self):
#     return self._currentPage
 
  #
  # triggerSetStatusText
  #
  # Allows the trigger to display a custom message
  # on the forms status bar
  #
  def triggerSetStatusText(self,tip=''):
    self.dispatchEvent('uiUPDATESTATUS',tip=tip, _form=self)

  #
  # triggerRunForm
  #
  # Allows the trigger to launch a new instance
  # of gnue-forms running a different form.
  #
  def triggerRunForm(self, fileName):
    from GFInstance import GFInstance
    instance = GFInstance(self._instance.manager,
                          self._instance.connections,
                          self._instance._uimodule,  1)
    instance.addFormFromFile(fileName)
    instance.activate()

  #
  # triggerActivateDialog
  #
  # Allows the trigger to launch a standard or custom
  # dialog.  Values can be passed back and forth thru
  # the parameters dictionary
  #
  def triggerActivateDialog(self,dialogName, parameters={}, modal=0):
    self._instance.activateForm(dialogName,parameters, modal)

  #
  # triggerClose
  #
  # Allows the trigger to close this copy of gnue-forms
  #
  def triggerClose(self):
    event = events.Event('requestEXIT', _form=self)
    #self._instance.dispatchEvent(event)
    self.dispatchEvent(event)
    if event.__error__:
      return 1

  #
  # triggerSetFeature
  #
  # Allows the trigger to set feature values
  # Features are things like toolbars, menubars, and statusbar
  #
  def triggerSetFeature(self, feature, value):
    if not self._features.has_key(feature):
      raise KeyError, "Trigger attempted to set unknown feature %s" % feature
    else:
      self._features[feature] = value

  #
  # triggerGetFeature
  #
  # Allows the trigger to get feature values
  # Features are things like toolbars, menubars, and statusbar
  #
  def triggerGetFeature(self, feature):
    try:
      return self._features[feature]
    except KeyError:
      raise KeyError, "Trigger attempted to get unknown feature %s" % feature

  #
  # triggerGetAuthenticatedUser
  #
  # TODO: Description
  #
  def triggerGetAuthenticatedUser(self, connection=None):
    return self._instance.connections.getAuthenticatedUser(connection)


