#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2004 Free Software Foundation
#
# FILE:
# UserAction.py
#
# DESCRIPTION:
# A generic UIdriver base for common message-based GUI toolkits.
#
# NOTES:
#

from gnue.common.apps import GConfig
from gnue.forms import GFKeyMapper
import os

_iconpath = GConfig.getInstalledBase('form_images', 'common_images')


######################################################################
#
#
#

class UserAction:

  def __init__(self, event, description, help, canToggle=0):
    self.event = event
    self.canToggle = canToggle
    self.__iconloc = {}

    # Description is short; appropriate for mouse-over tips
    self.description = description

    # Help is more detailed; suitable for a manual or online help
    self.help = help

  def getIconLocation(self, format="png",size="32x32"):
    try:
      return self.__iconloc[size + format]
    except:
      loc = os.path.join(_iconpath, 'forms',
          '%s-%s.%s' % (self.event.lower(),size, format))
      if not os.path.isfile(loc):
        loc = None

      self.__iconloc[size + format] = loc
      return loc

  def getHotKeyText(self, metamappings={}, separator='+'):
    if metamappings:
      m = copy.copy(_baseKeyMap)
      m.update(metamappings)
    else:
      m = _baseKeyMap
    return GFKeyMapper.KeyMapper.getEventKeystrokeRepr(self.event,
        metamappings=m, separator=separator)



######################################################################
#
# Build a map of the standard user actions
#

userActionMap = {}

for action in [

   UserAction(
       event="COMMIT",
       description=_("Save all changes to the database."),
       help=_("""Save all changes to the database.""") ),

    UserAction(
       event="ROLLBACK",
       description=_("Clear form."),
       help=_("Clear form and revert back to your original data.") ),
       
    UserAction(
       event="PRINTOUT",
       description=_("Perform print routine for this form."),
       help=_("Perform print routine for this form.") ), 
       
    UserAction(
       event="EXIT",
       description=_("Leave the application."),
       help=_("Leave the application.") ), 
       
    UserAction(
       event="CUT",
       description=_("Cut the selected text"),
       help=_("Cut the selected text") ), 
       
    UserAction(
       event="COPY",
       description=_("Copy the selected text"),
       help=_("Copy the selected text") ), 
       
    UserAction(
       event="COPYRECORD",
       description=_("Copy the current record to the clipboard."),
       help=_("Copy the current record to the clipboard. The record is copied with tabs separating values.") ), 
       
    UserAction(
       event="COPYSCREEN",
       description=_("Copy the current screen to the clipboard as plain text."),
       help=_("Copy the current screen to the clipboard as plain text.") ), 
       
    UserAction(
       event="PASTE",
       description=_("Paste text into the current field."),
       help=_("Paste text into the current field.") ), 
       
    UserAction(
       event="SELECTALL",
       description=_("Select all text."),
       help=_("Select all text.") ), 

    UserAction(
       event="PREVENTRY",
       description=_("Move to the previous logicalfield."),
       help=_("Move to the previous logical field.") ), 
       
    UserAction(
       event="NEXTENTRY",
       description=_("Move to the next logicalfield."),
       help=_("Move to the next logical field.") ), 
       
    UserAction(
       event="FIRSTRECORD",
       description=_("Jump to the first record in the current block."),
       help=_("Jump to the first record in the current block.") ),

    UserAction(
       event="PREVRECORD",
       description=_("Jump to the previous record in the current block."),
       help=_("Jump to the previous record in the current block.") ),

    UserAction(
       event="NEXTRECORD",
       description=_("Jump to the next record in the current block."),
       help=_("Jump to the next record in the current block.") ),

    UserAction(
       event="LASTRECORD",
       description=_("Jump to the last record in the current block."),
       help=_("Jump to the last record in the current block.") ),

    UserAction(
       event="JUMPPROMPT",
       description=_('Prompts for a record number to which the system should jump.'),
       help=_('Prompts for a record number to which the system should jump.') ),

    UserAction(
       event="NEWRECORD",
       description=_("Insert a new record into the current block."),
       help=_("Insert a new record into the current block.") ),

    UserAction(
       event="MARKFORDELETE",
       description=_('Mark record for removal at next commit.'),
       help=_('Mark record for removal at next commit.') ),

    UserAction(
       event="NEXTBLOCK",
       description=_('Navigate to the next data block.'),
       help=_('Navigate to the next data block.') ),

    UserAction(
       event="PREVBLOCK",
       description=_('Navigate to the previous data block.'),
       help=_('Navigate to the previous data block.') ),

    UserAction(
       event="NEXTPAGE",
       description=_('Navigate to the next page.'),
       help=_('Navigate to the next page.') ),

    UserAction(
       event="PREVPAGE",
       description=_('Navigate to the previous page.'),
       help=_('Navigate to the previous page.') ),

    UserAction(
       event="ENTERQUERY",
       description=_('Switch to query mode. Select Execute Query once your criteria has been entered.'),
       help=_('Switch to query mode. Select Execute Query once your criteria has been entered.'),
       canToggle='CANCELQUERY' ),

    UserAction(
       event="COPYQUERY",
       description=_('Switch to query mode (if not already) and retrieve the last query parameters. Select Execute Query once your criteria has been entered.'),
       help=_('Switch to query mode (if not already) and retrieve the last query parameters. Select Execute Query once your criteria has been entered.') ), 
    
    UserAction(
       event="CANCELQUERY",
       description=_('Cancel query mode.'),
       help=_('Cancel query mode.') ), 
    
    UserAction(
       event="EXECQUERY",
       description=_('Perform a query and show the results.'),
       help=_('Perform a query and show the results.') ),

    UserAction(
       event="ABOUT",
       description=_('Display info about GNUe Forms.'),
       help=_('Display info about GNUe Forms.') ),

    UserAction(
       event="HELPKEYS",
       description=_('Display the keystroke bindings currently in use.'),
       help=_('Display the keystroke bindings currently in use.') ),

  ]:

  userActionMap[action.event] = action


_baseKeyMap = {
  'META': 'Alt',
  'CONTROL': 'Ctrl',
  'SHIFT': 'Shift',
  'INSERT': 'Ins',
  'DELETE': 'Del',
  'HOME': 'Home',
  'END': 'End',
  'PAGEUP': 'PgUp',
  'PAGEDOWN': 'PgDn',
  'UP': 'Up',
  'DOWN': 'Down',
  'LEFT': 'Left',
  'RIGHT': 'Right',
  'TAB': 'Tab',
  'ENTER': 'Enter',
  'RETURN': 'Enter',
  'BACKSPACE': 'Back' }
