#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2004 Free Software Foundation
#
# FILE:
# win32/UILoginHandler.py
#
# DESCRIPTION:
# A PyWin32 based user interface driver for GNUe forms.
#
# NOTES:
#

import win32api, win32gui, win32con, commctrl
import os, struct

from gnue.common.datasources import GLoginHandler
from gnue.common.apps import GConfig
from gnue.forms.uidrivers.win32.common import getNextId, centerWindow
from gnue.forms.uidrivers.win32.ErrorHandler import handleStartupError

g_registeredClass = 0

# Shortcut
images_dir = GConfig.getInstalledBase('forms_images','common_images') + '/'

BUTTON = 0x0080
EDIT = 0x0081
STATIC = 0x0082

#####################################################################
##
## Login Support
##
## Everything below this point is in support of the win32 drivers
## UILoginHandler
##  
#####################################################################

#
# UILoginHandler
#
class UILoginHandler(GLoginHandler.LoginHandler):
  def __init__(self):
    win32gui.InitCommonControls()
    self.hinst = win32api.GetModuleHandle(None)

    _font = win32gui.GetStockObject(win32con.SYSTEM_FONT)
    fn = win32gui.GetObject(_font)
    #print fn.lfHeight, fn.lfWidth,fn.lfCharSet,fn.lfPitchAndFamily
    self.charWidth,self.charHeight = fn.lfWidth, fn.lfHeight

  def getLogin(self, loginData, errortext=None):#"Invalid username/password"):
    if 1: # TODO: This is left here to prevent me from having to unindent this code
          # TODO: Since the UI stuff is currently being gutted I'm not wasting the time
          # TODO: to do the right thingData.Items()
      self._loginData = loginData
      self._errortext = errortext
      
      if len(loginData[1]):
        loginMesg = _('Login required for\n"%s"') % (loginData[1])
      else:
        loginMesg = _('Login required for %s') % (loginData[0])

      if os.path.isabs(gConfigForms('loginBMP')):
        imageFile = gConfigForms('loginBMP')
      else:
        imageFile = images_dir+gConfigForms('loginBMP')

      if not imageFile[-3:].upper() == 'BMP':
        handleStartupError(_("The win32 forms driver supports only BMP format logo files!"))

      self._bmp = win32gui.LoadImage(0, imageFile, win32con.IMAGE_BITMAP, 
                                          0, 0, win32con.LR_LOADFROMFILE | win32con.LR_DEFAULTSIZE)

      message_map = {
            win32con.WM_COMMAND: self.OnCommand,
            win32con.WM_INITDIALOG: self.OnInitDialog,
            win32con.WM_CLOSE: self.OnClose,
            }
      dlgClassName = self._RegisterWndClass()

      style = win32con.WS_POPUP | win32con.WS_VISIBLE | win32con.WS_CAPTION | \
                win32con.DS_SETFONT | win32con.WS_SYSMENU

      cs = win32con.WS_CHILD | win32con.WS_VISIBLE
      title = _("GNU Enterprise: Login to %s") % loginData[0]

      dlg = [ [title, (0, 0, 0, 0), style, None, (8, "MS Sans Serif"), None, dlgClassName], ]

      self.ID_BITMAP = getNextId()
      dlg.append([STATIC, "", self.ID_BITMAP, \
                       (0, 0, 0, 0), \
                       cs | win32con.SS_BITMAP]) # , win32con.SS_BLACKFRAME

      self.ID_MESG = getNextId()
      dlg.append([STATIC, loginMesg, self.ID_MESG, \
                       (0, 0, 100, (self.charHeight-4)*(loginMesg.count('\n')+1)), \
                       cs | win32con.SS_CENTER])

      self.textctrlList = []
      self.labelList = []

      for prompt in loginData[2]:
        ID = getNextId()
        dlg.append([STATIC, '%s:' % prompt[1], ID, \
                         (0, 0, (self.charWidth-2)*len(prompt[1]), self.charHeight-4), \
                         cs | win32con.SS_LEFT])
        self.labelList.append(ID)

        s = cs | win32con.WS_TABSTOP | win32con.WS_BORDER
        ID = getNextId()
        if prompt[2]:
          dlg.append([EDIT, None, ID, (0, 0, 100, self.charHeight-4), \
                          s | win32con.ES_PASSWORD])
        else:
          dlg.append([EDIT, None, ID, (0, 0, 100, self.charHeight-4), s])
        self.textctrlList.append(ID)

      if self._errortext:
        self.ID_ERROR = getNextId()
        dlg.append([STATIC, str(self._errortext), self.ID_ERROR, \
                        (0, 0, 210, (self.charHeight-4)*(self._errortext.count('\n')+2)), \
                        cs | win32con.SS_CENTER])
        #errorField.SetForegroundColour(wxColour(223,0,0))

      # Buttons
      login = _('Login')
      cancel = _('Cancel')
      buttonWidth = max(50, (self.charWidth-2)*len(login), (self.charWidth-2)*len(cancel))
      s = cs | win32con.WS_TABSTOP
      self.ID_LOGIN = getNextId()
      dlg.append([BUTTON, login, self.ID_LOGIN, \
                      (0, 0, buttonWidth, self.charHeight-2), \
                      s | win32con.BS_DEFPUSHBUTTON]) \

      s = s | win32con.BS_PUSHBUTTON
      self.ID_CANCEL = getNextId()
      dlg.append([BUTTON, cancel, self.ID_CANCEL, \
                      (0, 0, buttonWidth, self.charHeight-2), \
                      s])

    # If user cancels, this will be set to 0
    self._completed = 0
    self._rv = {}
    win32gui.DialogBoxIndirect(self.hinst, dlg, 0, message_map)

    if not self._completed:
      raise GLoginHandler.UserCanceledLogin

    return self._rv


  def _RegisterWndClass(self):
        className = "GNUe login"
        global g_registeredClass
        if not g_registeredClass:
            message_map = {}
            wc = win32gui.WNDCLASS()
            wc.SetDialogProc() # Make it a dialog class.
            self.hinst = wc.hInstance = win32api.GetModuleHandle(None)
            wc.lpszClassName = className
            wc.style = win32con.CS_VREDRAW | win32con.CS_HREDRAW
            wc.hCursor = win32gui.LoadCursor( 0, win32con.IDC_ARROW )
            wc.hIcon = win32gui.LoadIcon(0, win32con.IDI_APPLICATION)
            wc.hbrBackground = win32con.COLOR_WINDOW
            wc.lpfnWndProc = message_map # could also specify a wndproc.
            wc.cbWndExtra = win32con.DLGWINDOWEXTRA + struct.calcsize("Pi")
            classAtom = win32gui.RegisterClass(wc)
            g_registeredClass = 1
        return className


  def OnInitDialog(self, hwnd, msg, wparam, lparam):
    self.hwnd = hwnd

    bmCtrl = win32gui.GetDlgItem(self.hwnd, self.ID_BITMAP)
    win32gui.SendMessage(bmCtrl, win32con.STM_SETIMAGE, win32con.IMAGE_BITMAP, self._bmp)

    dlgWidth = max(self.Width(self.ID_BITMAP), self.Width(self.ID_MESG)+20)
    dlgHeight = self.Height(self.ID_BITMAP) + self.Height(self.ID_MESG) + 95

    xSpacing = 0
    ySpacing = 0
    fieldLabelWidth = 0

    for i in range(len(self.textctrlList)):
      t =self.textctrlList[i]
      s = self.labelList[i]

      fieldLabelWidth = max(fieldLabelWidth, \
                      self.Width(s) + self.Width(t) + 10)

      dlgWidth = max(dlgWidth, \
                      self.Width(s) + self.Width(t) + 20)

      xSpacing = max(xSpacing, self.Width(s))
      ySpacing = max(ySpacing, self.Height(s))
      ySpacing = max(ySpacing, self.Height(t))

    dlgWidth = max(dlgWidth, self.Width(self.ID_LOGIN) +
                        self.Width(self.ID_CANCEL) + 6) + 20

    dlgHeight += max(self.Height(self.ID_LOGIN),
                   self.Height(self.ID_CANCEL)) -6

    if self._errortext:
      dlgWidth = max(dlgWidth, self.Width(self.ID_ERROR)+10)
      dlgHeight += self.Height(self.ID_ERROR) + 10

    firstY = self.Height(self.ID_BITMAP) + self.Height(self.ID_MESG) + 40
    lastY = firstY
    xSpacing += 3 # Add whitespace between widgets
    ySpacing += 6 # Add whitespace between widgets
    xPos = dlgWidth/2 - fieldLabelWidth/2

    self.SetPosition(self.ID_BITMAP,
                (dlgWidth - self.Width(self.ID_BITMAP))/2,
                12)

    self.SetPosition(self.ID_MESG,
                (dlgWidth - self.Width(self.ID_MESG))/2,
                30 + self.Height(self.ID_BITMAP))

    # Move the fields and labels into position
    for i in range(len(self.textctrlList)):
      dlgHeight = dlgHeight + ySpacing
      self.SetPosition(self.labelList[i], xPos, lastY)
      self.SetPosition(self.textctrlList[i], xPos + xSpacing, lastY)
      lastY = lastY + ySpacing

    if self._errortext:
      self.SetPosition(self.ID_ERROR,
              dlgWidth/2 - self.Width(self.ID_ERROR)/2,
              lastY+3)

    self.SetPosition(self.ID_CANCEL,
                dlgWidth - 20 - self.Width(self.ID_CANCEL),
                dlgHeight - 35 - max(self.Height(self.ID_LOGIN),
                                            self.Height(self.ID_CANCEL)))
    self.SetPosition(self.ID_LOGIN,
                dlgWidth - 26 - self.Width(self.ID_CANCEL) - \
                self.Width(self.ID_LOGIN),
                dlgHeight - 35 - max(self.Height(self.ID_LOGIN),
                                            self.Height(self.ID_CANCEL)))

    # resize the dialog
    win32gui.SetWindowPos(hwnd, 0,
                              0, 0,
                              dlgWidth, dlgHeight+8,
                              win32con.SWP_NOACTIVATE | win32con.SWP_NOZORDER)
    # center the dialog
    centerWindow(self.hwnd)

  def OnClose(self, hwnd, msg, wparam, lparam):
        win32gui.EndDialog(hwnd, 0)


  def OnCommand(self, hwnd, msg, wparam, lparam):
        id = win32api.LOWORD(wparam)
        if id == self.ID_LOGIN:
          for i in range(0, len(self._loginData[2])):
            dlgItem = win32gui.GetDlgItem(self.hwnd, self.textctrlList[i])
            self._rv[self._loginData[2][i][0]] = win32gui.GetWindowText(dlgItem)
          self._completed = 1
          win32gui.EndDialog(hwnd, 0)
        elif id == self.ID_CANCEL:
          self._completed = 0
          win32gui.EndDialog(hwnd, 0)


  def Width(self, id):
    item = win32gui.GetDlgItem(self.hwnd, id)
    l,t,r,b = win32gui.GetWindowRect(item)
    return r-l

  def Height(self, id):
    item = win32gui.GetDlgItem(self.hwnd, id)
    l,t,r,b = win32gui.GetWindowRect(item)
    return b-t

  def SetPosition(self, id, x, y):
    win32gui.SetWindowPos(win32gui.GetDlgItem(self.hwnd, id),
                              0,
                              x, y,
                              self.Width(id), self.Height(id),
                              win32con.SWP_NOACTIVATE | win32con.SWP_NOZORDER)
