#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2004 Free Software Foundation
#
# FILE:
# PrintForm.py
#
# DESCRIPTION:
# Handles the default "Print" option for the wx client
#
# NOTES:
# This works by copying from the wxWindowDC to a wxPrintDC
# using a memory copy (blit)
#
# TODO: Handle scaling if image won't fit on the page
#


__all__ = ['printForm']

from wxPython.wx import *
from gnue.forms import VERSION
import time

def printForm(form, driver):
  _Timer(_printForm, form, driver).Start(100,1)

def _printForm(form, driver):
  # Get a suitable title for the print job
  # We default to either "Form Name" or "Form Name" + "Page Name"
  # depending on if the page has a caption or not.
  subtitle = ""
  try:
    title = form.title
  except AttributeError:
    title = 'Untitled Form'
  try:
    if form._currentPage.caption:
      subtitle = ' (' + form._currentPage.caption + ')'
  except AttributeError:
    pass

##  window = driver._gfObjToUIWidget[form._currentPage]._containerToolkitWidget
  window = driver._gfObjToUIWidget[form].mainWindow

  printData = wxPrintData()
  printData.SetPaperId(wxPAPER_LETTER)

  # We have to get the DC snapshot before the
  # print dialog appears as for some reason,
  # you'll get garbage otherwise.

  src = wxClientDC(window)
  w, h = window.GetClientSizeTuple()
  bmp = wxEmptyBitmap(w, h)
  dest = wxMemoryDC()
  dest.SelectObject(bmp)
  dest.Blit(0,0, w, h, src, 0, 0)
##  dest.SelectObject(wxNullBitmap)

  # Now, back to printing...
  prd = wxPrintDialogData()
  prd.SetPrintData(printData)
  prn = wxPrinter(prd)

  printout = FormPrintout(window, title, subtitle,
       form.triggerGetAuthenticatedUser() or 'Anonymous', dest)

  if not prn.Print(driver._gfObjToUIWidget[form].mainWindow, printout):
    wxMessageBox(_("There was a problem printing.\nPlease verify your printer setup and try again."), _("Printing"), wxOK)

  printout.Destroy()


class FormPrintout(wxPrintout):

  def __init__(self, window, title, subtitle, login, dc):
    self.__window = window
    self.__title = title
    self.__subtitle = subtitle
    self.__login = login
    self.__dc = dc
    wxPrintout.__init__(self, title)

  def OnPrintPage(self, page):
    dest = self.GetDC()
    dest.StartDoc (self.__title + (self.__subtitle and ' (%s)' % self.__subtitle or ''))
    dest.StartPage()
    dest.SetFont(wxNORMAL_FONT)
    dest.SetPen(wxBLACK_PEN)
    dest.SetBrush(wxTRANSPARENT_BRUSH)


    src = self.__dc
    w, h = src.GetSizeTuple()
    screen_ppi_x, screen_ppi_y = self.GetPPIScreen()
    printer_ppi_x, printer_ppi_y = self.GetPPIPrinter()
    scale_x = printer_ppi_x / screen_ppi_x
    scale_y = printer_ppi_y / screen_ppi_y

    # Get page width/height, in mm
    page_width, page_height = self.GetPageSizePixels()


    # 3/4" margins
    page_margin_left = page_margin_right = int(printer_ppi_y*.75+.5)
    page_margin_top  = int(printer_ppi_y*.75+.5)
    page_margin_bottom = int(page_margin_top/2)
    page_canvas_x = page_width - page_margin_left - page_margin_right
    page_canvas_y = page_height - page_margin_top - page_margin_bottom

    # Draw a header:
    #   Form Name                                  <Timestamp>
    #   Page Caption                          Login: User Name
    #   ------------------------------------------------------

    # Form and page title
    x, y = self.drawText(dest, page_margin_left, page_margin_top, self.__title)
    x, y = self.drawText(dest, x, y, self.__subtitle)

    # Timestamp and login
    timestamp = time.strftime("%Y-%m-%d %H:%M:%S",time.localtime(time.time()))
    x, y = self.drawText(dest, page_width - page_margin_right, page_margin_top,
                         timestamp, rightAlign=1)
    x, y = self.drawText(dest, x, y, _('Login: ') + self.__login, rightAlign=1)

    # Draw the border line
    dest.DrawLine(page_margin_left, y, page_width - page_margin_right, y)

    # .. and the actual screenshot
    y += page_margin_top
    dest.Blit(int((page_canvas_x - w)/2)+page_margin_left,y, w, h, src, 0, 0)

    # .. with a border
    dest.DrawRectangle(int((page_canvas_x - w)/2)+page_margin_left, y, w, h)

    # Finish up with the footer
    y = page_height - page_margin_bottom
    self.drawText(dest, page_margin_left, y, 'GNUe Forms %s' % VERSION,  bottomAlign=1)
    self.drawText(dest, page_width - page_margin_right, y, _('Page 1 of 1'),  rightAlign=1, bottomAlign=1)

    dest.EndPage()
    dest.EndDoc()


  def drawText(self, dc, x, y, text, rightAlign=0, bottomAlign=0):
    w, h = dc.GetTextExtent(text)
    dc.DrawText(text, x - (rightAlign and w or 0), y - (bottomAlign and h or 0))
    return (x + (not rightAlign and w or 0), y + int(h * 1.3))


class _Timer(wxTimer):
    def __init__(self, method, *args, **params):
      self.__method = method
      self.__args = args
      self.__params = params
      wxTimer.__init__(self)

    def Notify(self):
      self.__method(*self.__args, **self.__params)
