#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2002-2003 Free Software Foundation
#
# FILE:
# GNObjects.py
#
# DESCRIPTION:
# Set of classes that implement GNUe Navigator's xml-objects
#
# NOTES:
#

import sys, string, os, os.path
from gnue.common.definitions import GObjects, GRootObj
from gnue.common.apps import GDebug, GConfig
from gnue import paths
import GNParser
try: 
  from gnue.reports.base import GREngine,GRExceptions
  REPORTS_SUPPORT=1
except ImportError:
  REPORTS_SUPPORT=0

# Base class for all Navigator objects
class GNObject(GObjects.GObj):
  pass


class GNProcesses(GRootObj.GRootObj, GNObject):

  def __init__(self, parent=None):
    GRootObj.GRootObj.__init__(self,'processes',GNParser.getXMLelements, GNParser)
    GNObject.__init__(self, parent, type="GNProcesses")
    self._clientHandlers = {
      'form': self._runForm,
      'report': self._runReport,
      'app': self._runApp,
    }

  def setClientHandlers(self, handlers):
    self._clientHandlers.update(handlers)

  def _runForm(self, step):
    formCommand = gConfigNav('RunFormCommand')

    if os.path.basename(step.location) == step.location:
      try:
        formdir = gConfigNav('FormDir')
      except KeyError:
        formdir = ""
      formfile = os.path.join (paths.data, formdir, step.location)
    else:
      formfile = step.location

    if sys.platform != 'win32':
      GDebug.printMesg(1,'Running command " %s %s"'%(formCommand,formfile))
      os.spawnlpe(os.P_NOWAIT,formCommand,formCommand,formfile,os.environ)
    else:
      command = string.split("%s"%(formCommand))
      command.append('"%s"' % formfile)

      GDebug.printMesg(1,'Running command " %s "'%(command))
      os.spawnve(os.P_NOWAIT,command[0],command,os.environ)

  def _runApp(self, step):

    GDebug.printMesg(1,'Running app "%s"'%(step.location))

    command = string.split(step.location)

    if sys.platform != 'win32':
      os.spawnvpe(os.P_NOWAIT,command[0],command,os.environ)
    else:
      # TODO: Not tested
      os.spawnve(os.P_NOWAIT,command[0],command,os.environ)

  def _runReport(self, step):
    global REPORTS_SUPPORT
    if not REPORTS_SUPPORT: 
      print "WARNING: Reports support not installed!"
      return 
      
    reportCommand = gConfigNav('RunReportCommand')

    # get the name of the report
    if os.path.basename(step.location) == step.location:
      try:
        reportdir = gConfigNav('ReportDir')
      except KeyError:
        reportdir = ""
      reportfile = os.path.join (paths.data, reportdir, step.location)
    else:
      reportfile = step.location
      
    # default parameter should be loaded from GREngine itself
    params ={
      'destination_type':'file',
      'destination':'-',
      'destination_options':'',
      'filter':'raw',
      'list_filters':0,
      'filter_options': '',
      'sortoption':'',
      'comment':None,
      'exclude_xml':None }
    
    # get parameters
    for child in step._children:
      if child._type=="GNParameter":
        params[child.name]=child.value

    i = self
    while i._parent!=None:
      i=i._parent

    try:
      engine=GREngine.GREngine(i._connections)
      engine.processReport(reportfile,
                           destination = params["destination"],
                           destinationType = params["destination_type"],
                           destinationOptions = params["destination_options"],
                           filter = params['filter'],
                           filterOptions = params['filter_options'],
                  #         parameters=userParameters,
                           sortoption=params["sortoption"],
                           includeStructuralComments=params["comment"],
                           omitGNUeXML=params["exclude_xml"])
    except GRExceptions.ReportMarkupException, msg:
      self.handleStartupError (msg)
    except GRExceptions.ReportUnreadable, msg:
      self.handleStartupError (msg)

  def handleStartupError(self,errortext):
    print
    print '-' * 60
    print _("Error: %s") % errortext
    print '-' * 60

    sys.exit()


class GNProcess(GNObject):
  def __init__(self, parent):
    GNObject.__init__(self, parent, type="GNProcess")


class GNStep(GNObject):
  def __init__(self, parent):
    GNObject.__init__(self, parent, type="GNStep")

  def _buildObject(self):
    self._controller = self.findParentOfType('GNProcesses')

  def run(self):
    try:
      GDebug.printMesg(1,'Running %s %s' % (self.type, self.location))
      self._controller._clientHandlers[self.type](self)
    except KeyError:
      print 'Error! Unknown step type "%s"' % self.type


class GNDescription(GNObject):
  def __init__(self, parent):
    GNObject.__init__(self, parent, type="GNDescription")

class GNParameter(GNObject):
  def __init__(self, parent):
    GNObject.__init__(self, parent, type="GNParameter")


class GNImportItem(GObjects.GObj):
  def __init__(self, parent=None, type="GNImport-Item"):
    GObj.__init__(self, parent, type=type)
    self._loadedxmlattrs = {} # Set by parser

  def _buildObject(self):
    if hasattr(self,'_xmltag'):
      self._type = 'GN%s' % self._xmltag
    if not hasattr(self,'_importclass'):
      self._importclass = GNParser\
         .getXMLelements()[string.lower(self._type[9:])][0]
    return GObj._buildObject(self)

