#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2004 Free Software Foundation
#
# FILE:
# labels/Adapter.py
#
# DESCRIPTION:
# The adapter for labels
#
# NOTES:
#

import string, sys, re
import xml.sax
import os.path

from gnue.reports.base.GROutputAdapter import TransformAdapter as Base
import LabelSpecs
import formatters

from gnue.common.external.shellwords import shellwords

class TransformAdapter(Base, xml.sax.ContentHandler):


  def __init__(self, *args, **params):
    Base.__init__(self, *args, **params)

  ###
  ### Methods for TransformAdapter support
  ### (these over-ride TransformAdapter methods)
  ###

  def open(self):
    # We need a temp file
    self.input, self.infile = self.createTempFile()
    return self.input

  def close(self):

    labeldefs = os.path.join(self.parameters['_filterbase'], 'Labels','label-specs.dat')

    # Grab all options...
    # these are of the form:  label="Brand 123" font=Helvetica fontsize=11
    self.options = {}
    for option in shellwords(self.filter_options):
      op, val = option.split('=',2)
      self.options[op.lower()] = val

    # Get the requested label type
    try:
      label = self.options['label'].lower()
    except KeyError:
      raise 'Please specify a label type. E.g., label="Brand 1010" '


    if label == 'custom':
      labelspec = {}
    else:
      LabelSpecs.initLabelDefs(labeldefs)
      labelspec = LabelSpecs.getByName(label)

    # Update with command-line overrides
    labelspec.update(self.options)


    # We are finished with the intermediate file, so
    # close in order for sablotron to be able to open it.
    self.input.close()

    # Get a file for output from the destination adapter.
    outfile = self.destination.getOutputHandle()

    # Get the name of the transform script
    try:
      module = formatters.loadFormatter(self.parameters['formatter'])
    except KeyError:
      raise "Filter configuration file is missing the 'formatter=' parameter"

    infile = open(self.infile,'r')

    self.formatter = module.LabelFormatter(outfile, labelspec)

    # Create a parser
    parser = xml.sax.make_parser()

    # Tell the parser to use our handler
    parser.setContentHandler(self)
    parser.parse(infile)

    infile.close()

    try:
      mimetype = self.parameters['mimetype']
    except:
      mimetype = self.MIMETYPE


    # Let the destination adapter do its thing
    self.destination.close(mimetype=mimetype)

    # clean up our temp files
    self.deleteTempFile(self.infile)


  ###
  ### Methods for Merge support
  ### (these are new methods)
  ###



  # XML handling stuff
  def startElement(self, name, attrs):
    if name == "labels":
      self.formatter.BeginJob()
    elif name == "label":
      self.formatter.BeginLabel()

    self._attrs = attrs
    self._text = ""

  def characters(self, text):
    self._text += text.encode(gConfig('textEncoding'))

  def endElement(self, name):
    if name == "labels":
      self.formatter.EndJob()
    elif name == "label":
      self.formatter.EndLabel()
    elif name == "line":

      # If suppressEmpty is true and this line is empty, skip
      try:
        skip = self._attrs['suppressEmpty'] == 'Y'
      except KeyError:
        skip = 0
      if skip and not len(self._text.strip()):
        return

      # Emphasized?
      try:
        emph = self._attrs['emphasize'] == 'Y'
      except KeyError:
        emph = 0

      self.formatter.AddLine(self._text, emphasized=emph)




  def getFieldValue(self, matchObject):
    name = string.lower(matchObject.group(1))
    try:
      return self.escape(self.values[name])
    except KeyError:
      return ""

