#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2002-2005 Free Software Foundation
#
# FILE:
# label/formatter/postscript.py
#
# DESCRIPTION:
# Base file for the GNUe Reports merge demos
#
# NOTES:
#

import string
from gnue.reports import VERSION
from gnue.common.utils.TextUtils import textToMeasurement

class LabelFormatter:

  baseFont =     ('Helvetica', 10)
  emphasisFont = ('Helvetica-Bold', 10)
  verticalSpacing = 11

  def __init__(self, destination, labelspec, options={}):
    self.dest = destination
    self.options = options
    self.labelspec = {}

    self._speccol = int(labelspec.get('columns','1')) or 1
    self._specrow = int(labelspec.get('rows','1')) or 1

    self._corners = {}

    measureScale = 72.0

    lm = textToMeasurement(labelspec.get('leftmargin','0'), measureScale)
    tm = textToMeasurement(labelspec.get('topmargin','0'), measureScale)
    ilm = textToMeasurement(labelspec.get('innerleftmargin','0'), measureScale)
    itm = textToMeasurement(labelspec.get('innertopmargin','0'), measureScale)
    vs = textToMeasurement(labelspec.get('verticalspacing','0'), measureScale)
    hs = textToMeasurement(labelspec.get('horizontalspacing','0'), measureScale)
    pw = textToMeasurement(labelspec.get('pagewidth','8.5in'), measureScale)
    ph = textToMeasurement(labelspec.get('pageheight','11in'), measureScale)
    self._width = textToMeasurement(labelspec.get('labelwidth'), measureScale)
    self._height = textToMeasurement(labelspec.get('labelheight'), measureScale)

    # Postscript works with bottom row being point "0"
    # figure out the total size of the label, so we can have a reverse number

    # Create a quick mapping of all label coordinates.
    # self._width * .1 ==> indent 10% (this was arbitrary!)
    for c in range(1,self._speccol+1):
      for r in range(1,self._specrow+1):
        self._corners[(c,r)] = ( lm + (c - 1) * hs + ilm + self._width * .1,
                                 ph - ( tm + (r - 1) * vs + itm ) - self.verticalSpacing )

    self._startedPage = 0
    self._page = 0

    # Set these out of bounds so that a new page will trigger
    self._row = self._specrow + 1
    self._col = self._speccol + 1

  def BeginJob(self):
    self.dest.write(PS_Header)

  def BeginPage(self):
    self._page += 1
    self.dest.write('%%%%Page: %s %s\n' % (self._page, self._page))
    self.dest.write(PS_Page_Header)
    self._startedPage = 1
    self._lastfont = None
    self._setFont(self.baseFont)

  def BeginLabel(self):
    self._col += 1
    if self._col > self._speccol:
      self._col = 1
      self._row += 1

    if self._row > self._specrow:
      self._row = 1
      self.EndPage()
      self.BeginPage()

    self._lines = []

  def AddLine(self, text, emphasized=0):
    self._lines.append((emphasized, text))

  def EndLabel(self):
    x, y = self._corners[ (self._col, self._row) ]

    # Center label vertically...
    vindent = (self._height - len(self._lines) * self.verticalSpacing) / 2
    y = y - vindent

    for emph, line in self._lines:
      stripped = string.replace(string.replace(line,'(',"\\("),')','\\)')
      self._setFont(emph and self.emphasisFont or self.baseFont)

      self.dest.write ('%s %s moveto (%s) show\n' % (
          x, y, stripped))
      y -= self.verticalSpacing

  def EndPage(self):
    if self._startedPage:
      self.dest.write(PS_Page_Footer)
    self._startedPage = 0

  def EndJob(self):
    self.EndPage()
    self.dest.write(PS_Footer)


  #
  # Set the font information
  #
  def _setFont(self, fontSpec):
    if fontSpec != self._lastfont:
      self.dest.write ('/%s findfont %s scalefont setfont\n' % fontSpec)
      self._lastfont = fontSpec


PS_Header = """\
%!PS-Adobe-3.0
%%Creator: GNUe Reports """ + VERSION + """ [Label Filter] (http://www.gnue.org/)
%%BoundingBox: 0 0 612 792
%%LanguageLevel: 2
%%Orientation: Portrait
%%EndComments
%%BeginProlog
%%EndProlog
%%BeginSetup
initgraphics
%%EndSetup
"""


PS_Footer = """\
%%Trailer
erasepage
initgraphics
%%EOF
"""

PS_Page_Header = """\
gsave
"""

PS_Page_Footer = """\
grestore
showpage
"""
