// Copyright (C) 2001 Nils Bokermann <Nils.Bokermann@mediaWays.net>
//
// PURPOSE OF THIS FILE: Implement the ldap_add functions
//
// - Automatic Version Information via RCS:
//   $Id: result.cxx,v 1.2 2001/12/19 05:26:56 nilsb Exp $
//   $Source: /cvsroot/openh323gk/openh323gk/ldap/src/result.cxx,v $
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "ldapapi.h"
#include <ldap-int.h> 
#include <ptlib/socket.h>


#ifndef lint
// mark object with version info
static const char vcid[] = "@(#) $Id: result.cxx,v 1.2 2001/12/19 05:26:56 nilsb Exp $";
static const char vcHid[] = __LDAP_OPENH323_H;
#endif


LDAPMessage *ldap_message_append(LDAP *ld, LDAPMessage* ldapmsg){
  for (vector<LDAPMessage*>::iterator iter=ld->ld_messages.begin();
       iter!=ld->ld_messages.end();iter++) {
    if ((*iter)->message->m_messageID==ldapmsg->message->m_messageID){
      LDAPMessage *entry=*iter;
      while(entry->next)
	entry=entry->next;
      entry->next=ldapmsg;
      ldapmsg->next=NULL;
      return *iter;
    }
  }
  ld->ld_messages.push_back(ldapmsg);
  return ldapmsg;
}

int ldap_read_msg(LDAP *ld, int msgid, int all, struct timeval *timeout, 
		  LDAPMessage **result) {
  PTimeInterval tm;
  PTime start;
    
  if(timeout==NULL) {
    tm=PMaxTimeInterval;
  }else{
    tm.SetInterval(timeout->tv_usec*1000, timeout->tv_sec);
  }
  do {
    PSocket::SelectList read;
    read+=(*ld->socket);
    
    PSocket::Errors err=PSocket::Select(read, tm);
    
    if(!((err==PSocket::Timeout)||(err==PSocket::NoError))) { // real error!
      ld->ld_errno=LDAP_OPERATIONS_ERROR;
      return -1;
  }
    
    if((err==PSocket::Timeout)&&(tm.GetMilliSeconds()!=0)) {
      ld->ld_errno=LDAP_TIMELIMIT_EXCEEDED;
      return -1;
    }
    
    // Ok -- there might be a socket readable 
    
    PBER_Stream decoding_stream;
    if(!decoding_stream.Read(*ld->socket)) {
      if(tm.GetMilliSeconds()==0){
	ld->ld_errno=LDAP_TIMELIMIT_EXCEEDED;
	return -1;
      }
    }
    while(!decoding_stream.GetPosition()<=decoding_stream.GetSize()) {
      LDAP_LDAPMessage *msg=new LDAP_LDAPMessage();
      if(!msg->Decode(decoding_stream)){
	//decoding error;
	break;
      }
#ifdef DEBUG 
      cerr << *msg << endl;
#endif
      LDAPMessage *ldapmsg=new LDAPMessage();
      LDAPMessage *master;
      ldapmsg->message=msg;
      ldapmsg->next=NULL;
      master=ldap_message_append(ld, ldapmsg);
      if((master->message->m_messageID==(unsigned int)msgid) ||
	 (LDAP_RES_ANY==msgid)) { // found.
	if(LDAP_MSG_ALL && 
	   (&(ldapmsg->message->m_protocolOp.GetObject()))->IsClass("LDAP_SearchResponse")) {
	  LDAP_SearchResponse &res=ldapmsg->message->m_protocolOp;
	  if (res.GetTag()==LDAP_SearchResponse::e_entry)
	    continue;
	}
	*result=master; 
	break;
      }
    }
    if(*result){
      for(vector<LDAPMessage *>::iterator iter=ld->ld_messages.begin();
	  iter!=ld->ld_messages.end(); iter++) {
	if(ldap_msgid(*iter)==ldap_msgid(*result)) {
	  ld->ld_messages.erase(iter);
	  break;
	}
      }
      return LDAP_SUCCESS;
    }
    PTime now;
    tm-=now-start;
  } while (tm.GetMilliSeconds()>0);
  ld->ld_errno=LDAP_TIMELIMIT_EXCEEDED;
  return -1;
}

int ldap_msgid (LDAPMessage *lm){
  if (lm)
    return lm->message->m_messageID;
  return -1;
}

int ldap_msgtype (LDAPMessage *lm) {
  if(!lm)
    return -1;
  LDAP_LDAPMessage_protocolOp &proto=lm->message->m_protocolOp;
  if((&proto.GetObject())->IsClass("LDAP_SearchResponse")) {
    LDAP_SearchResponse &response=proto;
    if(response.GetTag()==LDAP_SearchResponse::e_entry)
      return LDAP_RES_SEARCH_ENTRY;
    return LDAP_RES_SEARCH_RESULT;       
  }
  switch(proto.GetTag()) {
    case LDAP_LDAPMessage_protocolOp::e_modifyResponse:
      return LDAP_RES_MODIFY;
    case LDAP_LDAPMessage_protocolOp::e_addResponse:
      return LDAP_RES_ADD;
    case LDAP_LDAPMessage_protocolOp::e_delResponse:
      return LDAP_RES_DELETE;
    case LDAP_LDAPMessage_protocolOp::e_modifyRDNResponse:
      return LDAP_RES_MODDN;
    case LDAP_LDAPMessage_protocolOp::e_compareDNResponse:
      return LDAP_RES_COMPARE;
    case LDAP_LDAPMessage_protocolOp::e_bindResponse:
      return LDAP_RES_BIND;
  }
  return LDAP_RES_UNSOLICITED;
}

int ldap_result (LDAP *ld, int msgid, int all, struct timeval *timeout, 
		 LDAPMessage **result) {
  
  LDAPMessage *msg=NULL;
  *result=NULL;
  int rv;
  for(vector<LDAPMessage *>::iterator iter=ld->ld_messages.begin();
      iter!=ld->ld_messages.end(); iter++) {
    if(ldap_msgid(*iter)==msgid) {
      msg=*iter;
      ld->ld_messages.erase(iter);
      break;
    }
  }
  if(NULL!=msg) {
    result=&msg;
    if(LDAP_MSG_ALL!=all){
      return LDAP_SUCCESS;
    }
    while(msg->next){
      msg=msg->next;
    }
    if(LDAP_RES_SEARCH_RESULT==ldap_msgtype(msg)) {
      return LDAP_SUCCESS;
    }
  }
  if(LDAP_MSG_RECEIVED==all){
    result=NULL;
    ld->ld_errno=LDAP_OTHER;
    return -1;
  }

  //  Message is not in the buffer -- now fetch pending messages 
  if(LDAP_SUCCESS!=(rv=ldap_read_msg(ld, msgid, all, timeout, result))){
    return -1;
  }
  return ldap_msgtype(*result);
}


//
// End of result.cxx
//
