/*
     This file is part of GNUnet.
     (C) 2005, 2006, 2008 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file src/core/main.c
 * @brief Main function of gnunet-gtk
 * @author Christian Grothoff
 */

#include "platform.h"
#include "gnunetgtk_common.h"
#include <GNUnet/gnunet_directories.h>
#include <GNUnet/gnunet_util.h>

#ifdef WINDOWS
static int debug_mode = GNUNET_NO;
#endif

static char *cfgFilename = GNUNET_DEFAULT_CLIENT_CONFIG_FILE;

/**
 * All gnunet-gtk command line options
 */
static struct GNUNET_CommandLineOption gnunetgtkOptions[] = {
  GNUNET_COMMAND_LINE_OPTION_CFG_FILE (&cfgFilename),   /* -c */
#ifdef WINDOWS
  {'d', "debug", NULL,
   gettext_noop ("run in debug mode"),
   0, &GNUNET_getopt_configure_set_one, &debug_mode},
#endif
  GNUNET_COMMAND_LINE_OPTION_HELP (gettext_noop ("GNUnet GTK user interface.")),        /* -h */
  GNUNET_COMMAND_LINE_OPTION_HOSTNAME,  /* -H */
  GNUNET_COMMAND_LINE_OPTION_LOGGING,   /* -L */
  GNUNET_COMMAND_LINE_OPTION_VERSION (VERSION), /* -v */
  GNUNET_COMMAND_LINE_OPTION_VERBOSE,
  GNUNET_COMMAND_LINE_OPTION_END,
};

static void *
shutdownCode (void *unused)
{
  GNUNET_GTK_shutdown_plugins ();
  return NULL;
}

void
GNUNET_GTK_main_quit ()
{
  GNUNET_GE_setDefaultContext (NULL);
  GNUNET_GTK_run_with_save_calls (&shutdownCode, NULL);
  gtk_main_quit ();
}

static void
setIconSearchPath ()
{
  char *dataDir;
  char *buf;

  dataDir = GNUNET_get_installation_path (GNUNET_IPK_DATADIR);
  buf = GNUNET_malloc (strlen (dataDir) + strlen ("../icons/") + 2);
  strcpy (buf, dataDir);
  strcat (buf, "../icons/");
  gtk_icon_theme_append_search_path (gtk_icon_theme_get_default (), buf);
  GNUNET_free (buf);
  GNUNET_free (dataDir);
}

int
main (int argc, char *const *argv)
{
  GtkWidget *root;
  int i;
  struct GNUNET_GE_Context *ectx;
  struct GNUNET_GE_Context *my_ctx;
  struct GNUNET_GC_Configuration *cfg;
  char *log;
  GNUNET_GE_KIND mask;
#if ENABLE_NLS
  char *path;
#endif

#ifdef WINDOWS
  SetCursor (LoadCursor (NULL, IDC_APPSTARTING));
#endif

  g_thread_init (NULL);
  gtk_init (&argc, (char ***) &argv);
  i = GNUNET_init (argc,
                   argv,
                   "gnunet-gtk", &cfgFilename, gnunetgtkOptions, &ectx, &cfg);
  if (i == -1)
    {
      GNUNET_fini (ectx, cfg);
      return -1;
    }
#ifdef WINDOWS
  if (!debug_mode)
    FreeConsole ();
#endif

#if ENABLE_NLS
  setlocale (LC_ALL, "");
  path = GNUNET_get_installation_path (GNUNET_IPK_LOCALEDIR);
  BINDTEXTDOMAIN ("gnunet-gtk", path);
  GNUNET_free (path);
  textdomain ("gnunet-gtk");
  bind_textdomain_codeset ("GNUnet", "UTF-8");
  bind_textdomain_codeset ("gnunet-gtk", "UTF-8");
#endif
  setIconSearchPath ();
  GNUNET_GTK_initialize_common_library (cfg, &GNUNET_GTK_main_quit);
  /* configure GTK logging */
  GNUNET_GC_get_configuration_value_string (cfg,
                                            "LOGGING",
                                            "USER-LEVEL", "WARNING", &log);
  mask = GNUNET_GE_getKIND (log);
  GNUNET_free (log);
  mask |= mask - 1;             /* set all bits... */
  mask |= GNUNET_GE_USER | GNUNET_GE_BULK | GNUNET_GE_IMMEDIATE;
  if (GNUNET_YES == GNUNET_GC_get_configuration_value_yesno (cfg,
                                                             "LOGGING",
                                                             "DEVELOPER",
                                                             GNUNET_NO))
    mask |= GNUNET_GE_DEVELOPER | GNUNET_GE_REQUEST;
  my_ctx = GNUNET_GTK_create_gtk_logger (mask);
  GNUNET_GE_setDefaultContext (my_ctx);
  root =
    glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (), "mainWindow");
  gtk_window_maximize (GTK_WINDOW (root));
  gtk_widget_show (root);
  gtk_window_present (GTK_WINDOW (root));


  /* start the event loop */
  gdk_threads_enter ();

#ifdef WINDOWS
  SetCursor (LoadCursor (NULL, IDC_ARROW));
#endif

  gtk_main ();
  gdk_threads_leave ();
  GNUNET_GE_setDefaultContext (ectx);
  GNUNET_GE_free_context (my_ctx);
  GNUNET_GTK_shutdown_common_library ();
  GNUNET_fini (ectx, cfg);
  return 0;
}

/* ************* end of main.c ************ */
