/*
     This file is part of GNUnet.
     (C) 2005, 2006, 2007 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file src/plugins/daemon/daemon.c
 * @brief code for gnunet-gtk gnunetd interaction
 * @author Christian Grothoff
 */

#include "platform.h"
#include "gnunetgtk_common.h"
#include <GNUnet/gnunet_directories.h>
#include <GNUnet/gnunet_getoption_lib.h>
#include <GNUnet/gnunet_stats_lib.h>
#include <GNUnet/gnunet_util.h>
#include <gtk/gtk.h>
#include <glib/gerror.h>

#ifdef WITH_LIBGKSU2
/* Not used because libgksu2 headers have broken depends in Debian
And this is not really needed
#include <libgksu/libgksu.h> */
gboolean gksu_run (gchar * command_line, GError ** error);
#endif

static struct GNUNET_CronManager *cron;

static struct GNUNET_GE_Context *ectx;

static struct GNUNET_GC_Configuration *cfg;

/**
 * Do the actual update (in event thread).
 */
static void *
updateAppModelSafe (void *arg)
{
  GtkWidget *w;
  GtkListStore *model = arg;

  w =
    glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (),
                          "applicationList");
  gtk_tree_view_set_model (GTK_TREE_VIEW (w), GTK_TREE_MODEL (model));
  gtk_tree_selection_set_mode (gtk_tree_view_get_selection
                               (GTK_TREE_VIEW (w)), GTK_SELECTION_NONE);
  return NULL;
}

/**
 * cron job that periodically updates the model for the
 * application list.
 */
static void
updateAppModel (void *dummy)
{
  struct GNUNET_ClientServerConnection *sock;
  GtkTreeIter iter;
  char *apps;
  char *next;
  char *pos;
  char *desc;
  GtkListStore *model;

  model = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_STRING);
  apps = NULL;
  sock = GNUNET_client_connection_create (ectx, cfg);
  if (sock != NULL)
    apps =
      GNUNET_get_daemon_configuration_value (sock, "GNUNETD", "APPLICATIONS");
  if (apps != NULL)
    {
      next = apps;
      do
        {
          while (*next == ' ')
            next++;
          pos = next;
          while ((*next != '\0') && (*next != ' '))
            next++;
          if (*next == '\0')
            {
              next = NULL;      /* terminate! */
            }
          else
            {
              *next = '\0';     /* add 0-termination for pos */
              next++;
            }
          if (strlen (pos) > 0)
            {
              desc =
                GNUNET_get_daemon_configuration_value (sock, "ABOUT", pos);

              gtk_list_store_append (model, &iter);
              gtk_list_store_set (model,
                                  &iter,
                                  0, pos, 1, dgettext ("GNUnet", desc), -1);
              GNUNET_free_non_null (desc);
            }
        }
      while (next != NULL);
      GNUNET_free (apps);
    }
  GNUNET_client_connection_destroy (sock);
  GNUNET_GTK_save_call (&updateAppModelSafe, model);
  g_object_unref (model);
}


static void *
doUpdateMenus (void *arg)
{
  int ret = *(int *) arg;
  static GtkWidget *killEntry = NULL;
  static GtkWidget *launchEntry = NULL;
  static GtkWidget *statsEntryYes = NULL;
  static GtkWidget *statsEntryNo = NULL;
  static GtkWidget *statsEntryError = NULL;
  static GtkWidget *chooser = NULL;
  static int once = 1;
  static int isLocal;
  char *host;
  int canStart;
  struct GNUNET_GC_Configuration *dcfg;
  char *fn;
  char *user;

  if (once)
    {
      once = 0;
      killEntry =
        glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (), "stopDaemon");
      launchEntry =
        glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (),
                              "startDaemon");
      statsEntryYes =
        glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (),
                              "statusPixmapYes");
      statsEntryNo =
        glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (),
                              "statusPixmapNo");
      statsEntryError =
        glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (),
                              "statusPixmapError");
      chooser =
        glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (),
                              "gnunetdconfigFileChooserButton");
      GNUNET_GC_get_configuration_value_string (cfg, "NETWORK", "HOST",
                                                "localhost", &host);
      if (strncmp (host, "localhost:", 10) == 0)
        isLocal = TRUE;
      else
        isLocal = FALSE;
      GNUNET_free (host);
    }
  if (ret == 0)
    {
      canStart = 0;
      fn = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (chooser));
      if (NULL == fn)
        GNUNET_GC_get_configuration_value_filename (cfg, "DAEMON",
                                                    "CONFIGFILE",
                                                    GNUNET_DEFAULT_DAEMON_CONFIG_FILE,
                                                    &fn);
      if (isLocal && fn)
        {
          if (GNUNET_disk_file_test (ectx, fn) == GNUNET_YES)
            {
              dcfg = GNUNET_GC_create ();
              canStart = 1;
              if (0 != GNUNET_GC_parse_configuration (dcfg, fn))
                canStart = 0;
              user = NULL;
              GNUNET_GC_get_configuration_value_string (dcfg,
                                                        "GNUNETD",
                                                        "USER", "", &user);
              if (strlen (user) > 0)
                {
#if defined(WINDOWS) || defined(WITH_LIBGKSU2)
                  canStart = 1;
#else
                  struct passwd *pws;
                  if (NULL == (pws = getpwnam (user)))
                    {
                      canStart = 0;
                    }
                  else
                    {
                      if (pws->pw_uid != getuid ())
                        canStart = (geteuid () == 0);
                      else
                        canStart = 1;
                    }
#endif
                }
              GNUNET_free (user);
              GNUNET_GC_free (dcfg);
            }
          g_free (fn);
        }
      gtk_widget_hide (statsEntryYes);
      gtk_widget_set_sensitive (killEntry, FALSE);
      if (canStart && isLocal)
        {
          gtk_widget_set_sensitive (launchEntry, TRUE);
          gtk_widget_show_all (statsEntryNo);
          gtk_widget_hide (statsEntryError);
        }
      else
        {
          gtk_widget_set_sensitive (launchEntry, FALSE);
          gtk_widget_show_all (statsEntryError);
          gtk_widget_hide (statsEntryNo);
        }
    }
  else
    {
      gtk_widget_hide (statsEntryNo);
      gtk_widget_hide (statsEntryError);
      gtk_widget_show_all (statsEntryYes);
      gtk_widget_set_sensitive (killEntry, TRUE);
      gtk_widget_set_sensitive (launchEntry, FALSE);
    }
  return NULL;
}

/**
 * Cron job that checks if the daemon is running.
 */
static void
cronCheckDaemon (void *dummy)
{
  static int last = -1;
  int ret;

  if (GNUNET_OK == GNUNET_test_daemon_running (ectx, cfg))
    ret = 1;
  else
    ret = 0;
  if (last != ret)
    {
      updateAppModel (NULL);
      last = ret;
    }
  GNUNET_GTK_save_call (&doUpdateMenus, &ret);
}


/**
 * Launch gnunetd w/ checks
 */
void
on_startDaemon_clicked_daemon (GtkWidget * widget, gpointer data)
{
  GtkWidget *launchEntry;
  GtkWidget *chooser;
  char *fn;
  char *user;
  struct GNUNET_GC_Configuration *dcfg;
  int code = 0;

  launchEntry =
    glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (), "startDaemon");
  gtk_widget_set_sensitive (launchEntry, FALSE);
  if (GNUNET_OK == GNUNET_test_daemon_running (ectx, cfg))
    {
      GNUNET_cron_advance_job (GNUNET_GTK_get_cron_manager (),
                               &cronCheckDaemon,
                               15 * GNUNET_CRON_SECONDS, NULL);
      return;
    }
  else
    {
      GNUNET_GTK_add_log_entry (_("Launching gnunetd...\n"));
      chooser
        = glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (),
                                "gnunetdconfigFileChooserButton");
      fn = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (chooser));
      GNUNET_GE_BREAK (ectx, fn != NULL);

      dcfg = GNUNET_GC_create ();
      if (0 != GNUNET_GC_parse_configuration (dcfg, fn))
        user = NULL;
      GNUNET_GC_get_configuration_value_string (dcfg,
                                                "GNUNETD", "USER", "", &user);
#ifdef WITH_LIBGKSU2
      char *command;
      GError *gerror = NULL;
      struct passwd *pws;
      if (strlen (user) > 0)
        {
          pws = getpwnam (user);
          if (pws->pw_uid != getuid ())
            {
              command = g_strconcat ("gnunetd -c ", fn, NULL);
              code = gksu_run (command, &gerror);
              GNUNET_free (command);
              if (gerror)
                {
                  GNUNET_GTK_add_log_entry (_("Launching gnunetd failed\n"));

                  g_error_free (gerror);
                }
              else
                {
                  GNUNET_GTK_add_log_entry (_("Launched gnunetd\n"));
                }
              g_free (fn);
              GNUNET_cron_advance_job (GNUNET_GTK_get_cron_manager (),
                                       &cronCheckDaemon,
                                       15 * GNUNET_CRON_SECONDS, NULL);
              return;
            }
          else
            code = GNUNET_daemon_start (ectx, cfg, fn, GNUNET_YES);
        }
#else
      code = GNUNET_daemon_start (ectx, cfg, fn, GNUNET_YES);
#endif

      if (GNUNET_SYSERR != code)
        {
          GNUNET_GTK_add_log_entry (_("Launched gnunetd\n"));
        }
      else
        {
          GNUNET_GTK_add_log_entry (_("Launching gnunetd failed\n"));
        }
      if (fn != NULL)
        g_free (fn);
      GNUNET_cron_advance_job (GNUNET_GTK_get_cron_manager (),
                               &cronCheckDaemon,
                               15 * GNUNET_CRON_SECONDS, NULL);
    }
}

/**
 * Kill gnunetd
 */
void
on_stopDaemon_clicked_daemon (GtkWidget * widget, gpointer data)
{
  GtkWidget *killEntry = NULL;
  struct GNUNET_ClientServerConnection *sock;

  killEntry =
    glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (), "stopDaemon");
  gtk_widget_set_sensitive (killEntry, FALSE);

  if (GNUNET_OK == GNUNET_test_daemon_running (ectx, cfg))
    {
      sock = GNUNET_client_connection_create (ectx, cfg);
      if (GNUNET_OK !=
          GNUNET_client_connection_request_daemon_shutdown (sock))
        {
          GtkWidget *dialog;

          dialog = gtk_message_dialog_new
            (NULL,
             GTK_DIALOG_MODAL,
             GTK_MESSAGE_ERROR,
             GTK_BUTTONS_CLOSE, _("Error requesting shutdown of gnunetd.\n"));
          gtk_dialog_run (GTK_DIALOG (dialog));
          gtk_widget_destroy (dialog);
        }
      else
        {
          GNUNET_GTK_add_log_entry (_("Terminating gnunetd...\n"));
        }
      GNUNET_client_connection_destroy (sock);
    }
  GNUNET_cron_advance_job (GNUNET_GTK_get_cron_manager (),
                           &cronCheckDaemon, 15 * GNUNET_CRON_SECONDS, NULL);
}

void
init_daemon (struct GNUNET_GE_Context *e, struct GNUNET_GC_Configuration *c)
{
  GtkWidget *tab;
  GtkWidget *apps;
  GtkListStore *model;
  GtkCellRenderer *renderer;
  int col;
  char *daemon_config;

  ectx = e;
  cfg = c;
  apps =
    glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (),
                          "applicationList");
  model = gtk_list_store_new (1, G_TYPE_STRING);
  gtk_tree_view_set_model (GTK_TREE_VIEW (apps), GTK_TREE_MODEL (model));
  renderer = gtk_cell_renderer_text_new ();
  col = gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (apps),
                                                     -1,
                                                     _("Application"),
                                                     renderer,
                                                     "text", 0, NULL);
  gtk_tree_view_column_set_resizable (gtk_tree_view_get_column
                                      (GTK_TREE_VIEW (apps), col - 1), TRUE);
  renderer = gtk_cell_renderer_text_new ();
  col = gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (apps),
                                                     -1,
                                                     _("Description"),
                                                     renderer,
                                                     "text", 1, NULL);
  gtk_tree_view_column_set_resizable (gtk_tree_view_get_column
                                      (GTK_TREE_VIEW (apps), col - 1), TRUE);

  tab =
    glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (),
                          "daemonScrolledWindow");
  gtk_widget_show (tab);
  daemon_config = NULL;
  GNUNET_GC_get_configuration_value_filename (cfg,
                                              "DAEMON",
                                              "CONFIGFILE",
                                              GNUNET_DEFAULT_DAEMON_CONFIG_FILE,
                                              &daemon_config);
  if (GNUNET_YES == GNUNET_disk_file_test (NULL, daemon_config))
    {
      gtk_file_chooser_set_filename (GTK_FILE_CHOOSER
                                     (glade_xml_get_widget
                                      (GNUNET_GTK_get_main_glade_XML (),
                                       "gnunetdconfigFileChooserButton")),
                                     daemon_config);
    }
  else
    {
      GNUNET_GTK_add_log_entry (_
                                ("Configuration file for GNUnet daemon `%s' does not exist! Run `gnunet-setup -d'.\n"),
                                daemon_config);
    }

#ifndef WITH_LIBGKSU2
  GtkWidget *startDaemonConfWizard;
  GtkWidget *startDaemonConfTool;

  startDaemonConfWizard
    = glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (),
                            "startDaemonConfWizard");
  startDaemonConfTool
    = glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (),
                            "startDaemonConfTool");
  if (0 != ACCESS (daemon_config, W_OK))
    {
      gtk_widget_set_sensitive (startDaemonConfWizard, FALSE);
      gtk_widget_set_sensitive (startDaemonConfTool, FALSE);
    }
#endif
  GNUNET_free (daemon_config);

  cron = GNUNET_GTK_get_cron_manager ();
  GNUNET_cron_add_job (cron, &cronCheckDaemon, 2 * GNUNET_CRON_SECONDS,
                       15 * GNUNET_CRON_SECONDS, NULL);
  GNUNET_cron_add_job (cron, &updateAppModel, 5 * GNUNET_CRON_MINUTES,
                       5 * GNUNET_CRON_MINUTES, NULL);
}

void
done_daemon ()
{
  GtkWidget *w;

  GNUNET_cron_del_job (cron, &cronCheckDaemon, 15 * GNUNET_CRON_SECONDS,
                       NULL);
  GNUNET_cron_del_job (cron, &updateAppModel, 5 * GNUNET_CRON_MINUTES, NULL);
  w =
    glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (),
                          "applicationList");
  gtk_tree_view_set_model (GTK_TREE_VIEW (w), NULL);

  w =
    glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (),
                          "applicationList");
  gtk_tree_view_set_model (GTK_TREE_VIEW (w), NULL);

}

/**
* Launch 'gnunet-setup -d wizard-gtk' with needed rights
*/
void
on_startDaemonConfWizard_clicked_daemon (GtkWidget * widget, gpointer data)
{
  GtkWidget *filechooserbutton;
  char *conffile;

  filechooserbutton
    = glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (),
                            "gnunetdconfigFileChooserButton");
  conffile =
    gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (filechooserbutton));
  GNUNET_GTK_run_gnunet_setup (conffile, TRUE);
  GNUNET_free (conffile);
}

/**
* Launch 'gnunet-setup -d gconfig' with needed rights
*/
void
on_startDaemonConfTool_clicked_daemon (GtkWidget * widget, gpointer data)
{
  GtkWidget *filechooserbutton;
  char *conffile;

  filechooserbutton
    = glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (),
                            "gnunetdconfigFileChooserButton");
  conffile =
    gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (filechooserbutton));
  GNUNET_GTK_run_gnunet_setup (conffile, FALSE);
  GNUNET_free (conffile);
}

/**
* Update sensitivity of daemon buttons
*/
void
on_gnunetdconfigfile_set_daemon (GtkWidget * filechooserbutton, gpointer data)
{
#ifndef WITH_LIBGKSU2
  GtkWidget *startDaemonConfWizard;
  GtkWidget *startDaemonConfTool;
  char *conffile;
  int ret;

  startDaemonConfWizard
    = glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (),
                            "startDaemonConfWizard");
  startDaemonConfTool
    = glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (),
                            "startDaemonConfTool");
  conffile =
    gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (filechooserbutton));
  if (0 == ACCESS (conffile, W_OK))
    {
      gtk_widget_set_sensitive (startDaemonConfWizard, TRUE);
      gtk_widget_set_sensitive (startDaemonConfTool, TRUE);
    }
  else
    {
      gtk_widget_set_sensitive (startDaemonConfWizard, FALSE);
      gtk_widget_set_sensitive (startDaemonConfTool, FALSE);
    }
  ret = 0;
  GNUNET_GTK_save_call (&doUpdateMenus, &ret);
  GNUNET_free_non_null (conffile);
#endif
  return;
}

/* end of daemon.c */
