/* -*- c++ -*- */
/*
 * Copyright 2002 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef _GRINTERPOLATOR_H_
#define _GRINTERPOLATOR_H_

#include <GrFractionalInterpolatingSigProc.h>
#include <stdlib.h>
#include <gr_mmse_fir_interpolator.h>

/*!
 * \brief bandwidth limited interpolator.
 *
 * The input is interpolated at a ratio of N/M.
 * Currently, N must be 10 and M must be in [1, 10].
 * Phase must be in [0, N].
 */

template<class iType, class oType>
class GrInterpolator : public GrFractionalInterpolatingSigProc<iType,oType>
{
 public:
  //! N == 10, M in [1,10], phase in [0,N]
  GrInterpolator (int N, int M, int phase = 0);
  ~GrInterpolator () {};

  const char *name () { return "GrInterpolator"; }
  int work (VrSampleRange output, void *ao[], VrSampleRange inputs[], void *ai[]);

protected:
  int				phase;
  gr_mmse_fir_interpolator	intr;
};

template<class iType,class oType>
GrInterpolator<iType,oType>::GrInterpolator (int N, int M, int _phase = 0)
  : GrFractionalInterpolatingSigProc<iType,oType>(N, M), phase (_phase)
{
  if (N != 10){
    cerr << "GrInterpolator: N must be 10\n";
    exit (1);
  }
  if (M < 0 || M > 10){
    cerr << "GrInterpolator: M must be in [1, 10]\n";
    exit (1);
  }
  if (phase < 0 || phase > N){
    cerr << "GrInterpolator: phase must be in [0, N]\n";
    exit (1);
  }

  history = intr.ntaps () + 1;
}

template<class iType,class oType>
int
GrInterpolator<iType,oType>::work (VrSampleRange output, void *ao[],
				   VrSampleRange inputs[], void *ai[])
{
  iType *in =  ((iType **) ai)[0];
  oType *out = ((oType **) ao)[0];

  assert ((output.size % iratio) == 0);

  float	mu = (float) phase / N;			// fractional delay
  float mu_inc = (float) M / N;

  unsigned	ii = 0;				// input index
  unsigned	oo = 0;				// output index

  while (oo < output.size){
    out[oo++] = intr.interpolate (&in[ii], 1.0 - mu);

    // printf ("%4d %9.6f\n", ii, mu);

    float s = mu + mu_inc;
    float f = floor (s);
    int incr = (int) f;
    mu = s - f;
    ii += incr;
  }

  assert (mu == ((float) phase / N));

  return output.size;
}


#endif /* _GRINTERPOLATOR_H_ */
