/* -*- Mode: c++ -*- */
/*
 * Copyright 2001 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
/*
 *  Copyright 1997 Massachusetts Institute of Technology
 * 
 *  Permission to use, copy, modify, distribute, and sell this software and its
 *  documentation for any purpose is hereby granted without fee, provided that
 *  the above copyright notice appear in all copies and that both that
 *  copyright notice and this permission notice appear in supporting
 *  documentation, and that the name of M.I.T. not be used in advertising or
 *  publicity pertaining to distribution of the software without specific,
 *  written prior permission.  M.I.T. makes no representations about the
 *  suitability of this software for any purpose.  It is provided "as is"
 *  without express or implied warranty.
 * 
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <VrTypes.h>
#include <VrMultiTask.h>
#include <VrPerfGraph.h>

int BufferSizeMultiplier=1;
#ifdef PERFMON
//the scheduler could not find any data to mark
unsigned long long couldNotMarkAnyData=0;
#endif


// 
// CONSTRUCTORS
//
VrMultiTask::VrMultiTask()
  : numberSinks(0), goodSinks(NULL), numSinks(0), numThreads(0), stopFlag(1)
{
#ifdef THREADS
  FILE *f=fopen("/proc/cpuinfo","r");
  numThreads=1;
  if(f!=NULL) {
    char line[80];
    while(fgets(line,80,f)!=NULL) {
      int x=0;
      if(sscanf(line,"processor\t: %d",&x)!=0) {
		if(x<32)
	  		numThreads=x+1;
      }
    }
  }
  fprintf(stderr,"%d processors detected.\n",numThreads);
#endif
}

VrMultiTask::VrMultiTask(int n_threads)
  : numberSinks(0), goodSinks(NULL), numSinks(0), numThreads(n_threads), stopFlag(1)
{
}

VrMultiTask::~VrMultiTask() 
{
#ifdef THREADS
  if(pthread_cond_destroy(&start_condition) != 0) {
    fprintf(stderr, "Couldn't destory start condition variable!\n");
  }
  if(pthread_mutex_destroy(&mutex) != 0) {
    fprintf(stderr, "Couldn't destory mutex!\n");
  }
#endif

  if(goodSinks!=NULL)
    delete [] goodSinks;
  if (sinks != NULL)
    delete [] sinks;
}

//
// MANIPULATORS
//

void 
VrMultiTask::add(VrSigProc *s) 
{
  VrSigProc ** ss = new (VrSigProc *[numberSinks+1]);
  ss[numberSinks]=s;
  if(numberSinks>0) {
    for(int i=0;i<numberSinks;i++)
      ss[i]=sinks[i];
    delete [] sinks;
  }
  sinks=ss;
  numberSinks++;
}

void VrMultiTask::start()
{
  if(isStarted()) {
    fprintf(stderr,"VrMultiTask: already start()ed\n");
    exit(-1);
  }

#ifdef THREADS
  start_worker_threads();
#endif
    
  numSinks=0;
  int i;
  
  if(goodSinks!=NULL)
    delete goodSinks;
  goodSinks = new (VrSigProc *[numberSinks]);

#ifdef THREADS
  BufferSizeMultiplier = numThreads + 1;
#else
  // BufferSizeMultiplier = numberSinks + 1;
  BufferSizeMultiplier = 2;  // -eb FIXME what's the right answer?
#endif

  /* Figure out which sinks are connected and run init()*/
  for(i=0;i<numberSinks;i++){
    if(sinks[i]->isConnectedToSource()) {
      sinks[i]->init_base();
      goodSinks[numSinks++]=sinks[i];
    }
  }
  if(numSinks==0) {
    fprintf(stderr, "VrMultiTask: No connected sinks!\n");
  }


  //run setup() on all sinks
  for(i=0;i<numSinks;i++)
      goodSinks[i]->setup();


#ifdef PERFMON
  /* Make a list of all the modules in the system
     (for performance monitoring, VrPerfGraph, etc. */
  perfgraph = new VrPerfGraph();
  for(i=0;i<numSinks;i++)
    goodSinks[i]->addToGraph(perfgraph);
#endif

  startTimer();

  stopFlag = 0;
#ifdef THREADS
  //Wake-up the workers
  pthread_cond_broadcast(&start_condition);
#endif
}

#if !defined(THREADS)

void VrMultiTask::process()
{
  if(PARANOID && !isStarted()) {
    fprintf(stderr,"VrMultiTask.process(): not start()ed\n");
    abort ();
  }

#ifdef PERFMON
  overhead->startCount();
#endif

  schedule();

#ifdef PERFMON
  overhead->stopCount();
#endif
}

#else	// THREADS

void VrMultiTask::process()
{
  if(PARANOID && !isStarted()) {
    fprintf(stderr,"VrMultiTask.process(): not start()ed\n");
    abort ();
  }

  // not very exciting...
  // (The real work gets done in the worker threads)

  usleep(10000);
}

#endif // THREADS

void VrMultiTask::stop()
{

  if(!isStarted()) {
    fprintf(stderr, "VrMultiTask.stop(): chain not running.\n");
    exit(-1);
  }

  stopFlag=1;		// stop threads if present (doesn't wait) */

#ifdef THREADS
  //Wake-up the workers
  pthread_cond_broadcast(&start_condition);

  //wait for threads to finish
  for(int n=0;n<numThreads;n++) {
    if(pthread_join(threads[n],NULL) != 0) {
      fprintf(stderr, "Thread join error.\n");
      exit(-1);
    }
  }
#endif
}

#ifdef THREADS
//Starts a thread that runs the scheduling routine to
//  help work on all sinks
void *start_worker_thread(void *mt)
{
  ((VrMultiTask *) mt)->_start_worker();
  return NULL;
}

void VrMultiTask::_start_worker()
{
  pthread_mutex_lock(&mutex);
#ifdef PERFMON
  START_COUNTS();
#endif
  pthread_cond_wait(&start_condition,&mutex); //returns with mutex locked

#ifdef PERFMON
  overhead->startCount();
#endif

  if(!stopFlag) schedule();

#ifdef PERFMON
    overhead->stopCount();
#endif

  while(!stopFlag) {
    pthread_mutex_lock(&mutex);
#ifdef PERFMON
    overhead->startCount();
#endif
    if(!stopFlag) schedule(); //unlocks mutex
    else {
      pthread_mutex_unlock(&mutex);
      break;
    }
#ifdef PERFMON
    overhead->stopCount();
#endif
  }

#ifdef PERFMON
  //  overhead->stopCount();
  //  STOP_COUNTS();
#endif
}

void VrMultiTask::start_worker_threads()
{

  if(pthread_cond_init(&start_condition, NULL)!=0) {
    fprintf(stderr, "Couldn't initialize start condition variable\n");
    exit(-1);
  }
  if(pthread_mutex_init(&mutex, NULL)!=0) {
    fprintf(stderr, "Couldn't initialize mutex\n");
    exit(-1);
  }
  if(pthread_key_create(&startMarkedModule, NULL)!=0) {
    fprintf(stderr, "VrBuffer: Could not create thread specific variable.\n");
    exit(-1);
  }

  threads = new pthread_t[numThreads];
  
  for(int n=0;n<numThreads;n++) {
    if(pthread_create(& threads[n], NULL,
		      start_worker_thread,
		      (void *) this) != 0) {
      fprintf(stderr, "Thread creation error.\n");
      exit(1);
    }
  }
}
#endif

#ifdef PERFMON
void VrMultiTask::print_stats()
{
  fprintf(stderr, "\nGLOBAL counts:\n");
  fprintf(stderr, "--------------\n");
  fprintf(stderr, "couldNotMarkAnyData= %lld\n", couldNotMarkAnyData);

  fprintf(stderr, "\nPer module counts:\n");
  fprintf(stderr, "------------------\n\n");

  for(int i=0;i<numSinks;i++) {
    goodSinks[i]->print_stats();
  }

  fprintf(stderr, "\nSummary:\n");
  fprintf(stderr, "--------------\n");
  perfgraph->print_stats();
}
#endif

//! Figures out work to be done and then does it.
/*!
  Using markData schedule figures out what data needs to be computed.
  When this phase it complete, compute is called to geneate the data.
  
  Threads MUST acquire the mutex _before_ calling schedule()
*/

void VrMultiTask::schedule()
{
  //find the sink that is the furthest behind
  float seconds[numSinks];
  int min_index;
  int i=0;
  int incr=0;

//jca printf ("[%s:%d] schedule start\n", __FILE__, __LINE__);
#ifdef THREADS
  for(incr=0;incr<2*numThreads;incr++) {
#endif
    min_index=0;
    for(i=0;i<numSinks;i++) {
      seconds[i] = (goodSinks[i]->getMarkedWP()
			+incr*goodSinks[min_index]->getMaxOutputSize())
		/ goodSinks[i]->getSamplingFrequency();
      if(seconds[i]<seconds[min_index])
		min_index=i;
    }

    for(i=0;i<numSinks;i++) {
      VrSampleRange r = { goodSinks[min_index]->getMarkedWP()
			+ incr * goodSinks[min_index]->getMaxOutputSize(),
		goodSinks[min_index]->getMaxOutputSize() };

      switch(goodSinks[min_index]->markData(r)) {

		//could add case 3 -- buffer full upstream -- compute whatever
		//  you can at highest priority

      case VrSigProc::MARK_ALREADY: //data is finished upstream
		fprintf(stderr, "Inconsistency in scheduler.\n");
		exit(-1);

      case VrSigProc::MARK_READY:
		//data successfully marked upstream and more data may
		//be available to compute on this sink
		goto successfulMark; //break out of the for loop

      case VrSigProc::MARK_READY_NO_MARK: //data upstream is marked but no more can be computed
		goto successfulMark;      //break out of the for loop

      case VrSigProc::MARK_NO_READY: //no data can be computed upstream
      case VrSigProc::MARK_THREAD:   //data marked by another thread and not ready yet
		break;
      }
      //jca printf ("[%s:%d] schedule after mark\n", __FILE__, __LINE__);

      seconds[min_index]=HUGE_VAL; //Infinity
      for(int j=0;j<numSinks;j++)
		if(seconds[j]<seconds[min_index])
			min_index=j;
    }
#ifdef THREADS
  }
#endif

 successfulMark:
  //jca printf ("[%s:%d] schedule mark success\n", __FILE__, __LINE__);
  
  //release lock (also serves as a memory barrier)
  MUTEX_UNLOCK (&mutex);

  if(i < numSinks) {			// compute chunk
#ifdef THREADS
    ((VrSigProc *) pthread_getspecific(startMarkedModule))->compute();
#else
    startMarkedModule->compute();
#endif
  } else {
#ifdef PERFMON
    couldNotMarkAnyData++;
#endif
    YIELD();
  } 
} 

void
VrMultiTask::startTimer()
{
  if (gettimeofday(&initialTime, 0) < 0) {
    perror("VrMultiTask: gettimeofday failed!!!");
    exit(1);
  }
}

double
VrMultiTask::elapsedTime()
{
  unsigned long usecs;
  struct timeval t2;
  
  if (gettimeofday(&t2, 0) < 0) {
    perror("VrMultiTask: gettimeofday failed!!!");
    exit(1);
  }
  
  usecs = (t2.tv_sec - initialTime.tv_sec)*1000000 + 
    (t2.tv_usec - initialTime.tv_usec);
  return usecs/1000000.0;
}
