/*
 * Copyright (C) 2002,2003 Pascal Haakmat.
 * FIXME: I don't know anything about filtering.
 * Portions Copyright <baltrax@hotmail.com> (Zxform)
 */


#include <math.h>
#include <gtk/gtk.h>
#include <glade/glade.h>
#include "../modutils.h"

module modinfo;

typedef struct {
    module_id id;
    shell *shl;
    double frequency;
    double q;
    GtkSpinButton *frequency_control;
    GtkSpinButton *q_control;
    GtkWidget *dialog;
} filter_params;

#ifdef HAVE_GNOME2
#define LOWPASS_GLADE_FILE "lowpass-2.glade"
#else
#define LOWPASS_GLADE_FILE "lowpass.glade"
#endif

/* 
 * Resonant low pass filter source code.
 * By baltrax@hotmail.com (Zxform)
 * Formatted a little. -ph
 */

#include <stdlib.h>
#include <stdio.h>
#include <math.h>

/*
 * ----------------------------------------------------------
 *      bilinear.c
 *
 *      Perform bilinear transformation on s-domain coefficients
 *      of 2nd order biquad section.
 *      First design an analog filter and use s-domain coefficients
 *      as input to szxform() to convert them to z-domain.
 *
 * Here's the butterworth polinomials for 2nd, 4th and 6th order
 * sections.  When we construct a 24 db/oct filter, we take to 2nd
 * order sections and compute the coefficients separately for each
 * section.
 *
 *      n       Polinomials
 * --------------------------------------------------------------------
 *      2       s^2 + 1.4142s +1
 *      4       (s^2 + 0.765367s + 1) (s^2 + 1.847759s + 1)
 *      6       (s^2 + 0.5176387s + 1) (s^2 + 1.414214 + 1) (s^2 + 1.931852s + 1)
 *
 *      Where n is a filter order.  For n=4, or two second order
 *      sections, we have following equasions for each 2nd order
 *      stage:
 *      LPF:
 *      (1 / (s^2 + (1/Q) * 0.765367s + 1)) * (1 / (s^2 + (1/Q) * 1.847759s + 1))
 *      HPF:
 *      (s^2 / (s^2 + (1/Q) * 0.765367s + 1)) * (s^2 / (s^2 + (1/Q) * 1.847759s + 1))
 *
 *      Where Q is filter quality factor in the range of
 *      1 to 1000. The overall filter Q is a product of all
 *      2nd order stages. For example, the 6th order filter
 *      (3 stages, or biquads) with individual Q of 2 will
 *      have filter Q = 2 * 2 * 2 = 8.
 *
 *      The nominator part is just 1.
 *      The denominator coefficients for stage 1 of filter are:
 *      b2 = 1; b1 = 0.765367; b0 = 1;
 *      numerator is
 *      a2 = 0; a1 = 0; a0 = 1;
 *
 *      The denominator coefficients for stage 1 of filter are:
 *      b2 = 1; b1 = 1.847759; b0 = 1;
 *      numerator is
 *      a2 = 0; a1 = 0; a0 = 1;
 *
 *      These coefficients are used directly by the szxform()
 *      and bilinear() functions. For all stages the numerator
 *      is the same and the only thing that is different between
 *      different stages is 1st order coefficient. The rest of
 *      coefficients are the same for any stage and equal to 1.
 *
 *      Any filter could be constructed using this approach.
 *
 *      References:
 *             Van Valkenburg, "Analog Filter Design"
 *             Oxford University Press 1982
 *             ISBN 0-19-510734-9
 *
 *             C Language Algorithms for Digital Signal Processing
 *             Paul Embree, Bruce Kimble
 *             Prentice Hall, 1991
 *             ISBN 0-13-133406-9
 *
 *             Digital Filter Designer's Handbook
 *             With C++ Algorithms
 *             Britton Rorabaugh
 *             McGraw Hill, 1997
 *             ISBN 0-07-053806-9
 * ----------------------------------------------------------
 */

/*
 * ----------------------------------------------------------
 *      Pre-warp the coefficients of a numerator or denominator.
 *      Note that a0 is assumed to be 1, so there is no wrapping
 *      of it.
 * ----------------------------------------------------------
 */
inline void
prewarp(double *a0, 
        double *a1,
        double *a2,
        double cutoff, 
        double rate) {
    double wp;

    //    double omega = 2 * M_PI * cutoff / rate;
    //    wp = 1 / tan(omega / 2);

    wp = 2.0 * rate * tan(cutoff * M_PI / rate);

    *a2 = (*a2) / (wp * wp);
    *a1 = (*a1) / wp;
}

/*
 * ----------------------------------------------------------
 * bilinear()
 *
 * Transform the numerator and denominator coefficients
 * of s-domain biquad section into corresponding
 * z-domain coefficients.
 *
 *      Store the 4 IIR coefficients in array pointed by coef
 *      in following order:
 *             beta1, beta2    (denominator)
 *             alpha1, alpha2  (numerator)
 *
 * Arguments:
 *             a0-a2   - s-domain numerator coefficients
 *             b0-b2   - s-domain denominator coefficients
 *             k               - filter gain factor. initially set to 1
 *                                and modified by each biquad section in such
 *                                a way, as to make it the coefficient by
 *                                which to multiply the overall filter gain
 *                                in order to achieve a desired overall filter gain,
 *                                specified in initial value of k.
 *             fs             - sampling rate (Hz)
 *             coef    - array of z-domain coefficients to be filled in.
 *
 * Return:
 *             On return, set coef z-domain coefficients
 * ----------------------------------------------------------
 */
inline void
bilinear(double a0,
         double a1, 
         double a2,        /* numerator coefficients */
         double b0, 
         double b1,
         double b2,        /* denominator coefficients */
         double *k,        /* overall gain factor */
         double rate,        /* sampling rate */
         float *coef) {    /* pointer to 4 iir coefficients */
    double ad, bd;

    /*
     * Biquad transfer function:
     * (4 * a2 * s^2) + (2 * a1 * s) + a0
     * ----------------------------------
     * (4 * b2 * s^2) + (2 * b1 * s) + b0
     *
     */

    /* alpha (Numerator in s-domain) */
    ad = (4. * a2 * rate * rate) + (2. * a1 * rate) + a0;
    /* beta (Denominator in s-domain) */
    bd = (4. * b2 * rate * rate) + (2. * b1 * rate) + b0;
    
    /* update gain constant for this section */
    *k *= ad / bd;
    
    /* Denominator */

    /*
     *                1                
     * B1 = ---------------------- * (2 * b0 - 8 * b2 * s^2)
     *      b2 * s^2 + b1 * s + b0
     *
     *                1                
     * B2 = ---------------------- * (4 * b2 * s^2 - 2 * b1 * s + b0)
     *      b2 * s^2 + b1 * s + b0
     *
     */

    /* beta1 */
    *coef++ = ((2. * b0) - (8. * b2 * rate * rate)) * (1 / bd);
    /* beta2 */
    *coef++ = (((4. * b2 * rate * rate) - (2. * b1 * rate) + b0)) * (1 / bd);
  
    /*  
     *                1                
     * A1 = ---------------------- * (2 * a0 - 8 * a2 * s^2)
     *      a2 * s^2 + a1 * s + a0
     *
     *                1                
     * A1 = ---------------------- * (4 * a2 * s^2 - 2 * a1 * s + a0)
     *      a2 * s^2 + a1 * s + a0
     */
    /* Nominator */

    /* alpha1 */
    *coef++ = ((2. * a0) - (8. * a2 * rate * rate)) * (1 / ad);
    /* alpha2 */
    *coef = ((4. * a2 * rate * rate) - (2. * a1 * rate) + a0) * (1 / ad);
}


/*
 * ----------------------------------------------------------
 * Transform from s to z domain using bilinear transform
 * with prewarp.
 *
 * Arguments:
 *      For argument description look at bilinear()
 *
 *      coef - pointer to array of floating point coefficients,
 *                     corresponding to output of bilinear transofrm
 *                     (z domain).
 *
 * Note: frequencies are in Hz.
 * ----------------------------------------------------------
 */
inline void
szxform(double *a0,
        double *a1, 
        double *a2,          /* numerator coefficients */
        double *b0, 
        double *b1,
        double *b2,          /* denominator coefficients */
        double cutoff,           /* Filter cutoff frequency */
        double rate,           /* sampling rate */
        double *k,           /* overall gain factor */
        float *coef) {       /* pointer to 4 iir coefficients */
    /* Calculate a1 and a2 and overwrite the original values */
    prewarp(a0, a1, a2, cutoff, rate);
    prewarp(b0, b1, b2, cutoff, rate);
    bilinear(*a0, *a1, *a2, *b0, *b1, *b2, k, rate, coef);
}

/**************************************************************************

FILTER.C - Source code for filter functions

    iir_filter         IIR filter floats sample by sample (real time)

*************************************************************************/

/* FILTER INFORMATION STRUCTURE FOR FILTER ROUTINES */

typedef struct {
    unsigned int length;       /* size of filter */
    float *history;            /* pointer to history in filter */
    float *coef;               /* pointer to coefficients of filter */
} FILTER;

#define FILTER_SECTIONS   2    /* 2 filter sections for 24 db/oct filter */

typedef struct {
        double a0, a1, a2;       /* numerator coefficients */
        double b0, b1, b2;       /* denominator coefficients */
} BIQUAD;


/*
 * --------------------------------------------------------------------
 * 
 * iir_filter - Perform IIR filtering sample by sample on floats
 * 
 * Implements cascaded direct form II second order sections.
 * Requires FILTER structure for history and coefficients.
 * The length in the filter structure specifies the number of sections.
 * The size of the history array is 2*iir->length.
 * The size of the coefficient array is 4*iir->length + 1 because
 * the first coefficient is the overall scale factor for the filter.
 * Returns one output sample for each input sample.  Allocates history
 * array if not previously allocated.
 * 
 * float iir_filter(float input,FILTER *iir)
 * 
 *     float input        new float input sample
 *     FILTER *iir        pointer to FILTER structure
 * 
 * Returns float value giving the current output.
 * 
 * Allocation errors cause an error message and a call to exit.
 * --------------------------------------------------------------------
 */
inline float
iir_filter(float input,     /* new input sample */ 
           FILTER *iir) {    /* pointer to FILTER structure */
    unsigned int i;
    float *hist1_ptr,*hist2_ptr,*coef_ptr;
    float output,new_hist,history1,history2;

    //    DEBUG("in: %f\n", input);
/* allocate history array if different size than last call */

    if(!iir->history) {
        iir->history = (float *) calloc(2*iir->length,sizeof(float));
        if(!iir->history) {
            FAIL("Unable to allocate history array in iir_filter.\n");
            return 0;
        }
    }

    coef_ptr = iir->coef;                /* coefficient pointer */

    hist1_ptr = iir->history;            /* first history */
    hist2_ptr = hist1_ptr + 1;           /* next history */

    /* 1st number of coefficients array is overall input scale factor,
       or filter gain */
    output = input * (*coef_ptr++);

    for(i = 0 ; i < iir->length; i++) {
        history1 = *hist1_ptr;           /* history values */
        history2 = *hist2_ptr;

        /*
         * 
         * y[n] = gain - (history1 * a1) - (history2 * a2) + (history1 * b1) + (history2 * b2)? 
         */

        output = output - history1 * (*coef_ptr++);
        new_hist = output - history2 * (*coef_ptr++);    /* poles */

        output = new_hist + history1 * (*coef_ptr++);
        output = output + history2 * (*coef_ptr++);      /* zeros */

        *hist2_ptr++ = *hist1_ptr;
        *hist1_ptr++ = new_hist;
        hist1_ptr++;
        hist2_ptr++;
    }

    return(output);
}

// ----------------- file filterIIR00.c end -----------------

void
lowpass(shell *shl,
        int track,
        AFframecount start_offset,
        AFframecount end_offset,
        double frequency,
        double q) {
    float *out_buffer = mem_calloc(1, EFFECT_BUF_SIZE * sizeof(float));
    int i;
    unsigned nInd;
    double freq_adjust;
    double in1, in2, out1, out2;
    double a0, a1, a2, b0, b1, b2;
    double k;           /* overall gain factor */
    float *coef;
    FILTER iir;
    BIQUAD ProtoCoef[FILTER_SECTIONS];      /* Filter prototype coefficients,
                                               1 for each filter section */
    ITERATORF_INIT(start_offset, end_offset - start_offset);

    iir.length = FILTER_SECTIONS;         /* Number of filter sections */
    iir.history = NULL;
    /*
     * Allocate array of z-domain coefficients for each filter section
     * plus filter gain variable
     */
    iir.coef = (float *) mem_calloc(4 * iir.length + 1, sizeof(float));
    if(!iir.coef) {
        FAIL("Unable to allocate coef array, exiting.\n");
        ITERATORF_EXIT();
        return;
    }
    
    /*
     * Setup filter s-domain coefficients
     */

    /* Section 1 */
    ProtoCoef[0].a0 = 1.0;
    ProtoCoef[0].a1 = 0;
    ProtoCoef[0].a2 = 0;
    ProtoCoef[0].b0 = 1.0;
    ProtoCoef[0].b1 = 0.765367;
    ProtoCoef[0].b2 = 1.0;
    
    /* Section 2 */
    ProtoCoef[1].a0 = 1.0;
    ProtoCoef[1].a1 = 0;
    ProtoCoef[1].a2 = 0;
    ProtoCoef[1].b0 = 1.0;
    ProtoCoef[1].b1 = 1.847759;
    ProtoCoef[1].b2 = 1.0;

    if(!out_buffer) {
        FAIL("not enough memory for out buffer (%d bytes)\n",
             EFFECT_BUF_SIZE * sizeof(float));
        free(iir.coef);
        ITERATORF_EXIT();
        return;
    }

    in1 = in2 = 0;
    out1 = out2 = 0;
    ITERATOR_SKEL(shl, shl->sr->tracks[track],
                  ITERATORF_CHECKER(),
                  ITERATORF_READER(shl, shl->sr->tracks[track]),
             for(i = 0; i < iter_read; i++) {
                 k = 1.0;                 /* Set overall filter gain */                 
                 coef = iir.coef + 1;     /* Skip k, or gain */
                 freq_adjust = marker_list_slope_value(shl->sr->markers[track],
                                                       iter_frame_offset + i,
                                                       MARKER_SLOPE);
                 freq_adjust = MAX(1, frequency * freq_adjust);
                 /*
                  * Compute z-domain coefficients for each biquad
                  * section for new Cutoff Frequency and Resonance
                  */
                 for (nInd = 0; nInd < iir.length; nInd++) {
                     a0 = ProtoCoef[nInd].a0;
                     a1 = ProtoCoef[nInd].a1;
                     a2 = ProtoCoef[nInd].a2;
                     
                     b0 = ProtoCoef[nInd].b0;
                     b1 = ProtoCoef[nInd].b1 / q;      /* Divide by resonance or Q */
                     b2 = ProtoCoef[nInd].b2;
                     szxform(&a0, &a1, &a2, &b0, &b1, &b2,
                             frequency + freq_adjust, shl->sr->rate, &k, coef);
                     coef += 4;                       /* Point to next filter section */
                 }
                 
                 /* Update overall filter gain in coef array */
                 iir.coef[0] = k;
                 out_buffer[i] = iir_filter(float_frame_bits[i], &iir);
             }
                  track_float_frames_replace(shl->sr->tracks[track],
                                             out_buffer,
                                             iter_frame_offset,
                                             iter_read));
    free(out_buffer);
    free(iir.coef);
    free(iir.history);
    ITERATORF_EXIT();
    return;
}

module *
module_new() {
    MODULE_INIT(&modinfo, 
		"Lowpass",
		"Pascal Haakmat",
		"Copyright (C) 2002,2003");
    return &modinfo;
}

void
on_apply_clicked(GtkWidget *w,
                 gpointer user_data) {
    struct _module_state *module_state = 
        g_object_get_data(G_OBJECT(gtk_widget_get_toplevel(w)),
                          "user_data");
    filter_params *p = (filter_params *)module_state->data;
    p->frequency = gtk_spin_button_get_value(p->frequency_control);
    p->q = gtk_spin_button_get_value(p->q_control);
    action_do(ACTION_MODULE_EXECUTE_NEW(WITH_UNDO, 
                                        p->shl,
                                        p->id));
}

void
on_ok_clicked(GtkWidget *w,
              gpointer user_data) {
    struct _module_state *module_state = 
        g_object_get_data(G_OBJECT(gtk_widget_get_toplevel(w)),
                          "user_data");
    filter_params *p = (filter_params *)module_state->data;
    on_apply_clicked(w, NULL);
    gtk_object_destroy(GTK_OBJECT(p->dialog));
}

void
on_close_clicked(GtkWidget *w,
                  gpointer user_data) {
    struct _module_state *module_state = 
        g_object_get_data(G_OBJECT(gtk_widget_get_toplevel(w)),
                          "user_data");
    filter_params *p = (filter_params *)module_state->data;
    gtk_object_destroy(GTK_OBJECT(p->dialog));
}

void
on_dialog_destroy(GtkWidget *w,
                  gpointer user_data) {
    struct _module_state *module_state = 
        g_object_get_data(G_OBJECT(gtk_widget_get_toplevel(w)),
                          "user_data");
    module_state->is_open = 0;
    free(module_state->data);
}

void
module_open(module_id id,
            shell *shl, 
            int undo) {
    GladeXML *xml;
    GtkWidget *w;
    char path[4096];
    filter_params *p = mem_calloc(1, sizeof(filter_params));
    
    if(!p) {
        gui_alert("Not enough memory.");
        return;
    }

    snprintf(path, 4096, "%s/%s", dirname(modules[id].fname), LOWPASS_GLADE_FILE);
    xml = glade_xml_new(path, "dialog", NULL);
    
    if(!xml) {
        gui_alert("Lowpass: could not load interface %s.", path);
        free(p);
        return;
    }

    shl->module_state[id].is_open = 1;
    shl->module_state[id].data = p;
    p->id = id;
    p->shl = shl;
    p->frequency_control = GTK_SPIN_BUTTON(glade_xml_get_widget(xml, "frequency"));
    p->q_control = GTK_SPIN_BUTTON(glade_xml_get_widget(xml, "q"));
    p->dialog = GTK_WIDGET(glade_xml_get_widget(xml, "dialog"));
    g_signal_connect(GTK_OBJECT(p->dialog), "destroy", 
                     G_CALLBACK(on_dialog_destroy), shl);
    w = GTK_WIDGET(glade_xml_get_widget(xml, "ok"));
    g_signal_connect(GTK_OBJECT(w), "clicked", 
                     G_CALLBACK(on_ok_clicked), shl);
    w = GTK_WIDGET(glade_xml_get_widget(xml, "close"));
    g_signal_connect(GTK_OBJECT(w), "clicked", 
                     G_CALLBACK(on_close_clicked), shl);
    w = GTK_WIDGET(glade_xml_get_widget(xml, "apply"));
    g_signal_connect(GTK_OBJECT(w), "clicked", 
                     G_CALLBACK(on_apply_clicked), shl);
    g_object_set_data(G_OBJECT(p->dialog), "user_data",
                      GINT_TO_POINTER(&shl->module_state[id]));
    g_object_unref(G_OBJECT(xml));
}

void
module_close(mod_state *module_state) {
    gtk_object_destroy(GTK_OBJECT(((filter_params *)module_state->data)->dialog));
}

action_group *
module_execute(shell *shl, 
               int undo) {
    AFframecount start = shl->select_start,
        end = shl->select_end;
    int map = shl->select_channel_map;
    int t, id = shl->active_module_id;
    double frequency, q;
    filter_params *p = NULL;
    action_group *undo_ag = NULL;

    if(!shl->module_state[id].is_open) { 
        gui_alert("Lowpass does not have defaults.");
        return NULL;
    }
    p = shl->module_state[id].data;

    frequency = p->frequency;
    q = p->q;

    /* FIXME: deny this in interface (as well). */

    if(frequency > shl->sr->rate * 2) {
        gui_alert("Lowpass: frequency must be < %f.", shl->sr->rate * 2);
        return NULL;
    }
    
    rwlock_rlock(&shl->sr->rwl);
    if(undo) 
        undo_ag = action_group_undo_create(shl,
                                           map, 
                                           start,
                                           end - start,
                                           start,
                                           end - start);

    for(t = 0; t < snd_track_count(shl->sr); t++) 
        if((1 << t) & map)
            lowpass(shl,
                    t,
                    start,
                    end,
                    frequency,
                    q);
    rwlock_runlock(&shl->sr->rwl);

    return undo_ag;

}
