
// created 06.2003 by Stefan Kleine Stegemann
// 
// licensed under GPL


#include <PDFKit/PDFDocument.h>
#include <PDFKit/PDFImageRep.h>
#include "XPDFBridge.h"

#include <Foundation/NSException.h>
#include <Foundation/NSLock.h>

/*
 * Non-Public methods.
 */
@interface PDFDocument(Private)
- (id) initWithBridgeDocument: (XPDFObject)aPdfDoc;
- (void) translateRectToPageCoords: (int)pageNum
                              rect: (NSRect)rect
                              xMin: (double*)xMin
                              yMin: (double*)yMin
                              xMax: (double*)xMax
                              yMax: (double*)yMax;
@end


@implementation PDFDocument


- (id) init
{
   return [self initWithBridgeDocument: NULL];
}


/*
 * Designated initializer.
 */
- (id) initWithBridgeDocument: (XPDFObject)aPdfDoc
{
   if ((self = [super init]))
   {
      pdfDoc = aPdfDoc;
      outline = nil;
   }

   return self;
}   


- (void) dealloc
{
   NSLog(@"dealloc PDFDocument, retain count is %d", [self retainCount]);

   if (pdfDoc != NULL)
   {
      PDFDoc_delete(pdfDoc);
      pdfDoc = NULL;
   }
   else
   {
      NSLog(@"   no internal document to delete");
   }

   [outline dealloc];

   [super dealloc];
}


+ (void) initialize
{
   [super initialize];

   // Initialize XPDF backend
   XPDF_Initialize(PDFBaseResolution); // base resolution is
   NSLog(@"xpdf backend initialized");
}


+ (PDFDocument*)documentFromFile: (NSString*)fileName
{
   return [PDFDocument documentFromFile: fileName
                       ownerPassword: nil
                       userPassword: nil];
}

+ (PDFDocument*)documentFromFile: (NSString*)fileName 
                   ownerPassword: (NSString*)ownerPassword
                    userPassword: (NSString*)userPassword
{
   XPDFObject   newPdfDoc;
   const char*  cFileName;
   const char*  cOwnerPassword;
   const char*  cUserPassword;

   NSAssert(fileName != nil, @"no filename");

   cFileName = [fileName cString];
   
   cOwnerPassword = (ownerPassword != nil ? [ownerPassword cString] : NULL);
   cUserPassword  = (userPassword != nil ? [userPassword cString] : NULL);

   newPdfDoc = PDFDoc_create(cFileName,
                             cOwnerPassword,
                             cUserPassword);

   return [[[PDFDocument alloc] initWithBridgeDocument: newPdfDoc] autorelease];
}


- (BOOL) isOk
{
   NSAssert(pdfDoc != NULL, @"no document");

   return PDFDoc_isOk(pdfDoc);
}

- (int) errorCode
{
   NSAssert(pdfDoc != NULL, @"no document");
   
   return PDFDoc_getErrorCode(pdfDoc);
}


- (double) pageWidth: (int)pageNum
{
   NSAssert(pdfDoc != NULL, @"no document");
   
   return PDFDoc_getPageWidth(pdfDoc, pageNum);
}


- (double) pageHeight: (int)pageNum
{
   NSAssert(pdfDoc != NULL, @"no document");
   
   return PDFDoc_getPageHeight(pdfDoc, pageNum);
}


- (NSSize) pageSize: (int)pageNum considerRotation: (BOOL)rotation
{
   NSSize size;
   
   size = NSMakeSize([self pageWidth: pageNum], [self pageHeight: pageNum]);
   if (rotation)
   {
      if (([self pageRotate: pageNum] == 90) || ([self pageRotate: pageNum] == 180))
      {
         float height = size.height;
         size.height = size.width;
         size.width  = height;
      }
   }

   return size;
}


- (int) pageRotate: (int)pageNum
{
   NSAssert(pdfDoc != NULL, @"no document");
   
   return PDFDoc_getPageRotate(pdfDoc, pageNum);
}


- (int) countPages
{
   NSAssert(pdfDoc != NULL, @"no document");
   
   return PDFDoc_getNumPages(pdfDoc);
}


- (NSString*)metaData
{
   NSAssert(pdfDoc != NULL, @"no document");

   const char* data = PDFDoc_getMetaData(pdfDoc);
   return (data != NULL ? 
           [[[NSString alloc] initWithCString: data] autorelease]
           : 
           nil);
}


- (BOOL) hasOutline
{
   return PDFOutline_HasOutline(pdfDoc);
}


- (PDFOutline*) outline
{
   if ((!outline) && [self hasOutline])
   {
      outline =
         [[PDFOutline alloc] initWithOutlineItems: PDFOutline_GetOutlineItems(pdfDoc)
                             ofDocument: pdfDoc];
   }

   return outline;
}


- (BOOL) findText: (NSString*)text
             page: (int*)pageNum
           toPage: (int)maxPage
         position: (NSRect*)pos
          context: (NSString**)context
{
   double     xMin, yMin, xMax, yMax;
   char*      textCtxBuffer;
   int        textCtxBufferLength;

   // Note that PDFUtil_FindText uses upside down coords
   // (0,0 is upper left corner)

   if (pos->size.height == -1)
   {
      pos->size.height = [self pageHeight: *pageNum];
   }
   
   [self translateRectToPageCoords: *pageNum rect: *pos
         xMin: &xMin yMin: &yMin xMax: &xMax yMax: &yMax];

   if (!PDFUtil_FindText(pdfDoc, [text cString], pageNum, maxPage,
                         &xMin, &yMin, &xMax, &yMax,
                         &textCtxBuffer, &textCtxBufferLength))
   {
      //NSLog(@"%@ not found", text);
      return NO;
   }

   // translate resulting coords
   yMin = [self pageHeight: *pageNum] - yMin;
   yMax = [self pageHeight: *pageNum] - yMax;

   //NSLog(@"found %@ at %f, %f, %f, %f", text, xMin, yMin, xMax, yMax);
   pos->origin.x = (xMin < xMax ? xMin : xMax);
   pos->origin.y = (yMin < yMax ? yMin : yMax);
   pos->size.width = (xMax > xMin ? xMax - xMin : xMin - xMax);
   pos->size.height = (yMax > yMin ? yMax - yMin : yMin - yMax);

   if (textCtxBuffer != NULL)
   {
      if (context != NULL)
      {
         *context = [NSString stringWithCString: textCtxBuffer
                              length: textCtxBufferLength];
      }
      // we copy the buffer here instead of using initWithCStringNoCopy
      // because it has not been allocated with NSZoneMalloc
      free(textCtxBuffer);
   }
   else
   {
      if (context != NULL)
      {
         *context = @"??";
      }
   }
   
   return YES;
}


- (NSString*) getTextAtPage: (int)pageNum inRect: (NSRect)pos
{
   NSString* text = nil;
   char*     buffer;
   int       buffLen;
   double    xMin, yMin, xMax, yMax;
   
   [self translateRectToPageCoords: pageNum rect: pos
         xMin: &xMin yMin: &yMin xMax: &xMax yMax: &yMax];

   PDFUtil_GetText(pdfDoc, pageNum, xMin, yMin, xMax, yMax, &buffer, &buffLen);

   if (buffer != NULL)
   {
      text = [NSString stringWithCString: buffer length: buffLen];
      // we copy the buffer here instead of using initWithCStringNoCopy
      // because it has not been allocated with NSZoneMalloc
      free(buffer);
   }

   return text;
}


/*
 * Returns the internal wrapped object.
 */
- (void*) xpdfobject
{
   return pdfDoc;
}


/*
 * Translate a rectangular are to coordinates on a page.
 * The coordinate system is upside down.
 */
- (void) translateRectToPageCoords: (int)pageNum
                              rect: (NSRect)rect
                              xMin: (double*)xMin
                              yMin: (double*)yMin
                              xMax: (double*)xMax
                              yMax: (double*)yMax
{
   *xMin = rect.origin.x;
   *yMin = [self pageHeight: pageNum] - (rect.origin.y + rect.size.height);
   *xMax = rect.origin.x + rect.size.width;
   *yMax = [self pageHeight: pageNum] - rect.origin.y;
}

@end
