
// created 06.2003 by Stefan Kleine Stegemann
// 
// licensed under GPL

#ifndef _H_XPDF_OBJC_BRIDGE
#define _H_XPDF_OBJC_BRIDGE

#include <stdlib.h>

/*
 * 'Bridge-Functions' to create a Objective-C
 * Interface to (a part of) the XPDF System.
 */

#ifdef __cplusplus 
extern "C" {
#endif

/*
 * Types.
 */
typedef void* XPDFObject;

typedef struct OutputOptions
{
   int  resolution;
   int  firstPage;
   int  lastPage;
   int  paperWidth;
   int  paperHeight;
} OutputOptions;


typedef enum RenderResult
{
   RenderResult_Ok,
   RenderResult_ErrorOutputDevice,
   RenderResult_MiscError
} RenderResult;


typedef enum DisplayFontType
{
   T1DisplayFont,
   TTDisplayFont
} DisplayFontType;


/*
 * Managing access to the xpdf engine. Since this
 * engine seems not to be thread-safe, we use
 * a locking mechanism to guarantee that only one
 * thread uses the engine at a time.
 */
void XPDF_AcquireLock();
void XPDF_ReleaseLock();

/*
 * The bridge has to be initialized before it can be used.
 * To free resources that have been allocated during initialization,
 * the destroy function has to be called.
 */
void   XPDF_Initialize(double dpi);
void   XPDF_Destroy(void);
int    XPDF_IsInitialized(void);
double XPDF_DPI(void);

/* 
 * PDFDoc class 
 */
XPDFObject PDFDoc_create(const char* filename,
                         const char* ownerPassword,
                         const char* userPassword);
void PDFDoc_delete(XPDFObject pdfDoc);

int PDFDoc_isOk(XPDFObject pdfDoc);
int PDFDoc_getErrorCode(XPDFObject pdfDoc);

double PDFDoc_getPageWidth(XPDFObject pdfDoc, int pageNum);
double PDFDoc_getPageHeight(XPDFObject pdfDoc, int pageNum);
int    PDFDoc_getPageRotate(XPDFObject pdfDoc, int pageNum);

int PDFDoc_getNumPages(XPDFObject pdfDoc);
 
const char* PDFDoc_getMetaData(XPDFObject pdfDoc);

 /*
  * Searching
  */

// defines how many text from left to rigth
// is included in the textContext when a
// text has been found somewhere.
#define FOUND_CONTEXT_LEFT  80
#define FOUND_CONTEXT_RIGHT 80

int PDFUtil_FindText(XPDFObject pdfDoc,
                     const char* text,
                     int* pageA,
                     int toPage,
                     double *xMin,
                     double* yMin,
                     double* xMax,
                     double *yMax,
                     char** textContext,
                     int* textContextLength);

/*
 * Access content
 */
void PDFUtil_GetText(XPDFObject pdfDoc,
                     int page,
                     double xMin,
                     double yMin,
                     double xMax,
                     double yMax,
                     char** textA,
                     int* length);

/*
 * Outline
 */
int PDFOutline_HasOutline(XPDFObject pdfDoc);
XPDFObject PDFOutline_GetOutlineItems(XPDFObject pdfDoc);
int PDFOutline_CountItems(XPDFObject outlineItems);
XPDFObject PDFOutline_ItemAt(XPDFObject outlineItems, int index);
void PDFOutline_ItemOpen(XPDFObject outlineItem);
void PDFOutline_ItemClose(XPDFObject outlineItem);
int PDFOutline_HasKids(XPDFObject outlineItem);
XPDFObject PDFOutline_GetKids(XPDFObject outlineItem);
char* PDFOutline_GetTitle(XPDFObject outlineItem);
int PDFOutline_GetTargetPage(XPDFObject outlineItem, XPDFObject pdfDoc);


/*
 * Font Management
 */
void PDFFont_AddDisplayFont(const char* fontName,
                            const char* fontFile,
                            DisplayFontType type);
void PDFFont_GetDisplayFont(const char* fontName,
                            const char** fontFile,
                            DisplayFontType* type);

/*
 * Rendering
 */
XPDFObject PDFRender_CreateOutputDevice(XPDFObject pdfDoc);
void PDFRender_DestroyOutputDevice(XPDFObject device);
XPDFObject PDFRender_RenderPage(XPDFObject pdfDoc,
                                XPDFObject device,
                                int page,
                                double dpi,
                                int rotate);
void PDFRender_GetBitmapSize(XPDFObject bitmap, int* width, int* height);
void PDFRender_GetRGB(XPDFObject bitmap, unsigned char** buffer);


#ifdef __cplusplus 
};
#endif



#endif
