/*-
 * Copyright (c) 2001 Jordan DeLong
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of contributors may be
 *    used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include "wm.h"

/* list of pixmap structs */
static SLIST_HEAD(, pixmap) pixmap_list = SLIST_HEAD_INITIALIZER(pixmap_list);

/* free all pixmaps in preperation for a shutdown/restart */
void pixmap_shutdown() {
	pixmap_t *pixmap, *next;
	int i;

	pixmap = SLIST_FIRST(&pixmap_list);
	while (pixmap) {
		/*
		 * free all the memory associated with the pixmap;
		 * pixmap->xpmfn is freed when pixmap_getpixmaps is
		 * called, so it is not neccessary to free it here.
		 */
		for (i = 0; i < screen_count; i++) {
			XFreePixmap(display, pixmap->pixmaps[i]);
			XFreePixmap(display, pixmap->shapemasks[i]);
		}
		free(pixmap->ident);
		free(pixmap->pixmaps);
		free(pixmap->shapemasks);

		/* now free the pixmap_t */
		next = SLIST_NEXT(pixmap, p_list);
		free(pixmap);
		pixmap = next;
	}
	SLIST_INIT(&pixmap_list);
}

/* add a pixmap to our list */
pixmap_t *pixmap_add(char *ident, char *xpmfn) {
	pixmap_t *pixmap;

	pixmap = calloc(1, sizeof(pixmap_t));
	if (!pixmap)
		return NULL;
	pixmap->ident = ident;
	pixmap->xpmfn = xpmfn;
	SLIST_INSERT_HEAD(&pixmap_list, pixmap, p_list);

	return pixmap;
}

/* find a pixmap by identifier; used during rcfile parsing */
pixmap_t *pixmap_ident(char *ident) {
	pixmap_t *pixmap;

	SLIST_FOREACH(pixmap, &pixmap_list, p_list)
		if (strcmp(pixmap->ident, ident) == 0)
			return pixmap;

	return NULL;
}

/*
 * allocate X pixmaps for each pixmap_t; a Pixmap is made
 * for each managed screen.
 */
int pixmap_getpixmaps() {
	pixmap_t *pixmap;
	screen_t *screen;
	Window dumwin;
	int dumint;

	SLIST_FOREACH(pixmap, &pixmap_list, p_list) {
		/* get memory */
		pixmap->pixmaps = malloc(sizeof(Pixmap) * screen_count);
		if (!pixmap->pixmaps)
			return -1;
		pixmap->shapemasks = malloc(sizeof(Pixmap) * screen_count);
		if (!pixmap->shapemasks)
			return -1;

		/* get the pixmaps */
		TAILQ_FOREACH_REVERSE(screen, &screen_list, screenlist, s_list)
			XpmReadFileToPixmap(display, screen->root,
				pixmap->xpmfn, &pixmap->pixmaps[screen->num],
				&pixmap->shapemasks[screen->num], NULL);

		/* get the dimensions */
		XGetGeometry(display, pixmap->pixmaps[0], &dumwin,
			&dumint, &dumint, &pixmap->width,
			&pixmap->height, &dumint, &dumint);

		/* free the pixmap filename; it's no longer needed */
		free(pixmap->xpmfn);
	}

	return 0;
}
