/*
 * GooCanvas. Copyright (C) 2005 Damon Chaplin.
 * Released under the GNU LGPL license. See COPYING for details.
 *
 * goocanvasitem.h - interface for canvas items & groups.
 */
#ifndef __GOO_CANVAS_ITEM_H__
#define __GOO_CANVAS_ITEM_H__

#include <gtk/gtk.h>

G_BEGIN_DECLS


/**
 * GooCanvasAnimateType
 * @GOO_CANVAS_ANIMATE_FREEZE: the item remains in the final position.
 * @GOO_CANVAS_ANIMATE_RESET: the item is moved back to the initial position.
 * @GOO_CANVAS_ANIMATE_RESTART: the animation is restarted from the initial
 *  position.
 * @GOO_CANVAS_ANIMATE_BOUNCE: the animation bounces back and forth between the
 *  start and end positions.
 *
 * #GooCanvasAnimateType is used to specify what happens when the end of an
 * animation is reached.
 */
typedef enum
{
  GOO_CANVAS_ANIMATE_FREEZE,
  GOO_CANVAS_ANIMATE_RESET,
  GOO_CANVAS_ANIMATE_RESTART,
  GOO_CANVAS_ANIMATE_BOUNCE
} GooCanvasAnimateType;


/**
 * GooCanvasBounds
 * @x1: the left edge.
 * @y1: the top edge.
 * @x2: the right edge.
 * @y2: the bottom edge.
 *
 * #GooCanvasBounds represents the bounding box of an item in the canvas.
 */
typedef struct _GooCanvasBounds GooCanvasBounds;
struct _GooCanvasBounds
{
  gdouble x1, y1, x2, y2;
};


#define GOO_TYPE_CANVAS_ITEM            (goo_canvas_item_get_type ())
#define GOO_CANVAS_ITEM(obj)            (G_TYPE_CHECK_INSTANCE_CAST ((obj), GOO_TYPE_CANVAS_ITEM, GooCanvasItem))
#define GOO_CANVAS_ITEM_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST ((klass), GOO_TYPE_CANVAS_ITEM, GooCanvasItemClass))
#define GOO_IS_CANVAS_ITEM(obj)         (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GOO_TYPE_CANVAS_ITEM))
#define GOO_CANVAS_ITEM_GET_IFACE(obj)  (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GOO_TYPE_CANVAS_ITEM, GooCanvasItemIface))


/* Workaround for circular dependencies. Include this file first. */
typedef struct _GooCanvasModel      GooCanvasModel;
typedef struct _GooCanvasView       GooCanvasView;
typedef struct _GooCanvasItemView   GooCanvasItemView;


/**
 * GooCanvasItem
 *
 * #GooCanvasItem is a typedef used for objects that implement the
 * #GooCanvasItem interface.
 *
 * (There is no actual #GooCanvasItem struct, since it is only an interface.
 * But using '#GooCanvasItem' is more helpful than using '#GObject'.)
 */
typedef struct _GooCanvasItem       GooCanvasItem;


/**
 * GooCanvasItemIface
 *
 * #GooCanvasItemIFace holds the virtual methods that make up the
 * #GooCanvasItem interface.
 */
typedef struct _GooCanvasItemIface  GooCanvasItemIface;

struct _GooCanvasItemIface
{
  /*< private >*/
  GTypeInterface base_iface;

  /*< public >*/
  /* Virtual methods that group items must implement. */
  GooCanvasModel*      (* get_model)	   (GooCanvasItem       *item);
  gint		       (* get_n_children)  (GooCanvasItem       *group);
  GooCanvasItem*       (* get_child)	   (GooCanvasItem       *group,
					    gint                 child_num);

  /* Virtual methods that group items may implement. */
  void                 (* add_child)       (GooCanvasItem       *group,
					    GooCanvasItem       *child,
					    gint                 position);
  void                 (* move_child)      (GooCanvasItem       *group,
					    gint                 old_position,
					    gint                 new_position);
  void                 (* remove_child)    (GooCanvasItem       *group,
					    gint                 child_num);

  /* Virtual methods that all canvas items must implement. */
  GooCanvasItem*       (* get_parent)	  (GooCanvasItem       *item);
  void                 (* set_parent)	  (GooCanvasItem       *item,
					   GooCanvasItem       *parent);

  cairo_matrix_t*      (* get_transform)  (GooCanvasItem       *item);
  void                 (* set_transform)  (GooCanvasItem       *item,
					   cairo_matrix_t      *matrix);

  GooCanvasItemView*   (* create_view)	  (GooCanvasItem       *item,
					   GooCanvasView       *canvas_view,
					   GooCanvasItemView   *parent_view);

  /* Signals. */
  void                 (* child_added)	  (GooCanvasItem      *item,
					   gint                child_num);
  void                 (* child_moved)	  (GooCanvasItem      *item,
					   gint                old_child_num,
					   gint                new_child_num);
  void                 (* child_removed)  (GooCanvasItem      *item,
					   gint                child_num);
  void                 (* changed)	  (GooCanvasItem      *item,
					   gboolean            recompute_bounds);
};


GType                goo_canvas_item_get_type       (void) G_GNUC_CONST;

GooCanvasItem *      goo_canvas_item_new	    (GooCanvasItem *parent,
						     GType          type,
						     const gchar   *first_property,
						     ...);

/*
 * Group functions - these should only be called on container items.
 */
gint                 goo_canvas_item_get_n_children (GooCanvasItem *group);
GooCanvasItem*       goo_canvas_item_get_child      (GooCanvasItem *group,
						     gint           child_num);
void                 goo_canvas_item_add_child      (GooCanvasItem *group,
						     GooCanvasItem *item,
						     gint           position);
void                 goo_canvas_item_move_child     (GooCanvasItem *group,
						     gint           old_position,
						     gint           new_position);
void                 goo_canvas_item_remove_child   (GooCanvasItem *group,
						     gint           child_num);
gint                 goo_canvas_item_find_child     (GooCanvasItem *group,
						     GooCanvasItem *child);

/*
 * Item functions - these are safe to call on all items.
 */
GooCanvasModel*      goo_canvas_item_get_model      (GooCanvasItem *item);
GooCanvasItem*       goo_canvas_item_get_parent     (GooCanvasItem *item);
void                 goo_canvas_item_set_parent	    (GooCanvasItem *item,
						     GooCanvasItem *parent);
gboolean             goo_canvas_item_is_container   (GooCanvasItem *item);

void                 goo_canvas_item_raise          (GooCanvasItem *item,
						     GooCanvasItem *above);
void                 goo_canvas_item_lower          (GooCanvasItem *item,
						     GooCanvasItem *below);

cairo_matrix_t*      goo_canvas_item_get_transform  (GooCanvasItem *item);
void                 goo_canvas_item_set_transform  (GooCanvasItem *item,
						     cairo_matrix_t *matrix);

void                 goo_canvas_item_translate      (GooCanvasItem *item,
						     double         tx,
						     double         ty);
void                 goo_canvas_item_scale          (GooCanvasItem *item,
						     double         sx,
						     double         sy);
void                 goo_canvas_item_rotate         (GooCanvasItem *item,
						     double         degrees,
						     double         cx,
						     double         cy);
void                 goo_canvas_item_skew_x         (GooCanvasItem *item,
						     double         degrees,
						     double         cx,
						     double         cy);
void                 goo_canvas_item_skew_y         (GooCanvasItem *item,
						     double         degrees,
						     double         cx,
						     double         cy);

void                 goo_canvas_item_animate        (GooCanvasItem *item,
						     double         x,
						     double         y,
						     double         scale,
						     double         degrees,
						     gint           duration,
						     gint           step_time,
						     GooCanvasAnimateType type);
void                 goo_canvas_item_stop_animation (GooCanvasItem *item);


G_END_DECLS

#endif /* __GOO_CANVAS_ITEM_H__ */
