/*
 * GooCanvas. Copyright (C) 2005 Damon Chaplin.
 * Released under the GNU LGPL license. See COPYING for details.
 *
 * goocanvasitemsimple.h - abstract base class for simple items with styles.
 */
#ifndef __GOO_CANVAS_ITEM_SIMPLE_H__
#define __GOO_CANVAS_ITEM_SIMPLE_H__

#include <gtk/gtk.h>
#include "goocanvasitem.h"
#include "goocanvasutils.h"

G_BEGIN_DECLS


/**
 * GooCanvasStyleValuesMask
 * @GOO_CANVAS_STYLE_STROKE_PATTERN: the stroke pattern has been set.
 * @GOO_CANVAS_STYLE_FILL_PATTERN: the fill pattern has been set.
 * @GOO_CANVAS_STYLE_FILL_RULE: the fill rule has been set.
 * @GOO_CANVAS_STYLE_OPERATOR: the operator has been set.
 * @GOO_CANVAS_STYLE_ANTIALIAS: the antialias setting has been set.
 * @GOO_CANVAS_STYLE_LINE_WIDTH: the line width has been set.
 * @GOO_CANVAS_STYLE_LINE_CAP: the line cap style has been set.
 * @GOO_CANVAS_STYLE_LINE_JOIN: the line join style has been set.
 * @GOO_CANVAS_STYLE_LINE_JOIN_MITER_LIMIT: the miter limit of line joins has
 *  been set.
 * @GOO_CANVAS_STYLE_LINE_DASH: the line dash pattern has been set.
 *
 * Specifies which fields of a #GooCanvasStyle object have been set.
 */
typedef enum
{
  GOO_CANVAS_STYLE_STROKE_PATTERN	 = 1 << 0,
  GOO_CANVAS_STYLE_FILL_PATTERN		 = 1 << 1,
  GOO_CANVAS_STYLE_FILL_RULE		 = 1 << 2,
  GOO_CANVAS_STYLE_OPERATOR		 = 1 << 3,
  GOO_CANVAS_STYLE_ANTIALIAS		 = 1 << 4,

  GOO_CANVAS_STYLE_LINE_WIDTH		 = 1 << 5,
  GOO_CANVAS_STYLE_LINE_CAP		 = 1 << 6,
  GOO_CANVAS_STYLE_LINE_JOIN		 = 1 << 7,
  GOO_CANVAS_STYLE_LINE_JOIN_MITER_LIMIT = 1 << 8,
  GOO_CANVAS_STYLE_LINE_DASH		 = 1 << 9
} GooCanvasStyleValuesMask;


/* These are the standard cairo drawing attributes. We allow the style to
   be shared between multiple objects to avoid wasting memory. */
/**
 * GooCanvasStyle
 * @ref_count: the reference count of the struct.
 * @mask: a mask specifying which fields of the #GooCanvasStyle have been set.
 * @stroke_pattern: the stroke pattern (or color).
 * @fill_pattern: the fill pattern (or color).
 * @line_width: the line width.
 * @line_join_miter_limit: the minimum angle in degrees at which the miter
 *  join style will be used. For smaller angles a bevel join is used instead.
 * @dash: the dash pattern.
 * @fill_rule: the fill rule.
 * @op: the drawing operator.
 * @antialias: the type of antialiasing to do.
 * @line_cap: the line cap style.
 * @line_join: the line join style.
 *
 * #GooCanvasStyle describes the style in which an item is to be drawn, e.g.
 * its stroke and fill colors or patterns, and its line width and style.
 */
typedef struct _GooCanvasStyle GooCanvasStyle;
struct _GooCanvasStyle
{
  int ref_count;

  /* This specifies which fields are actually set. If the STROKE_PATTERN bit
     is set, and stroke_pattern is NULL, no stroke will be drawn. */
  GooCanvasStyleValuesMask mask;

  cairo_pattern_t *stroke_pattern;
  cairo_pattern_t *fill_pattern;

  double line_width, line_join_miter_limit;

  GooCanvasLineDash *dash;

  /* We use bitmasks here to cut down on memory use a bit. I've given each
     field a few bits more than it needs to allow for new values. */
  cairo_fill_rule_t fill_rule       : 3;
  cairo_operator_t  op              : 6;
  cairo_antialias_t antialias       : 4;
  cairo_line_cap_t  line_cap        : 4;
  cairo_line_join_t line_join       : 4;
};


#define GOO_TYPE_CANVAS_STYLE (goo_canvas_style_get_type ())
GType           goo_canvas_style_get_type (void);
GooCanvasStyle* goo_canvas_style_new	  (void);
GooCanvasStyle* goo_canvas_style_ref	  (GooCanvasStyle *style);
void            goo_canvas_style_unref	  (GooCanvasStyle *style);


#define GOO_TYPE_CANVAS_ITEM_SIMPLE            (goo_canvas_item_simple_get_type ())
#define GOO_CANVAS_ITEM_SIMPLE(obj)            (G_TYPE_CHECK_INSTANCE_CAST ((obj), GOO_TYPE_CANVAS_ITEM_SIMPLE, GooCanvasItemSimple))
#define GOO_CANVAS_ITEM_SIMPLE_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST ((klass), GOO_TYPE_CANVAS_ITEM_SIMPLE, GooCanvasItemSimpleClass))
#define GOO_IS_CANVAS_ITEM_SIMPLE(obj)         (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GOO_TYPE_CANVAS_ITEM_SIMPLE))
#define GOO_IS_CANVAS_ITEM_SIMPLE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GOO_TYPE_CANVAS_ITEM_SIMPLE))
#define GOO_CANVAS_ITEM_SIMPLE_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS ((obj), GOO_TYPE_CANVAS_ITEM_SIMPLE, GooCanvasItemSimpleClass))


typedef struct _GooCanvasItemSimple       GooCanvasItemSimple;
typedef struct _GooCanvasItemSimpleClass  GooCanvasItemSimpleClass;

/**
 * GooCanvasItemSimple
 *
 * The #GooCanvasItemSimple-struct struct contains private data only.
 */
struct _GooCanvasItemSimple
{
  GObject parent_object;

  /* The parent item. */
  GooCanvasItem *parent;

  /* The style to draw with. */
  GooCanvasStyle *style;

  /* The transformation matrix, or NULL. */
  cairo_matrix_t *transform;

  /* Whether the item is visible, invisible, or visible above a given scale. */
  GooCanvasItemVisibility visibility;

  /* If visibility is VISIBLE_ABOVE_THRESHOLD the item is visible if the canvas
     scale setting is above this threshold (or equal to it). */
  gdouble visibility_threshold;

  /* What events the item should receive. */
  GooCanvasPointerEvents pointer_events;

  /* The title and description of the item for accessibility. */
  gchar *title;
  gchar *description;
};

struct _GooCanvasItemSimpleClass
{
  GObjectClass parent_class;
};


GType          goo_canvas_item_simple_get_type    (void) G_GNUC_CONST;

void           goo_canvas_item_simple_set_style	  (GooCanvasItemSimple *item,
						   GooCanvasStyle      *style);


void           goo_canvas_item_simple_set_fill_options   (GooCanvasItemSimple *item,
							  cairo_t             *cr);
void           goo_canvas_item_simple_set_stroke_options (GooCanvasItemSimple *item,
							  cairo_t             *cr);
void           goo_canvas_item_simple_get_path_bounds (GooCanvasItemSimple *item,
						       cairo_t           *cr,
						       GooCanvasBounds  *bounds);
void           goo_canvas_item_simple_user_bounds_to_device (GooCanvasItemSimple *item,
							     cairo_t             *cr,
							     GooCanvasBounds     *bounds);
gboolean       goo_canvas_item_simple_check_in_path (GooCanvasItemSimple *item,
						     gdouble              x,
						     gdouble              y,
						     cairo_t             *cr,
						     GooCanvasPointerEvents pointer_events);
void           goo_canvas_item_simple_paint_path    (GooCanvasItemSimple *item,
						     cairo_t             *cr);


G_END_DECLS

#endif /* __GOO_CANVAS_ITEM_SIMPLE_H__ */
